/*!
 * @file quantized_vector.h
 * @brief Quantized vector of either 1, 2, 4, 8, 16 bit-width
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 *
 */

#if !defined(QUANTIZED_VECTOR_HEADER_)
#define      QUANTIZED_VECTOR_HEADER_

#include <stdint.h>
#include <vector>
#include <iosfwd>
#include <string>

#include "xml_serdes.h"

/*! 
 * @class QuantizedVector
 * @brief Quantized vector of either 1, 2, 4, 8, 16 bit-width
 * 
 * float/double値を送信する際の通信データ量を削減するため，
 * 値を量子化して送るためのvectorクラス。
 *
 * 
 * データビット数としては1,2,4,8,16が可能。ただし，使用する際は最小値，
 * 最大値，「取りうる値の数」を設定して用いる。必要なbit数は自動計算さ
 * れ，現在の実装では1, 2, 4, 8, 16のいずれかになる。
 * 
 * @todo 必要最低限のbit数をサポート( 3bit, 5bitなど)
 *
 * @warning 値が設定された後で，「取りうる値の数」を変更する操作には未
 *          対応である。
 *   
 *  切り捨ての方法(精度低下時)，値の補完（精度向上時）の処理の方針が決
 *  められないため，このようにした。以下の順序の使用のみサポートする。
 *
 *   <ol>
 *     <li> 取りうる値の数，rangeを設定</li>
 *     <li> 値(データ)を設定 </li>
 *   </ol>
 */
class QuantizedVector {
public:
	typedef size_t      size_type;
	typedef float       value_type;
	typedef uint16_t storage_type;

	QuantizedVector();	//!< デフォルトコンストラクタ。空データを作る

	/*! コンストラクタ
	 *  @param[in] n_values 取りうる値の数
	 *  @param[in] range_min 最小値
	 *  @param[in] range_max 最大値
	 */
	QuantizedVector(size_t n_values, value_type range_min, value_type range_max);
	virtual ~QuantizedVector() { } //!< 仮想デストラクタ。何もしない

	//! 取りうる最小値を返す
	value_type rangeMin() const { return range_min_; }
	//! 取りうる最大値を返す
	value_type rangeMax() const { return range_max_; }

	/*! 値の取りうる幅を設定する.
	 *  @param[in] range_min 最小値
	 *  @param[in] range_max 最大値
	 *  @return 成功/失敗。現在は range_min = range_maxの場合はエラーとする
	 */
	bool setRange(value_type range_min, value_type range_max);

	/*! 何通りの値を取りうるかを設定する。
	 * @param [in] n_values 取りうる値の数
	 * @return 設定可能範囲の場合trueが返る。
	 */
	bool setNumberOfValues(size_t n_values);

	//! 何通りの値を取りうるかを返す
	//! @return n_values 取りうる値の数
	size_t numberOfValues() const { return n_values_; }

	//! データの個数を返す。
	size_t size() const { return data_.size(); }
	/*! データの個数を変更
	 * @param[in] new_size 新しいサイズ
	 * @param[in] x 個数が増えた場合に，生成された要素にセットされる値
	 */
	void resize(size_type new_size, value_type x = value_type());

	//! 値をセットする
	template <typename InputIterator>
	void assign(InputIterator first, InputIterator last);

	//! 内部の値を他のシーケンスにコピーする
	template <typename OutputIterator>
	OutputIterator copy_to(OutputIterator out) const;

	//! 内部の値を他のシーケンスにコピーする
	template <typename OutputIterator>
	OutputIterator copy_to(OutputIterator out, OutputIterator oend) const;


	//! 内部表現を返す添字オペレータ(const)
	storage_type operator[](size_type index) const { return data_[index]; }
	//! 内部表現を返す添字オペレータ(non const)
	storage_type& operator[](size_type index) { return data_[index]; }

	/*! Xmlへの書込みを行う
	 * @param os 書込み先ストリーム
	 * @param[in] tag XMLタグ文字列
	 * @retval 0 成功
	 * @retval negative 失敗
	 */
	int writeToXml(std::ostream& os, const std::string& tag) const;

	/*! Xmlからの読み込みを行う
	 * @param is 読み込み元ストリーム
	 * @param[in] tag XMLタグ文字列
	 * @retval 0 成功
	 * @retval negative 失敗
	 */
	int readFromXml(std::istream& is, const std::string& tag);
protected:
	typedef std::vector<storage_type> vector_type;

	/*! 値を量子化.
	 *  @param[in] v 量子化前の値
	 *  @return      量子化された値
	 */
	storage_type quantize(value_type v) const;

	/*! 量子化された値の復元
	 *  @param[in] s 量子化された値
	 *  @return      復元された値
	 */
	value_type dequantize(storage_type s) const;

	/*! 刻み幅を計算する。
	 * 
	 * 例。
	 * <pre>
	 *  range_min_ = 1.0, range_max_ = 2.0, n_values_ = 3;
	 *  
	 *        +----------+----------+
	 *       1.0        1.5        2.0  元の値(value_type)
	 *        0          1          2   エンコードされた値
	 *       
	 *    --&gt; 3通りなので，2bit必要
	 *
	 *        step = (2.0 - 1.0) / (3 - 1) = 0.5
	 * </pre>
	 *        
	 *
	 */  
	value_type step() const
	{
		return (range_max_ - range_min_) / (n_values_ - 1);
	}

	//! 取りうる値の数から必要なbit数を計算する
	static int bits_necessary_for(size_t n_values);

	/*! bit数縮小処理を行う.
	 * @param[out] packed 縮小された値の書込み先
	 */
	void pack_to(std::vector<uint8_t>& packed) const;

	/*! bit数縮小された値からの復元処理
	 * @param[in] packed 縮小された値をここから読む
	 * @param[in] n_bits bit数
	 * @param[in] n_elems 格納された要素数
	 */
	void unpack_from(const std::vector<uint8_t>& packed, int n_bits, size_t n_elems);
private:

	std::vector<storage_type> data_; //!< 内部表現された値。16bit
	size_t n_values_;	//!< 表現可能な値の数(何通りか?)
	value_type range_min_;	//!< 表現可能な最小値
	value_type range_max_;	//!< 表現可能な最大値
	
	/* ---- decls for serialization ---- */

	friend class boost::serialization::access;

	//! シリアライズ処理をsave/loadに分けるためのboostマクロ
	BOOST_SERIALIZATION_SPLIT_MEMBER();

	/*! シリアライズの際にboostから呼ばれるメンバテンプレート.
	 *  @tparam Archive Boostのアーカイブクラス名
	 *  @param[out] ar Archive Boostのアーカイブクラスのインスタンス
	 *  @param[in] (未使用) バージョン
	 */
	template <class Archive>
	void save(Archive& ar, const unsigned int version) const;

	/*! デシリアライズの際にboostから呼ばれるメンバテンプレート.
	 *  @tparam Archive Boostのアーカイブクラス名
	 *  @param[out] ar Archive Boostのアーカイブクラスのインスタンス
	 *  @param[in] (未使用) バージョン
	 */
	template <class Archive>
	void load(Archive& ar, const unsigned int version);
};

template <class Archive>
void
QuantizedVector::save(Archive& ar, const unsigned int /* version */) const
{
	using namespace boost::serialization;
	using namespace XmlSerDes;
	using namespace std;

	ar & make_nvp("range_min", range_min_);
	ar & make_nvp("range_max", range_max_);
	ar & make_nvp("n_values_", n_values_);
	size_t n_elems = data_.size();
	ar & make_nvp("n_elems", n_elems);

	size_t n_bits = bits_necessary_for(n_values_);
	ar & make_nvp("n_bits", n_bits);

	vector<uint8_t> packed;
	pack_to(packed);

	ByteArray ba(packed, ByteArray::Save());
	ar & make_nvp("packed_bytes", ba);
}
template <class Archive>
void
QuantizedVector::load(Archive& ar, const unsigned int /* version */)
{
	using namespace boost::serialization;
	using namespace XmlSerDes;
	using namespace std;

	ar & make_nvp("range_min", range_min_);
	ar & make_nvp("range_max", range_max_);
	ar & make_nvp("n_values_", n_values_);

	size_t n_elems;
	ar & make_nvp("n_elems", n_elems);

	size_t n_bits;
	ar & make_nvp("n_bits", n_bits);

	vector<uint8_t> packed;
	ByteArray ba(packed, ByteArray::Load());
	ar & make_nvp("packed_bytes", ba);

	unpack_from(packed, n_bits, n_elems);
}

/*! @details
 *
 *  [@a first, @a last)の範囲を自分に読み込む。読み込む際に量子化を行う
 *  
 *  @tparam InputIterator 入力反復子クラス
 *  @param[in] first 入力範囲の最初
 *  @param[in] last 入力範囲の最後の一つ後。逆参照されない。
 *
 */
template <typename InputIterator>
void
QuantizedVector::assign(InputIterator first, InputIterator last)
{
	data_.resize(std::distance(first, last));

	vector_type::iterator dst = data_.begin();
	for (InputIterator src = first ; src != last; ++src, ++dst)
		*dst = quantize(*src); // FIXME: slow!
}
/*! @details
 *
 *  @tparam OutputIterator 出力反復子クラス
 *  @param[out] out 出力先を表す反復子。ここから値を書き始める。
 *  @return out + (data_.end() - data.begin())に等しい出力反復子
 */
template <typename OutputIterator>
OutputIterator
QuantizedVector::copy_to(OutputIterator out) const
{
	vector_type::const_iterator cit = data_.begin();
	vector_type::const_iterator end = data_.end();

	for ( ; cit != end; ++cit, ++out)
		*out = dequantize(*cit);
	return out;
}
/*! @details
 *
 *  @tparam OutputIterator 出力反復子クラス
 *  @param[out] out 出力先を表す反復子。ここから値を書き始める。
 *  @param[out] oend 出力先の最後の一つ後を表す反復子。ここには書かない。
 *  @return out + min(oend-out, data_.end()-data_.begin())に等しい反復子
 */
template <typename OutputIterator>
OutputIterator
QuantizedVector::copy_to(OutputIterator out, OutputIterator oend) const
{
	vector_type::const_iterator cit = data_.begin();
	vector_type::const_iterator end = data_.end();

	for ( ; cit != end && out != oend; ++cit, ++out)
		*out = dequantize(*cit);

	return out;
}



#endif	  /*!QUANTIZED_VECTOR_HEADER_*/

/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 8
 * indent-tabs-mode: t
 * End:
 *
 */

