/*!
 * @file xml_serdes.h
 * @brief XML SERialization/DESerialization Internal header file
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 *
 * Boostを利用した，オブジェクトのシリアライズ，デシリアライズ処理。
 * また，シリアライズの際に中間的に使用されるデータ構造などを定義している。
 *
 * @warning boost 1.34でのみテストされている。
 *
 */
#if !defined(XML_SERDES_HEADER_)
#define      XML_SERDES_HEADER_

#include <stdint.h>
#include <exception>
#include <iostream>

#include <boost/lexical_cast.hpp>

#include <boost/serialization/nvp.hpp>
#include <boost/archive/xml_oarchive.hpp>
#include <boost/archive/xml_iarchive.hpp>
#include <boost/serialization/binary_object.hpp>
#include <boost/serialization/base_object.hpp>

#include <boost/serialization/utility.hpp>
#include <boost/serialization/vector.hpp>
#include <boost/serialization/string.hpp>

#include <boost/version.hpp>

#define VERSION_USE_ASSUME_ABSTRACT 104000 // ちゃんと調べる必要あり

#if BOOST_VERSION >= VERSION_USE_ASSUME_ABSTRACT
#include <boost/serialization/assume_abstract.hpp>
#else
#include <boost/serialization/is_abstract.hpp>
#endif


namespace XmlSerDes {

/*!
 * @class ByteArray
 * @brief  Helper Struct to Handle (de)serialization of byte array.
 * 
 * バイト列としてシリアライズする際に用いられるヘルパークラス。
 * メモリデータのコピーを避けるために，バッファを参照で渡している。
 * Save/Loadで中間的に使用される想定なので，コンストラクタで
 * 使用法を指定する。(Tagged Dispatchにより，Load()/Save()のいずれかを指定)
 *
 * Save 指定時にはbuf参照はreadonlyとなり，Load指定時には参照は書込み可
 * 能となる。
 *
 */
class ByteArray {
public:
	struct Load { };	//<! @brief empty struct for tag-dispatch
	struct Save { };   	//<! @brief empty struct for tag-dispatch

	/*! 読み出し(Load)用コンストラクタ
	 *  @param[out] buf 読み出したデータをこのバッファに書く
	 *  @param[in] Load() このコンストラクタを指定するためのタグ
	 */
	ByteArray(std::vector<uint8_t>& buf, const Load&);

	/*! 書き出し(Save)用コンストラクタ
	 *  @param[out] buf このバッファから読みだしてsaveする
	 *  @param[in] Save() このコンストラクタを指定するためのタグ
	 */
	ByteArray(const std::vector<uint8_t>& buf, const Save&);

private:
	std::vector<uint8_t> dummy_; //!< 参照初期化のための空ベクタ
	const std::vector<uint8_t>& sbuf_; //!< saveバッファへの参照 
	std::vector<uint8_t>& lbuf_;	   //!< loadバッファへの参照

	/* ---- decls for serialization ---- */
	friend class boost::serialization::access;

	//! シリアライズ処理をsave/loadに分けるためのboostマクロ
	BOOST_SERIALIZATION_SPLIT_MEMBER();

	/*! シリアライズの際にboostから呼ばれるメンバテンプレート.
	 *  @tparam Archive Boostのアーカイブクラス名
	 *  @param[out] ar Archive Boostのアーカイブクラスのインスタンス
	 *  @param[in] (未使用) バージョン
	 */
	template <class Archive> void save(Archive& ar, const unsigned int /* version */) const
	{
		using namespace boost::serialization;
		size_t n_bytes = sbuf_.size();
		ar & make_nvp(SIZE_TAG, n_bytes);
		if (n_bytes == 0)
			return;
		ar & make_nvp(DATA_TAG, make_binary_object((void*)(&sbuf_[0]), n_bytes));
	}
	/*! デシリアライズの際にboostから呼ばれるメンバテンプレート.
	 *  @tparam Archive Boostのアーカイブクラス名
	 *  @param[out] ar Archive Boostのアーカイブクラスのインスタンス
	 *  @param[in] (未使用) バージョン
	 */
	template <class Archive>
	void load(Archive& ar, const unsigned int /* version */)
	{
		using namespace boost::serialization;

		size_t n_bytes = -1;
		ar & make_nvp(SIZE_TAG, n_bytes);
		if (n_bytes <= 0)
			return;

		lbuf_.resize(n_bytes);
		ar & make_nvp(DATA_TAG, make_binary_object(&(lbuf_[0]), n_bytes));
	}

	static const char SIZE_TAG[]; //! サイズを格納する際のタグ名
	static const char DATA_TAG[]; //! データを格納する際のタグ名
};

/*! XMLへシリアライズするためのヘルパテンプレート関数.
 *  @tparam Serializable シリアライズ可能なクラス名
 *  @param os 出力先ストリーム
 *  @param r シリアライズ可能オブジェクトへの参照
 *  @param tag XMLに出力するタグ名
 *  @retval 0   成功
 *  @retval negative  失敗
 */
template <typename Serializable>
int writeToXmlSub(std::ostream& os, const Serializable& r, const std::string& tag)
{
	using namespace boost::archive;
	using namespace boost::serialization;
	using namespace std;
	try {
		xml_oarchive oa(os);
		oa << make_nvp(tag.c_str(), r);
	} catch (const exception& e) {
		cerr << "writeToXmlSub: ";
		cerr << "Exception thrown while Serialization: ";
		cerr << e.what() << endl;
		return -1;
	}
	return 0;
}
/*! XMLからデシリアライズするためのヘルパテンプレート関数.
 *  @tparam Serializable シリアライズ可能なクラス名
 *  @param is 入力元ストリーム
 *  @param r シリアライズ可能オブジェクトへの参照
 *  @param tag XMLに出力するタグ名
 *  @retval 0   成功
 *  @retval negative  失敗
 */
template <typename Serializable>
int readFromXmlSub(std::istream& is, Serializable& r,  const std::string& tag)
{
	using namespace boost::archive;
	using namespace boost::serialization;
	using namespace std;
	try {
		xml_iarchive ia(is);
		ia >> make_nvp(tag.c_str(), r);
	} catch (const exception& e) {
		cerr << "readFromXmlSub: ";
		cerr << "Exception thrown while Deserialization: ";
		cerr << e.what() << endl;
		return -1;
	}
	return 0;
}

} /* namespace XmlSerDes */


#endif    /*!XML_SERDES_HEADER_*/
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 8
 * indent-tabs-mode: t
 * End:
 *
 */
