/**
 * @file once.c
 * @brief implement of once function  
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "once.h"
#include "nes_posix_pthread.h"

/* Global lock use to once protected */
pthread_mutex_t __g_oncelock;

/**********************************************************************/
/* Function name: __once_init                                         */
/* Description:   Init once module                                    */
/* Return type: int return 0 if success, else return error code       */
/* Argument - void:                                                   */
/**********************************************************************/
int __once_init(void)
{
	return nes_posix_pthread_mutex_init(&__g_oncelock, NULL);
}

/**********************************************************************/
/* Function name: __once_finish                                       */
/* Description: Clean once module                                     */
/* Return type: int return 0 if success, else return error code       */
/* Argument - void:                                                   */
/**********************************************************************/
int __once_finish(void)
{
	return nes_posix_pthread_mutex_destroy(&__g_oncelock);
}

/**********************************************************************/
/* Function name: nes_posix_pthread_once                                        */
/* Description:Guarantee that the initialization function INIT_ROUTINE*/
/*             will be called only once, even if pthread_once is      */
/*             executed several times with the same ONCE_CONTROL      */
/*             argument. ONCE_CONTROL must point to a static or extern*/
/*             variable initialized to PTHREAD_ONCE_INIT.             */
/* The initialization functions might throw exception which is why    */
/* this function is not marked with __THROW.                          */
/*                                                                    */
/* Return type: int  Always return 0                                  */
/* Argument - pthread_once_t *__once_control:  once flag              */
/* Argument - __init_routinee:  once function                         */
/**********************************************************************/
int nes_posix_pthread_once(pthread_once_t *__once_control, void (*__init_routine) (void))
{
	/* 
	 *	Double lock check pattern
	 */
	if( *__once_control == PTHREAD_ONCE_INIT ) 
	{
		nes_posix_pthread_mutex_lock(&__g_oncelock);
		if( *__once_control == PTHREAD_ONCE_INIT ) 
		{
			__init_routine();
			*__once_control = PTHREAD_ONCE_DONE;
		}
		nes_posix_pthread_mutex_unlock(&__g_oncelock);
	}

	return 0;
}


