/**
 * @file internal_lock.c
 * @brief implement of internal_lock fucntion 
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <btron/errcode.h>
#include <btron/taskcomm.h>

#include "nes_posix_error.h"
#include "common.h"
#include "internal_lock.h"

/**********************************************************************/
/* Function name: __internal_lock_init                                */
/* Description:  Init internal lock                                   */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - __internal_lock_t* lock:                                */
/**********************************************************************/
int __internal_lock_init(__internal_lock_t* lock)
{
	WERR wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value
	
	if( !lock ) 
	{
		return EINVAL;
	}

	wCall = b_cre_sem(1, SEM_SYNC);

	/* Create successfully */
	if ( wCall > 0 )
	{
		*lock = wCall;		 /* save sem_t */
		return 0;
	} 

	/* Error code mapping */
	switch ( wCall )
	{
	case ER_OK:
		nRet = 0;
		break;
	case ER_LIMIT:
		nRet = ERANGE;
		break;
	case ER_ID:
	case ER_NOEXS:
	case ER_OBJ:
	default:
		nRet = EINVAL;
		break;
	}

	return nRet;
}

/**********************************************************************/
/* Function name: __internal_lock_destroy                             */
/* Description:  Destroy internal MUTEX.                              */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - __internal_lock_t* lock:                                */
/**********************************************************************/
int __internal_lock_destroy(__internal_lock_t* lock)
{
	WERR wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value

	if( !lock ) 
	{
		return EINVAL;
	}

	wCall = b_del_sem(*lock);

	/* Error code mapping */
	switch(wCall)
	{
	case ER_OK:
		nRet = 0;
		break;
	case ER_ID:			/* Invalid sem_t */
	case ER_NOEXS:
		nRet = EINVAL;
		break;
	default:				/* Unknow error */
		nRet = EBUSY;
	}

	return nRet;
}

/**********************************************************************/
/* Function name: __internal_lock                                     */
/* Description:  Wait until lock for internal MUTEX becomes available */
/*               and lock it.                                         */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - __internal_lock_t* lock:                                */
/**********************************************************************/
int __internal_lock(__internal_lock_t* lock)
{
	WERR wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value

	if( !lock ) 
	{
		return EINVAL;
	}

	wCall = b_wai_sem(*lock, T_FOREVER);

	switch(wCall)
	{
	case ER_OK:
		nRet = 0;
		break;
	case ER_ID:			/* Invalid sem_t */
	case ER_NOEXS:
	case ER_PAR:
		nRet = EINVAL;
		break;
	case ER_DLT:
	case ER_MINTR:
	case ER_NONE:
	case ER_OBJ:
		nRet = EBUSY;
		break;
	default:				/* Unknow error */
		nRet = 0;
	}

	return nRet;
}

/**********************************************************************/
/* Function name: __internal_unlock                                   */
/* Description: Unlock internal MUTEX                                 */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - __internal_lock_t* lock:                                */
/**********************************************************************/
int __internal_unlock(__internal_lock_t* lock)
{
	WERR wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value
	
	if( !lock ) 
	{
		return EINVAL;
	}
	wCall = b_sig_sem(*lock);

	switch ( wCall )
	{
	case ER_OK:
		nRet = 0;
		break;
	case ER_LIMIT:
		nRet = ERANGE;
		break;
	case ER_ID:
	case ER_NOEXS:
	case ER_OBJ:
	default:
		nRet = EINVAL;
		break;
	}

	return nRet;
}

/**********************************************************************/
/* Function name: __internal_try_lock                                 */
/* Description:  try lock                                             */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - __internal_lock_t* lock:                                */
/**********************************************************************/
int __internal_try_lock(__internal_lock_t* lock)
{
	WERR wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value

	if( !lock ) 
	{
		return EINVAL;
	}

	wCall = b_wai_sem(*lock, T_NOWAIT);

	switch(wCall)
	{
	case ER_OK:
		nRet = 0;
		break;
	case ER_ID:			/* Invalid sem_t */
	case ER_NOEXS:
	case ER_PAR:
		nRet = EINVAL;
		break;
	case ER_NONE:			/* Can't hold lock */
		nRet = EBUSY;
		break;
	case ER_DLT:
	case ER_MINTR:
	case ER_OBJ:
		nRet = EBUSY;
		break;
	default:				/* Unknow error */
		nRet = 0;
	}

	return nRet;
}
