/**
 * @file nes_posix_semaphore.c
 * @brief implement of semaphore operation
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <btron/errcode.h>

#include "nes_posix_error.h"
#include "common.h"
#include "nes_posix_semaphore.h"
#include "internal_lock.h"

/**********************************************************************/
/* Function name: nes_posix_sem_init                                            */
/* Description:  Initialize semaphore object SEM to VALUE.            */
/*               If PSHARED then share it with other processes.       */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*                  EINVAL: invalid params                            */
/*                  ERANGE: reach max semaphore count                 */
/*                  ENOMEM: not enough memory                         */
/* Argument - sem_t *sem:  semaphore handle                           */
/* Argument - int pshared:  share type. here only support pshare == 0 */
/* Argument - unsigned int value:  semaphore init resource value      */
/**********************************************************************/
int nes_posix_sem_init(sem_t *sem, int pshared, unsigned int value)
{
	WERR wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value

	/* Param invalid */
	if ( !sem ) 
	{
		nRet = EINVAL;		
		goto exit;
	}
	memset(sem, 0, sizeof(sizeof(sem_t)));

	/* the semaphore resource number exceed the limition */	
	if ( value > SEM_VALUE_MAX)
	{
		nRet = ERANGE;		
		goto exit;
	}

	/* Real create */
	wCall = b_cre_sem(value, SEM_SYNC);

	/* semaphore is created successfully */
	if ( wCall > 0 )
	{
		sem->semid = wCall;			/* save sem_t */
		sem->value = value;
		if( __internal_lock_init(&(sem->lock)) ) /* Create internal lock failed, all is failed */
		{
			b_del_sem(wCall);
			nRet = EINVAL;
		}
		nRet = 0;				 /* Create successfully */
		goto exit;
	} 

	/* Error code mapping */
	switch(wCall)
	{
	case ER_NOSPC:			/* No memory */
		nRet = ENOMEM;
		break;
	case ER_LIMIT:			/* Reach out semaphore max count limit */
		nRet = ERANGE;		/* not ERANGE; */
		break;
	case ER_PAR:			/* Invalid parameter */
		nRet = EINVAL;
		break;
	default:			/* Unknow error */
		nRet = 0;
	}
	
exit:
	return nRet;
}


/**********************************************************************/
/* Function name: nes_posix_sem_destroy                                         */
/* Description: Free resources associated with semaphore object SEM.  */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*              EINVAL: invalid params                            */
/* Argument - sem_t *sem:  semaphore handle                           */
/**********************************************************************/
int nes_posix_sem_destroy(sem_t *sem) 
{		
	WERR wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value
	
	/*
	* Get current resource count of semaphore
	*/
	if ( !sem )
	{
		nRet = EINVAL;
		goto exit;
	}

	/* Destroy internal lock that protect counter */
	nRet = __internal_lock_destroy(&(sem->lock));

	/* Destroy really semaphore */
	wCall = b_del_sem(sem->semid);

	/* Error code mapping */
	switch(wCall)
	{
	case ER_OK:
		nRet = 0;
		break;
	case ER_ID:			/* Invalid sem_t */
	case ER_NOEXS:
		nRet = EINVAL;
		break;
	default:			/* Unknow error */
		nRet = EBUSY;
	}

exit:
	return nRet;
}


/**********************************************************************/
/* Function name: nes_posix_sem_wait                                            */
/* Description:   Wait for SEM being posted.                          */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*                  EINVAL: invalid params                            */
/*                  EBUSY: semaphore wait failed                      */
/* Argument - sem_t *sem:  semaphore handle                           */
/**********************************************************************/
int nes_posix_sem_wait(sem_t *sem)
{	
	WERR wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value

	/*
	 * At linux, this function always return 0, 
	 * here we return some error codes when what hppend
	 */
	if( !sem )
	{
		nRet = EINVAL;
		DP1(D_MODU_SEM, ("sem_wait() Failed for param is invalid!\n"));		
		goto exit;
	}

	wCall = b_wai_sem(sem->semid, T_FOREVER);
	/* Error code mapping */
	switch(wCall)
	{
	case ER_OK:
		__internal_lock(&(sem->lock));			/* Decrease resource count with lock */
		sem->value--;
		__internal_unlock(&(sem->lock));
		nRet = 0;					/* Return ok */
		break;
	case ER_ID:						/* Invalid sem_t */
	case ER_NOEXS:
	case ER_PAR:
		nRet = EINVAL;
		break;
	case ER_DLT:
	case ER_MINTR:
	case ER_NONE:
	case ER_OBJ:
		nRet = EBUSY;
		break;
	default:						/* Unknow error */
		nRet = EBUSY;
	}
exit:	
	return nRet;
}


/**********************************************************************/
/* Function name: nes_posix_sem_trywait                                         */
/* Description:   Test whether SEM is posted.                         */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*                  EINVAL: invalid params                            */
/*                  EBUSY: semaphore wait failed                      */
/* Argument - sem_t *sem:  semaphore handle                           */
/**********************************************************************/
int nes_posix_sem_trywait(sem_t *sem) 
{
	WERR wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value

	if ( !sem )
	{
		nRet  = EINVAL;
		goto exit;
	}

	wCall = b_wai_sem(sem->semid, T_NOWAIT); 

	/* Error code mapping */
	switch(wCall)
	{
	case ER_OK:
		__internal_lock(&(sem->lock));	/* Decrease resource count with lock */
		sem->value--;
		__internal_unlock(&(sem->lock));
		nRet = 0;					/* Return ok */
		break;
	case ER_ID:						/* Invalid sem_t */
	case ER_NOEXS:
	case ER_PAR:
		nRet = EINVAL;
		break;
	case ER_DLT:
	case ER_MINTR:
	case ER_NONE:
	case ER_OBJ:
		nRet = EBUSY;
		break;
	default:						/* Unknow error */
		nRet = EAGAIN;
	}

exit:
	return nRet;
}


/**********************************************************************/
/* Function name: nes_posix_sem_post                                            */
/* Description:  Post SEM.                                            */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*                  EINVAL: invalid params                            */
/*                  ERANGE: reach max semaphore count                 */
/*                  EBUSY: semaphore wait failed                      */
/* Argument - sem_t *sem:  semaphore handle                           */
/**********************************************************************/
int nes_posix_sem_post(sem_t *sem) 
{
	WERR wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value

	if( !sem )
	{
		nRet = EINVAL;
		goto exit;
	}

	wCall = b_sig_sem(sem->semid);

	switch ( wCall )
	{
	case ER_OK:
		__internal_lock(&(sem->lock));			/* Decrease resource count with lock */
		sem->value++;
		__internal_unlock(&(sem->lock));
		nRet = 0;					/* Return ok */
		break;
	case ER_LIMIT:
		nRet = ERANGE;
		break;
	case ER_ID:
	case ER_NOEXS:
	case ER_OBJ:
	default:
		nRet = EINVAL;
		break;
	}

exit:
	return nRet;
}


/**********************************************************************/
/* Function name: nes_posix_sem_getvalue                                        */
/* Description:   Get current value of SEM and store it in *SVAL.     */
/*                But not support here                                */
/* Return type:   0 Successfully                                      */
/*                EINVAL param is invalid                             */
/* Argument - sem_t *sem:  semaphore handle                           */
/**********************************************************************/
int nes_posix_sem_getvalue(sem_t *sem, int *sval)
{
	/* Invalid param */
	if( !sem ) {
		return EINVAL;
	}

	/*
	 * Get current resource count of semaphore
	 */
	__internal_lock(&(sem->lock));
	*sval = sem->value;
	__internal_unlock(&(sem->lock));

	return 0;
}
