/**
 * @file nes_posix_semaphore.c
 * @define implement of semaphore operation
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <string.h>
#include <tkse/extension/typedef.h>
#include <tkse/extension/tkcall.h>

#include "nes_posix_error.h"
#include "common.h"
#include "nes_posix_semaphore.h"

/**********************************************************************/
/* Function name: nes_posix_sem_init                                            */
/* Description:  Initialize semaphore object SEM to VALUE.            */
/*               If PSHARED then share it with other processes.       */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*                  EINVAL: invalid params                            */
/*                  ERANGE: reach max semaphore count                 */
/*                  ENOMEM: not enough memory                         */
/* Argument - sem_t *sem:  semaphore handle                           */
/* Argument - int pshared:  share type. here only support pshare == 0 */
/* Argument - unsigned int value:  semaphore init resource value      */
/**********************************************************************/
int nes_posix_sem_init(sem_t *sem, int pshared, unsigned int value)
{
	INT wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value
	T_CSEM csem;

	/* Param invalid */
	if ( !sem ) 
	{
		nRet = EINVAL;		
		goto exit;
	}
	memset(sem, 0, sizeof(sizeof(sem_t)));

	/* the semaphore resource number exceed the limition */	
	if ( value > SEM_VALUE_MAX)
	{
		nRet = ERANGE;		
		goto exit;
	}

	/* Real create */
	csem.exinf = NULL;
	csem.sematr = TA_TFIFO | TA_FIRST | TA_DELEXIT;
	csem.isemcnt = value;
	csem.maxsem = SEM_VALUE_MAX;
	csem.dsname[0] = 0;

	wCall = tkse_cre_sem(&csem);

	/* semaphore is created successfully */
	if ( wCall > 0 )
	{
		sem->semid = wCall;			/* save sem_t */
		nRet = 0;				 /* Create successfully */
		goto exit;
	} 

	/* Error code mapping */
	switch( wCall )
	{
	case E_OK:
		nRet = 0;
		break;
	case E_NOMEM:
		nRet = ENOMEM;
		break;
	case E_LIMIT:
		nRet = ERANGE;
		break;
	case E_ID:
	case E_PAR:
	case E_RSATR:
	default:
		nRet = EINVAL;
		break;
	}

exit:
	return nRet;
}


/**********************************************************************/
/* Function name: nes_posix_sem_destroy                                         */
/* Description: Free resources associated with semaphore object SEM.  */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*              EINVAL: invalid params                            */
/* Argument - sem_t *sem:  semaphore handle                           */
/**********************************************************************/
int nes_posix_sem_destroy(sem_t *sem) 
{		
	INT wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value
	
	/*
	* Get current resource count of semaphore
	*/
	if ( !sem )
	{
		nRet = EINVAL;
		goto exit;
	}

	wCall = tkse_del_sem(sem->semid);

	/* Error code mapping */
	switch(wCall)
	{
	case E_OK:
		nRet = 0;
		break;
	case E_ID: 		/* Invalid ID number */			
	case E_NOEXS: 		/* Object does not exsit */
	default:
		nRet = EINVAL;
		break;
	}

exit:
	return nRet;
}


/**********************************************************************/
/* Function name: nes_posix_sem_wait                                            */
/* Description:   Wait for SEM being posted.                          */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*                  EINVAL: invalid params                            */
/*                  EBUSY: semaphore wait failed                      */
/* Argument - sem_t *sem:  semaphore handle                           */
/**********************************************************************/
int nes_posix_sem_wait(sem_t *sem)
{	
	INT wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value

	/*
	 * At linux, this function always return 0, 
	 * here we return some error codes when what hppend
	 */
	if( !sem )
	{
		nRet = EINVAL;
		DP1(D_MODU_SEM, ("sem_wait() Failed for param is invalid!\n"));		
		goto exit;
	}

	wCall = tkse_wai_sem(sem->semid, 1, T_FOREVER);

	/* Error code mapping */
	switch(wCall)
	{
	case E_OK:
		nRet = 0;
		break;
	case E_ID:		
	case E_NOEXS:
	case E_PAR:
	case E_CTX:
		nRet = EINVAL;
		break;
	case E_DLT:
	case E_RLWAI:
	case E_DISWAI:
		nRet = EINTR;
		break;
	case E_ILUSE:
		nRet = EDEADLK;
		break;
	case E_TMOUT: 				/* Never reach here */
		nRet = EBUSY;
		break;
	default:				/* Unknow error */
		nRet = EINVAL;
	}
exit:	
	return nRet;
}


/**********************************************************************/
/* Function name: nes_posix_sem_trywait                                         */
/* Description:   Test whether SEM is posted.                         */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*                  EINVAL: invalid params                            */
/*                  EBUSY: semaphore wait failed                      */
/* Argument - sem_t *sem:  semaphore handle                           */
/**********************************************************************/
int nes_posix_sem_trywait(sem_t *sem) 
{
	INT wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value

	if ( !sem )
	{
		nRet  = EINVAL;
		goto exit;
	}

	wCall = tkse_wai_sem(sem->semid, 1, T_NOWAIT); 

	/* Error code mapping */
	switch(wCall)
	{
	case E_OK:
		nRet = 0;
		break;
	case E_ID:			/* Invalid sem_t */
	case E_NOEXS:
	case E_PAR:
	case E_CTX:
		nRet = EINVAL;
		break;
	case E_DLT:
	case E_RLWAI:
	case E_DISWAI:
		nRet = EINTR;
		break;
	case E_ILUSE:
		nRet = EDEADLK;
		break;
	case E_TMOUT:
		nRet = EBUSY;
		break;
	default:				/* Unknow error */
		nRet = EINVAL;
	}
exit:	
	return nRet;
}


/**********************************************************************/
/* Function name: nes_posix_sem_post                                            */
/* Description:  Post SEM.                                            */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*                  EINVAL: invalid params                            */
/*                  ERANGE: reach max semaphore count                 */
/*                  EBUSY: semaphore wait failed                      */
/* Argument - sem_t *sem:  semaphore handle                           */
/**********************************************************************/
int nes_posix_sem_post(sem_t *sem) 
{
	INT wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value

	if( !sem )
	{
		nRet = EINVAL;
		goto exit;
	}

	wCall = tkse_sig_sem(sem->semid, 1);

	switch ( wCall )
	{
	case E_OK:
		nRet = 0;
		break;
	case E_ID:
	case E_NOEXS:
		nRet = EINVAL;
		break;
	case E_ILUSE:
		nRet = EPERM;
		break;
	default:
		nRet = EINVAL;
		break;
	}
exit:
	return nRet;
}


/**********************************************************************/
/* Function name: nes_posix_sem_getvalue                                        */
/* Description:   Get current value of SEM and store it in *SVAL.     */
/*                But not support here                                */
/* Return type:   0 Successfully                                      */
/*                EINVAL param is invalid                             */
/* Argument - sem_t *sem:  semaphore handle                           */
/**********************************************************************/
int nes_posix_sem_getvalue(sem_t *sem, int *sval)
{
	INT err = 0;
	T_RSEM rsem = {0};

	/* Invalid param */
	if( !sem || !sval ) {
		return EINVAL;
	}

	err = tkse_ref_sem(sem->semid, &rsem);
	if( err < 0 ) {
		return EINVAL;
	}

	*sval = rsem.semcnt;
	return 0;
}
