/**
 * @file nes_posix_access.c
 * @brief This file is part of the posix library
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <tstring.h>
#include <extension/file.h>
#include <extension/seio.h>
#include <tkse/errno.h>
#include <tk/errno.h>
#include <tcode.h>

#include "common.h"
#include "nes_posix_access.h"


/*****************************************************************************
 *
 * Function     : conv_pathname
 * Purpose      : convert cli's path name to TC 
 *                
 * Returns      : void
 *
 ****************************************************************************/
static void conv_pathname( const char *src, TC *dest )
{
	char c;
	const char *p;
	int n, k;
	TC buf;

	p = src;
	n = 0;

#ifndef PROCESS_BASE
	if (*p != '/') {
		dest[n++] = TC_FDLM;
		dest[n++] = TK_S;
		dest[n++] = TK_Y;
		dest[n++] = TK_S;
		dest[n++] = TC_FDLM;
	}
#endif
	for ( ; (c = *p) != '\0' && n < FILENAME_MAX - 1; p++) {
		if (c == '/') {
			dest[n++] = TC_FDLM;
		} else if (c == ':') {
			dest[n++] = TC_FSEP;
		} else if ((k = euctotc(&buf, p)) > 0) {
			dest[n++] = buf;
			p += (k - 1);
		}
	}
	dest[n] = 0x0000;
}



/**
 * access - check user's permissions for a file
 * @param 	pathname: file name
 * @param   mode: attribute
 *          R_OK: read 
 *          W_OK: write 
 *          X_OK: exexute
 *          F_OK: read  
 * @retval int
 *      		0: succeed
 *          -1: do not suit to the attibute
 *              EACCES : The requested access would be denied
 *                       to the file or search permission is denied
 *                       for one of the directories in the path 
 *                       prefix of pathname. (See also path_resolution(2).) 
 *              EROFS :  Write permission was requested for a file on a read-only filesystem.
 *              EFAULT:  pathname points outside your accessible address space. 
 *              EINVAL:  mode was incorrectly specified. 
 *              EIO   :  An I/O error occurred. 
 *              ENOMEM : Insufficient kernel memory was available. 
 * 
 */
int nes_posix_access(const char *pathname, int mode)
{
	int wRet = 0;
	struct stat tmpStat;
	int tmpErrFlag=0;
	if(!pathname || (strlen(pathname) == 0))
	{
		errno = EINVAL;
		return -1;
	}
	if(mode!=F_OK)
	{
		if(!((mode&W_OK)||(mode&R_OK)||(mode&X_OK)))
		{
			errno = EINVAL;
			return -1;
		}
	}
	wRet=stat(pathname,&tmpStat);
	if(wRet==0)
	{
		
		if(mode&W_OK)
		{
			if(!(tmpStat.st_mode&S_IWUSR))
			{
				errno=EACCES;
			        tmpErrFlag=1;
			}
		}
		if(mode&R_OK)
		{
			if(!(tmpStat.st_mode&S_IRUSR))
			{
				errno=EACCES;
			        tmpErrFlag=1;
			}

		}
		if(mode&X_OK)
		{
			if(!(tmpStat.st_mode&S_IXUSR))
			{
				errno=EACCES;
			        tmpErrFlag=1;
			}

		}
		//printf("st_mode=[%08x]\n",tmpStat.st_mode);
		if(tmpErrFlag)
		{
			return -1;
		}
	}
	else
	{
		
		printf("access error %d %s\n",wRet,pathname);
		errno=EINVAL;	
		return -1;	
	}
	return wRet;
}

