// -*- C++ -*-
/*!
 * @file ConfigAdmin.h
 * @brief Configuration Administration classes
 * @date $Date: 2007-12-31 03:08:02 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2007-2009
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: ConfigAdmin.h 1819 2010-01-28 00:52:47Z fsi-katami $
 *
 */

#ifndef RTC_CONFIGADMIN_H
#define RTC_CONFIGADMIN_H

#include <coil/Properties.h>
#include <coil/stringutil.h>
#include <string>
#include <vector>
#include <iostream>

/*!
 * @if jp
 * @namespace RTC
 *
 * @brief RTݡͥ
 *
 * @else
 *
 * @namespace RTC
 *
 * @brief RT-Component
 *
 * @endif
 */
namespace RTC
{
  /*!
   * @if jp
   * @class OnUpdateCallback
   * @brief OnUpdate Хåݥ饹
   *
   * @else
   * @class OnUpdateCallback
   * @brief Callback functor abstract for OnUpdate
   *
   * @endif
   */
  class OnUpdateCallback
  {
  public:
    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     *
     * ۥǥȥ饯
     *
     * @else
     *
     * @brief Virtual destructor
     *
     * Virtual Destructor
     *
     * @endif
     */
    virtual ~OnUpdateCallback(void){};
    /*!
     * @if jp
     *
     * @brief ۥХå᥽å
     *
     * ե졼ѥ᡼ι(ID)˸ƤӽФ
     * Хå᥽å
     *
     * @else
     *
     * @brief Virtual Callback method
     *
     * This is the callback method invoked when the update of the 
     * configuration parameter (ID specification).
     *
     * @endif
     */
    virtual void operator()(const char* config_set) = 0;
  };

  /*!
   * @if jp
   * @class OnUpdateParamCallback
   * @brief OnUpdateParam Хåݥ饹
   *
   * @else
   * @class OnUpdateParamCallback
   * @brief Callback functor abstract for OnUpdateParam
   *
   * @endif
   */
  class OnUpdateParamCallback
  {
  public:
    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     *
     * ۥǥȥ饯
     *
     * @else
     *
     * @brief Virtual destructor
     *
     * Virtual Destructor
     *
     * @endif
     */
    virtual ~OnUpdateParamCallback(void){};
    /*!
     * @if jp
     *
     * @brief ۥХå᥽å
     *
     * ե졼ѥ᡼ι(̾λ)˸ƤӽФ
     * Хå᥽å
     *
     * @else
     *
     * @brief Virtual Callback method
     *
     * This is the callback method invoked when the update of the 
     * configuration parameter (name specification).
     *
     * @endif
     */
    virtual void operator()(const char* config_set, const char* config_param) = 0;
  };

  /*!
   * @if jp
   * @class OnSetConfigurationSetCallback
   * @brief OnSetConfigurationSet Хåݥ饹
   *
   * @else
   * @class OnSetConfigurationSetCallback
   * @brief Callback functor abstract for OnSetConfigurationSet
   *
   * @endif
   */
  class OnSetConfigurationSetCallback
  {
  public:
    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     *
     * ۥǥȥ饯
     *
     * @else
     *
     * @brief Virtual destructor
     *
     * Virtual Destructor
     *
     * @endif
     */
    virtual ~OnSetConfigurationSetCallback(void){};
    /*!
     * @if jp
     *
     * @brief ۥХå᥽å
     *
     * ꤷץѥƥΥե졼󥻥åȤؤɲä줿
     * 뤵륳Хå᥽å
     *
     * @else
     *
     * @brief Virtual Callback method
     *
     * This is the callback method invoked when added to the configuration 
     * set of the property that specifies it. 
     *
     * @endif
     */
    virtual void operator()(const coil::Properties& config_set) = 0;
  };

  /*!
   * @if jp
   * @class OnAddConfigurationAddCallback
   * @brief OnAddConfigurationAdd Хåݥ饹
   *
   * @else
   * @class OnAddConfigurationAddCallback
   * @brief callback functor abstract for OnAddConfigurationAdd
   *
   * @endif
   */
  class OnAddConfigurationAddCallback
  {
  public:
    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     *
     * ۥǥȥ饯
     *
     * @else
     *
     * @brief Virtual destructor
     *
     * Virtual Destructor
     *
     * @endif
     */
    virtual ~OnAddConfigurationAddCallback(void){};
    /*!
     * @if jp
     *
     * @brief ۥХå᥽å
     *
     * ե졼󥻥åȤͤɲä줿Ȥ˥뤵
     * Хå᥽å
     *
     * @else
     *
     * @brief Virtual Callback method
     *
     * This is the callback method invoked when a set value is added to the 
     * configuration set. 
     *
     * @endif
     */
    virtual void operator()(const coil::Properties& config_set) = 0;
  };

  /*!
   * @if jp
   * @class OnRemoveConfigurationSetCallback
   * @brief OnRemoveConfigurationSet Хåݥ饹
   *
   * @else
   * @class OnRemoveConfigurationSetCallback
   * @brief Callback functor abstract for OnRemoveConfigurationSet
   *
   * @endif
   */
  class OnRemoveConfigurationSetCallback
  {
  public:
    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     *
     * ۥǥȥ饯
     *
     * @else
     *
     * @brief Virtual destructor
     *
     * Virtual Destructor
     *
     * @endif
     */
    virtual ~OnRemoveConfigurationSetCallback(void){};
    /*!
     * @if jp
     *
     * @brief ۥХå᥽å
     *
     * ե졼󥻥åȤƤȤ˥뤵
     * Хå᥽å
     *
     * @else
     *
     * @brief Virtual Callback method
     *
     * This is the callback method invoked when the configuration set has 
     * been deleted. 
     *
     * @endif
     */
    virtual void operator()(const char* config_set) = 0;
  };

  /*!
   * @if jp
   * @class OnActivateSetCallback
   * @brief OnActivateSet Хåݥ饹
   *
   * @else
   * @class OnActivateSetCallback
   * @brief Callback functor abstract for OnActivateSet
   *
   * @endif
   */
  class OnActivateSetCallback
  {
  public:
    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     *
     * ۥǥȥ饯
     *
     * @else
     *
     * @brief Virtual destructor
     *
     * Virtual Destructor
     *
     * @endif
     */
    virtual ~OnActivateSetCallback(void){};
    /*!
     * @if jp
     *
     * @brief ۥХå᥽å
     *
     * ե졼󥻥åȤƥֲ줿Ȥ˥뤵
     * Хå᥽å
     *
     * @else
     *
     * @brief Virtual Callback method
     *
     * This is the callback method invoked when * the configuration set is 
     * made active.
     *
     * @endif
     */
    virtual void operator()(const char* config_id) = 0;
  };

  //============================================================
  // ConfigBase class
  //============================================================
  /*!
   * @if jp
   * @class ConfigBase
   * @brief ConfigBase ݥ饹
   * 
   * Ƽ拾ե졼ݻ뤿ݥ饹ݥ
   * ե졼󥯥饹ϡʲν貾۴ؿμ󶡤ʤ
   * Фʤʤ
   *
   * public󥿡եȤưʲΤΤ󶡤롣
   * - update(): ե졼ѥ᡼ͤι
   *
   * @since 0.4.0
   *
   * @else
   * @class ConfigBase
   * @brief ConfigBase abstract class
   *
   * This is the abstract interface class to hold various configuration 
   * information.
   * Concrete configuration classes must implement the following pure virtual
   * functions.
   *
   * This class provides public interface as follows.
   * - update(): update configuration parameter value
   *
   * @since 0.4.0
   *
   * @endif
   */
  struct ConfigBase
  {
    /*!
     * @if jp
     *
     * @brief 󥹥ȥ饯
     * 
     * 󥹥ȥ饯
     *
     * @param name_ ե졼̾
     * @param def_val ʸΥǥե
     * 
     * @else
     *
     * @brief Constructer
     *
     * Constructer
     *
     * @param name_ Configuration name
     * @param def_val Default value in string format
     *
     * @endif
     */
    ConfigBase(const char* name_, const char* def_val)
      : name(name_), default_value(def_val) {}
    
    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     * 
     * ۥǥȥ饯
     *
     * @else
     *
     * @brief Virtual Destructor
     *
     * Virtual Destructor
     *
     * @endif
     */
    virtual ~ConfigBase(void){};
    
    /*!
     * @if jp
     *
     * @brief ե졼ѥ᡼͹ѽ貾۴ؿ
     * 
     * ե졼ͤǥե졼ѥ᡼
     * 뤿ν貾۴ؿ
     *
     * @param val ѥ᡼ͤʸɽ
     *
     * @return 
     * 
     * @else
     *
     * @brief Pure virtual function to update configuration parameter values
     * 
     * Pure virtual function to update configuration parameter 
     * by the configuration value.
     *
     * @param val The parameter values converted into character string format
     *
     * @return Result of the setup
     *
     * @endif
     */
    virtual bool update(const char* val) = 0;
    
    /*!
     * @if jp
     * @brief  ե졼̾
     * @else
     * @brief  Configuration name
     * @endif
     */
    const char* name;
    
    /*!
     * @if jp
     * @brief  ʸΥǥե
     * @else
     * @brief  Default value in string format
     * @endif
     */
    const char* default_value;
  };
  
  //============================================================
  // Config template class
  //============================================================
  /*!
   * @if jp
   * @class Config
   * @brief Config 饹
   * 
   * ե졼ѥ᡼ξݻ륯饹
   * \<VarType\>Ȥƥե졼Υǡꤹ롣
   * \<TransFunc\>Ȥꤵ줿ǡʸѴѴؿ
   * ꤹ롣
   *
   * @param VarType ե졼ѥ᡼Ǽѿ
   * @param TransFunc ǼǡʸѴѴؿ
   *
   * @since 0.4.0
   *
   * @else
   * @class Config
   * @brief Config class
   * 
   * Class to hold the configuration parameter information.
   * Specify the data type of the configuration as \<VarType\>
   * Specify transformation function to convert data type set as \<TransFunc\>
   * into string format.
   *
   * @param VarType Cariable to hold configuration parameter
   * @param TransFunc Transformation function to transform the stored data 
   * type into string format.
   *
   * @since 0.4.0
   *
   * @endif
   */
  template <typename VarType,
	    typename TransFunc = bool (*)(VarType&, const char*)>
  class Config
    : public ConfigBase
  {
  public:
    /*!
     * @if jp
     *
     * @brief 󥹥ȥ饯
     * 
     * 󥹥ȥ饯
     *
     * @param name ե졼ѥ᡼̾
     * @param var ե졼ѥ᡼Ǽѿ
     * @param def_val ʸΥǥե
     * @param trans ʸѴؿ
     * 
     * @else
     *
     * @brief Constructor
     * 
     * Constructor
     *
     * @param name Configuration parameter name
     * @param var Configuration parameter variable
     * @param def_val Default value in string format
     * @param trans Function to transform into string format
     *
     * @endif
     */
    Config(const char* name, VarType& var, const char* def_val,
	   TransFunc trans = coil::stringTo)
      : ConfigBase(name, def_val), m_var(var), m_trans(trans)
    {
    }
    
    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     * 
     * ۥǥȥ饯
     *
     * @else
     *
     * @brief Virtual Destructor
     * 
     * Virtual Destructor.
     *
     * @endif
     */
    virtual ~Config(void){}
    
    /*!
     * @if jp
     *
     * @brief Хɥѥ᡼ͤ򹹿
     * 
     * ե졼ͤǥե졼ѥ᡼򹹿
     *
     * @param val ѥ᡼ͤʸɽ
     *
     * @return (:true:false)
     * 
     * @else
     *
     * @brief Update a bind parameter value
     * 
     * Update configuration paramater by the configuration value.
     *
     * @param val The parameter values converted into character string format
     *
     * @return Update result (Successful:true, Failed:false)
     *
     * @endif
     */
    virtual bool update(const char* val)
    {
      if ((*m_trans)(m_var, val)) { return true; }
      (*m_trans)(m_var, default_value);
      return false;
    }
    
  protected:
    /*!
     * @if jp
     * @brief  ե졼ѥ᡼Ǽѿ
     * @else
     * @brief  Configuration parameter variable
     * @endif
     */
    VarType& m_var;
    
    /*!
     * @if jp
     * @brief  ե졼ѥ᡼ʸѴؿ
     * @else
     * @brief  Transformation function to convert configuration parameter type 
     *         into string format.
     * @endif
     */
    TransFunc m_trans;
  };
  
  //============================================================
  // ConfigAdmin class
  //============================================================
  /*!
   * @if jp
   * @class ConfigAdmin
   * @brief ConfigAdmin 饹
   * 
   * Ƽ拾ե졼륯饹
   * ѸʲΤ褦롣
   *
   * - ե졼: ݡͥȤ
   *
   * - (ե졼)ѥ᡼ key-value ʤ
   *   coil::Properties ѿȤư졢keyvalue ʸȤ
   *   롣key 򥳥ե졼ѥ᡼̾value 򥳥
   *   ե졼ѥ᡼ͤȸƤ֡
   *
   * - ե졼󥻥åȡ ե졼ѥ᡼
   *   ΥꥹȤǡ̾ (ID) ˤäƶ̤롣ID򥳥ե졼
   *   󥻥åIDȸƤ֡
   *
   * - (ե졼)ѥ᡼ѿե졼
   *   ᡼RTCΥƥӥƥǼºݤѤݤ˻Ȥ
   *   ѥ᡼Ȥ˸ͭηġ
   *
   * - ƥ(ե졼)åȡͭʥե
   *   졼󥻥åȤΤȤǤꡢͣ¸ߤ롣§Ȥơƥ
   *   ֥ե졼󥻥åȤΥѥ᡼ե졼
   *   ѥ᡼ѿȿǤ롣
   *
   * Υ饹Ǥϡե졼Τΰʲ2Ĥξ
   * Ƥ롣
   *
   * -# ե졼󥻥åȤΥꥹ
   * -# ѥ᡼ѿΥꥹ
   *
   * Ūˤϡ(1) Υե졼󥻥åȤΥꥹȤΤĤ
   * (2) Υѥ᡼ѿȿǤ롢Τܥ饹ŪǤ롣̾
   * ѥ᡼ѿѹϡե졼󥻥åȤѹȥ
   * ᡼ѿؤȿǤ2ʳǹԤ롣
   *
   * ե졼󥻥åȤΥꥹȤˤϡʲδؿѤ롣
   *
   * - getConfigurationSets()
   * - getConfigurationSet()
   * - setConfigurationSetValues()
   * - getActiveConfigurationSet()
   * - addConfigurationSet()
   * - removeConfigurationSet()
   * - activateConfigurationSet()
   *
   * δؿˤꡢե졼󥻥åȤѹɲá
   * ƥֲԤˤѹ줿ե
   * 졼󥻥åȤRTCΥƥӥƥѤѥ᡼ѿ
   * ȿǤˤϡʲ update() ؿѤ롣
   *
   * - update(void)
   * - update(const char* config_set)
   * - update(const char* config_set, const char* config_param)
   *
   * ե졼եå뤿˥Хåե󥯥
   * Ϳ뤳ȤǤ롣եåǤϰʲ̤ꡣ
   *
   * - ON_UPDATE                   : update() 
   * - ON_UPDATE_PARAM             : update(param) 
   * - ON_SET_CONFIGURATIONSET     : setConfigurationSet() 
   * - ON_ADD_CONFIGURATIONSET     : addConfigurationSet() 
   * - ON_REMOVE_CONFIGURATIONSET  : removeConfigurationSet() 
   * - ON_ACTIVATE_CONFIGURATIONSET: activateConfigurationSet() 
   *
   * @since 0.4.0
   *
   * @else
   * @class ConfigAdmin
   * @brief ConfigAdmin class
   * 
   * Class to manage various configuration information.
   * Now terms for this class are defined as follows.
   *
   * - Configurations: The configuration information for the RTCs.
   *
   * - (Configuration) parameters: Configuration information that
   *   consists of a key-value pair. The "key" and the "value" are
   *   both stored as character string values in a coil::Properties
   *   variable in this class. The "key" is called the "configuration
   *   parameter name", and the "value" is called the "configuration
   *   parameter value".
   *
   * - Configuration-sets: This is a list of configuration parameters,
   *   and it is distinguished by name (ID). The ID is called
   *   configuration-set ID.
   *
   * - (Configuration) parameter variables: The variables to be
   *   referred when configuration parameters are actually used within
   *   the activity of an RTC. Each variable has each type.
   *
   * - Active (configuration) set: This is the only configuration-set
   *   that is currently active. The parameter values of the active
   *    configuration-set are substituted into configuration variables
   *   in principle.
   *
   * The following two configuration informations are stored in this class.
   *
   * -# A list of configuration-set
   * -# A list of configuration parameter variables
   *
   * Basically, the purpose of this class is to set one of the
   * configuration-set in the list of (1) into parameter variables of
   * (2). Usually, configuration parameter variables manipulation is
   * performed with two-phases of configuration-set setting and
   * parameter variables setting.
   *
   * The configuration-set manipulations are performed by the
   * following functions.
   *
   * - getConfigurationSets()
   * - getConfigurationSet()
   * - setConfigurationSetValues()
   * - getActiveConfigurationSet()
   * - addConfigurationSet()
   * - removeConfigurationSet()
   * - activateConfigurationSet()
   *
   * Modification, addition, deletion, acquisition and activation of
   * configuration-set are performed by these functions. In order to
   * reflect configuration-set, which is manipulated by these
   * functions, on parameter variables that are used from RTC
   * activities, the following update() functions are used .
   *
   * - update(void)
   * - update(const char* config_set)
   * - update(const char* config_set, const char* config_param)
   *
   * Callback functors can be given to hook configuration
   * operation. Operations to be hooked are as follows.
   *
   * - ON_UPDATE                   : when update() is called
   * - ON_UPDATE_PARAM             : when update(param) is called
   * - ON_SET_CONFIGURATIONSET     : when setConfigurationSet() is called
   * - ON_ADD_CONFIGURATIONSET     : when addConfigurationSet() is called
   * - ON_REMOVE_CONFIGURATIONSET  : when removeConfigurationSet() is called
   * - ON_ACTIVATE_CONFIGURATIONSET: when activateConfigurationSet() is called
   *
   * @since 0.4.0
   *
   * @endif
   */
  class ConfigAdmin
  {
  public:
    /*!
     * @if jp
     *
     * @brief 󥹥ȥ饯
     * 
     * 󥹥ȥ饯
     *
     * @param prop оݥץѥƥ̾
     * 
     * @else
     *
     * @brief Constructor
     * 
     * Constructor
     *
     * @param prop The target property name for setup
     *
     * @endif
     */
    ConfigAdmin(coil::Properties& prop);
    
    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     * 
     * ۥǥȥ饯
     *
     * @else
     *
     * @brief Virtual Destructor
     * 
     * Virtual Destructor
     *
     * @endif
     */
    ~ConfigAdmin(void);
    
    /*!
     * @if jp
     *
     * @brief ե졼ѥ᡼
     * 
     * ե졼ѥ᡼ѿХɤ
     * ꤷ̾ΤΥե졼ѥ᡼¸ߤ
     * false֤
     * \<VarType\>Ȥƥե졼ѥ᡼Υǡꤹ롣
     *
     * @param param_name ե졼ѥ᡼̾
     * @param var ե졼ѥ᡼Ǽѿ
     * @param def_val ե졼ѥ᡼ǥե
     * @param trans ե졼ѥ᡼ʸѴѴؿ
     *
     * @return (:true꼺:false)
     * 
     * @else
     *
     * @brief Setup for configuration parameters
     * 
     * Bind configuration parameter to its variable.
     * Return false, if configuration parameter of specified name has already 
     * existed.
     * Specify the data type of the configuration as \<VarType\>.
     *
     * @param param_name Configuration parameter name
     * @param var Configuration parameter variable
     * @param def_val Default value of configuration parameter
     * @param trans Function to transform configuration parameter type into 
     *        string format
     *
     * @return Setup result (Successful:true, Failed:false)
     *
     * @endif
     */
    template <typename VarType>
    bool bindParameter(const char* param_name, VarType& var,
		       const char* def_val,
		       bool (*trans)(VarType&, const char*) = coil::stringTo)
    {
      if (param_name == 0) { return false; }
      if (def_val == 0) { return false; }
      if (isExist(param_name)) { return false; }
      if (!trans(var, def_val)) { return false; }
      m_params.push_back(new Config<VarType>(param_name, var, def_val, trans));
      return true;
    }
        
    /*!
     * @if jp
     *
     * @brief ե졼ѥ᡼ι
     *        (ƥ֥ե졼󥻥å)
     * 
     * ե졼󥻥åȤƤˡߥƥ
     * ֤ˤʤäƤ륳ե졼ꤷͤǡե
     * 졼ѥ᡼ͤ򹹿롣νǤιϡƥ
     * ֤ȤʤäƤ륳ե졼󥻥åȤ¸ߤƤ硢
     * ι饳ե졼󥻥åȤƤƤ
     * Τ߼¹Ԥ롣
     *
     * @else
     *
     * @brief Update the values of configuration parameters
     *        (Active configuration set)
     * 
     * When configuration set is updated, update the configuration
     * parameter value to the value that is set to the current active
     * configuration.  This update will be executed, only when an
     * active configuration set exists and the content of the
     * configuration set has been updated from the last update.
     *
     * @endif
     */
    void update(void);

    /*!
     * @if jp
     *
     * @brief ե졼ѥ᡼ι(ID)
     * 
     * ե졼ѿͤ򡢻ꤷIDĥե졼
     * 󥻥åȤͤǹ롣ˤꡢƥ֤ʥե
     * 졼󥻥åȤѹʤäơƥ֥ե
     * 졼󥻥åȤȥѥ᡼ѿδ̷֤⤬ȯǽ
     * ΤդɬפǤ롣
     *
     * ꤷIDΥե졼󥻥åȤ¸ߤʤϡ
     * ˽λ롣
     *
     * @param config_set оݤΥե졼󥻥åID
     * 
     * @else
     *
     * @brief Update configuration parameter (By ID)
     * 
     * This operation updates configuration variables by the
     * configuration-set with specified ID. This operation does not
     * change current active configuration-set. Since this operation
     * causes inconsistency between current active configuration set
     * and actual values of configuration variables, user should
     * carefully use it.
     *
     * This operation ends without doing anything, if the
     * configuration-set does not exist.
     *
     * @param config_set The target configuration set's ID to setup
     *
     * @endif
     */
    void update(const char* config_set);
    
    /*!
     * @if jp
     *
     * @brief ե졼ѥ᡼ι(̾λ)
     * 
     * Υե졼ѿͤ򡢻ꤷIDĥե
     * 졼󥻥åȤͤǹ롣ˤꡢƥ֤ʥ
     * ե졼󥻥åȤѹʤäơƥ֥
     * ե졼󥻥åȤȥѥ᡼ѿδ̷֤⤬ȯ
     * ǽΤդɬפǤ롣
     *
     * ꤷIDΥե졼󥻥åȤ䡢ꤷ̾ΤΥѥ᡼
     * ¸ߤʤϡ⤻˽λ롣
     *
     * @param config_set ե졼ID
     * @param config_param ե졼ѥ᡼̾
     * 
     * @else
     *
     * @brief Update the values of configuration parameters (By name)
     * 
     * This operation updates a configuration variable by the
     * specified configuration parameter in the
     * configuration-set. This operation does not change current
     * active configuration-set. Since this operation causes
     * inconsistency between current active configuration set and
     * actual values of configuration variables, user should carefully
     * use it.
     *
     * This operation ends without doing anything, if the
     * configuration-set or the configuration parameter do not exist.
     *
     * @param config_set configuration-set ID.
     * @param config_param configuration parameter name.
     *
     * @endif
     */
    void update(const char* config_set, const char* config_param);
    
    /*!
     * @if jp
     *
     * @brief ե졼ѥ᡼¸߳ǧ
     * 
     * ꤷ̾Τĥե졼ѥ᡼ѿ¸ߤ
     * ǧ롣¸߳ǧԤѥ᡼ѿȤϡ
     * bindParameter() ˤäϿ롢ѿĥѥ᡼Ǥ롣
     *
     * @param name ե졼ѥ᡼̾Ρ
     *
     * @return ¸߳ǧ(ѥ᡼:trueѥ᡼ʤ:false)
     *
     * @else
     *
     * @brief Check the existence of configuration parameters
     * 
     * Check the existence of configuration parameters of specified name.
     *
     * @param name Configuration parameter name
     *
     * @return Result of existance confirmation 
     *         (Parameters exist:true, else:false)
     *
     * @endif
     */
    bool isExist(const char* name);
    
    /*!
     * @if jp
     *
     * @brief ե졼ѥ᡼ѹǧ
     * 
     * ե졼ѥ᡼ѹ줿ǧ롣
     *
     * @return ѹǧ(ѹ:trueѹʤ:false)
     *
     * @else
     *
     * @brief Confirm to change configuration parameters
     * 
     * Confirm that configuration parameters have changed.
     *
     * @return Result of change confirmation
     *         (There is a change:trueNo change:false)
     *
     * @endif
     */
    bool isChanged(void) {return m_changed;}
    
    /*!
     * @if jp
     *
     * @brief ƥ֡ե졼󥻥åIDμ
     * 
     * ߥƥ֤ʥե졼󥻥åȤID롣
     *
     * @return ƥ֡ե졼󥻥åID
     *
     * @else
     *
     * @brief Get ID of active configuration set
     * 
     * Get ID of the current active configuration set.
     *
     * @return The active configuration set ID
     *
     * @endif
     */
    const char* getActiveId(void) {return m_activeId.c_str();}
    
    /*!
     * @if jp
     *
     * @brief ե졼󥻥åȤ¸߳ǧ
     * 
     * ꤷե졼󥻥åȤ¸ߤ뤫ǧ롣
     *
     * @param config_id ǧоݥե졼󥻥åID
     *
     * @return ¸߳ǧ(ꤷConfigSet:trueʤ:false)
     *
     * @else
     *
     * @brief Check the existence of configuration set
     * 
     * Check the existence of specified configuration set.
     *
     * @param config_id ID of target configuration set for confirmation
     *
     * @return Result of existence confirmation 
     *         (Specified ConfigSet exists:true, else:false)
     *
     * @endif
     */
    bool haveConfig(const char* config_id)
    {
      return (m_configsets.hasKey(config_id) == NULL) ? false : true;
    }
    
    /*!
     * @if jp
     *
     * @brief ե졼󥻥åȤΥƥֲǧ
     * 
     * ե졼󥻥åȤƥֲƤ뤫ǧ롣
     *
     * @return ֳǧ(ƥ־:true󥢥ƥ־:false)
     *
     * @else
     *
     * @brief Confirm to activate configuration set
     * 
     * Confirm that configuration set has been activated.
     *
     * @return Result of state confirmation
     *         (Active state:true, Inactive state:false)
     *
     * @endif
     */
    bool isActive(void)
    {
      return m_active;
    }
    //    const std::vector<Properties*>* getConfigurationParameterValues();
    //    const Properties* getConfigurationParameterValue(const char* name);
    //    bool setConfigurationParameter(const char* name, const char* value);
    
    /*!
     * @if jp
     *
     * @brief ե졼󥻥åȤμ
     * 
     * ꤵƤե졼󥻥åȤ롣
     *
     * @return ե졼󥻥å
     *
     * @else
     *
     * @brief Get all configuration sets
     * 
     * Get all specified configuration sets
     *
     * @return All configuration sets
     *
     * @endif
     */
    const std::vector<coil::Properties*>& getConfigurationSets(void);
    
    /*!
     * @if jp
     *
     * @brief ꤷIDΥե졼󥻥åȤμ
     * 
     * IDǻꤷե졼󥻥åȤ롣
     * ꤷե졼󥻥åȤ¸ߤʤϡ
     * Υե졼󥻥åȤ֤
     *
     * @param config_id оݥե졼󥻥åȤID
     *
     * @return ե졼󥻥å
     *
     * @else
     *
     * @brief Get a configuration set by specified ID
     * 
     * Get a configuration set that was specified by ID
     * Return empty configuration set, if a configuration set of
     * specified ID doesn't exist.
     *
     * @param config_id ID of the target configuration set for getting
     *
     * @return The configuration set
     *
     * @endif
     */
    const coil::Properties& getConfigurationSet(const char* config_id);
    
    /*!
     * @if jp
     *
     * @brief ꤷץѥƥΥե졼󥻥åȤؤɲ
     * 
     * ꤷץѥƥIDǻꤷե졼󥻥åȤɲä롣
     * ꤷIDȰפ륳ե졼󥻥åȤ¸ߤʤϡ
     * false ֤
     *
     * @param config_id ɲоݥե졼󥻥åȤID
     * @param configuration_set ɲäץѥƥ
     *
     * @return ɲý¹Է(ɲ:trueɲü:false)
     *
     * @else
     *
     * @brief Add to configuration set from specified property
     * 
     * Add specified property to configuration set that was specified by ID.
     * Return false if configuration set, that matches specified ID, 
     * doesn't exist.
     *
     * @param config_id ID of the target configuration set for add
     * @param configuration_set Property to add
     *
     * @return Add result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool setConfigurationSetValues(const coil::Properties& configuration_set);
    
    /*!
     * @if jp
     *
     * @brief ƥ֡ե졼󥻥åȤ
     * 
     * ߥƥ֤ȤʤäƤ륳ե졼󥻥åȤ롣
     * ƥ֤ȤʤäƤ륳ե졼󥻥åȤ¸ߤʤϡ
     * Υե졼󥻥å ֤
     *
     * @return ƥ֡ե졼󥻥å
     *
     * @else
     *
     * @brief Get the active configuration set
     * 
     * Get the current active configuration set.
     * Return empty configuration set, if an active configuration set 
     * doesn't exist.
     *
     * @return The active configuration set
     *
     * @endif
     */
    const coil::Properties& getActiveConfigurationSet(void);
    
    /*!
     * @if jp
     *
     * @brief ե졼󥻥åȤͤɲ
     * 
     * ե졼󥻥åȤͤɲä롣
     *
     * @param configuration_set ɲäץѥƥ
     *
     * @return ɲý(ɲ:trueɲü:false)
     *
     * @else
     *
     * @brief Add the configuration value to configuration set
     * 
     * Add the configuration value to configuration set
     *
     * @param configuration_set Property to add
     *
     * @return Add Result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool addConfigurationSet(const coil::Properties& configuration_set);
    
    /*!
     * @if jp
     *
     * @brief ե졼󥻥åȤκ
     * 
     * ꤷIDΥե졼󥻥åȤ롣
     *
     * ꤷIDΥե졼󥻥åȤ¸ߤʤϡ
     * false֤ǽʥե졼󥻥åȤϡ
     * addConfigruationSet() ˤäɲäե졼󥻥
     * ȤΤߤǤꡢǥեȥե졼󥻥åȡݡ
     * ȵư˥ե뤫ɤ߹ޤ륳ե졼󥻥å
     * Ϻ뤳ȤǤʤ
     *
     * ޤꤷե졼󥻥åȤߥƥ֤Ǥ
     * ˤϡʤ륳ե졼󥻥åȤǤǤʤ
     *
     * δؿˤºݤ˥ե졼󥻥åȤ줿硢
     * setOnRemoveConfigurationSet() ǥåȤ줿Хåؿ
     * ӽФ롣
     *
     * @param config_id оݥե졼󥻥åȤID
     *
     * @return (:true:false)
     *
     * @else
     *
     * @brief Remove the configuration set
     * 
     * Remove the configuration set of specified ID Return empty
     * configuration set, if a configuration set of specified ID
     * doesn't exist.
     *
     * The configuration-sets that can be removed by this function are
     * only configuration-sets newly added by the
     * addConfigurationSet() function. The configuration that can be
     * removed by this function is only newly added configuration-set
     * by addConfigurationSet() function.  The "default"
     * configuration-set and configurationi-sets that is loaded from
     * configuration file cannot be removed.
     *
     * If the specified configuration is active currently, any
     * configurations are not deleted.
     *
     * Callback functions that are set by
     * addOnRemovedConfigurationSet() will be called if a
     * configuration-set is deleted actually by this function.
     *
     * @param config_id ID of the target configuration set for remove
     *
     * @return Remove result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool removeConfigurationSet(const char* config_id);
    
    /*!
     * @if jp
     *
     * @brief ե졼󥻥åȤΥƥֲ
     * 
     * ꤷIDΥե졼󥻥åȤ򥢥ƥֲ롣
     * ꤷIDΥե졼󥻥åȤ¸ߤʤϡ
     * false֤
     *
     * @param config_id оݥե졼󥻥åȤID
     *
     * @return ƥֽ(:true:false)
     *
     * @else
     *
     * @brief Activate the configuration set
     * 
     * Activate the configuration set of specified ID
     * Return empty configuration set, if a configuration set of
     * specified ID doesn't exist.
     *
     * @param config_id ID of the target configuration set for remove
     *
     * @return Activate result (Remove success:trueRemove failure:false)
     *
     * @endif
     */
    bool activateConfigurationSet(const char* config_id);

    /*!
     * @if jp
     *
     * @brief OnUpdate ΥХå
     *
     * OnUpdate ǸƤФ륳ХåΥ֥Ȥꤹ롣
     * 
     * @param cb OnUpdateCallbackΥ֥
     *
     * @else
     *
     * @brief Set callback that is called by OnUpdate. 
     * 
     * @param cb OnUpdateCallback type object
     *
     * @endif
     */
    void setOnUpdate(OnUpdateCallback* cb);

    /*!
     * @if jp
     *
     * @brief OnUpdateParam ΥХå
     *
     * OnUpdateParam ǸƤФ륳ХåΥ֥Ȥꤹ롣
     * 
     * @param cb OnUpdateParamCallbackΥ֥
     *
     * @else
     *
     * @brief Set callback that is called by OnUpdateParam. 
     * 
     * @param cb OnUpdateParamCallback type object
     *
     * @endif
     */
    void setOnUpdateParam(OnUpdateParamCallback* cb);

    /*!
     * @if jp
     *
     * @brief OnSetConfigurationSet ΥХå
     *
     * OnSetConfigurationSet ǸƤФ륳ХåΥ֥Ȥꤹ롣
     * 
     * @param cb OnSetConfigurationSetCallbackΥ֥
     *
     * @else
     *
     * @brief Set callback that is called by OnSetConfiguration. 
     * 
     * @param cb OnSetConfigurationSetCallback type object
     *
     * @endif
     */
    void setOnSetConfigurationSet(OnSetConfigurationSetCallback* cb);

    /*!
     * @if jp
     *
     * @brief OnAddConfigurationSet ΥХå
     *
     * OnAddConfigurationSet ǸƤФ륳ХåΥ֥Ȥꤹ롣
     * 
     * @param cb OnAddConfigurationAddCallbackΥ֥
     *
     * @else
     *
     * @brief Set callback that is called by OnSetConfiguration. 
     * 
     * @param cb OnSetConfigurationSetCallback type object
     *
     * @endif
     */
    void setOnAddConfigurationSet(OnAddConfigurationAddCallback* cb);

    /*!
     * @if jp
     *
     * @brief OnRemoveConfigurationSet ΥХå
     *
     * OnRemoveConfiguration ǸƤФ륳ХåΥ֥Ȥꤹ롣
     * 
     * @param cb OnRemoveConfigurationSetCallbackΥ֥
     *
     * @else
     *
     * @brief Set callback that is called by OnRemoveConfigurationSet. 
     * 
     * @param cb OnRemoveConfigurationSetCallback type object
     *
     * @endif
     */
    void setOnRemoveConfigurationSet(OnRemoveConfigurationSetCallback* cb);

    /*!
     * @if jp
     *
     * @brief OnActivateSet ΥХå
     *
     * OnActivateSet ǸƤФ륳ХåΥ֥Ȥꤹ롣
     * 
     * @param cb OnActivateSetCallbackΥ֥
     *
     * @else
     *
     * @brief Set callback that is called by OnActivateSet. 
     * 
     * @param cb OnActivateSetCallback type object
     *
     * @endif
     */
    void setOnActivateSet(OnActivateSetCallback* cb);

  protected:
    /*!
     * @if jp
     *
     * @brief ե졼ѥ᡼ι(ID)˥뤵
     *
     * ꤵƤ륳Хå֥ȤƤӽФ
     *
     * @param config_set оݤΥե졼󥻥åID
     *
     * @else
     *
     * @brief When the configuration parameter is updated, it is called. 
     *
     * Call the set callback object.
     * 
     * @param config_set The target configuration set's ID to setup
     *
     * @endif
     */
    void onUpdate(const char* config_set);

    /*!
     * @if jp
     *
     * @brief ե졼ѥ᡼ι(̾λ)˥뤵
     *
     * ꤵƤ륳Хå֥ȤƤӽФ
     *
     * @param config_set ե졼ID
     * @param config_param ե졼ѥ᡼̾
     *
     * @else
     *
     * @brief When the configuration parameter is updated, it is called. 
     *
     * Call the set callback object.
     * 
     * @param config_set configuration-set ID.
     * @param config_param configuration parameter name.
     *
     * @endif
     */
    void onUpdateParam(const char* config_set, const char* config_param);

    /*!
     * @if jp
     *
     * @brief ꤷץѥƥΥե졼󥻥åȤؤɲä줿˥뤵
     *
     * ꤵƤ륳Хå֥ȤƤӽФ
     *
     * @param configuration_set ץѥƥ
     *
     * @else
     *
     * @brief When the property is added to the configuration set, this function is called. 
     *
     * Call the set callback object.
     * 
     * @param configuration_set property
     *
     * @endif
     */
    void onSetConfigurationSet(const coil::Properties& config_set);

    /*!
     * @if jp
     *
     * @brief ե졼󥻥åȤͤɲä줿Ȥ˥뤵롣
     *
     * ꤵƤ륳Хå֥ȤƤӽФ
     *
     * @param configuration_set ץѥƥ
     *
     * @else
     *
     * @brief When a set value is added to the configuration set, this function is called.
     *
     * Call the set callback object.
     * 
     * @param configuration_set property
     *
     * @endif
     */
    void onAddConfigurationSet(const coil::Properties& config_set);

    /*!
     * @if jp
     *
     * @brief ե졼󥻥åȤƤȤ˥뤵롣
     *
     * ꤵƤ륳Хå֥ȤƤӽФ
     *
     * @param config_id ץѥƥ
     *
     * @else
     *
     * @brief When the configuration set has been deleted, this function is called. 
     *
     * Call the set callback object.
     * 
     * @param config_id property
     *
     * @endif
     */
    void onRemoveConfigurationSet(const char* config_id);

    /*!
     * @if jp
     *
     * @brief ե졼󥻥åȤƥֲ줿Ȥ˥뤵롣
     *
     * ꤵƤ륳Хå֥ȤƤӽФ
     *
     * @param config_id ץѥƥ
     *
     * @else
     *
     * @brief When the configuration set is made active, this function is called.
     *
     * Call the set callback object.
     * 
     * @param config_id property
     *
     * @endif
     */
    void onActivateSet(const char* config_id);
    
  private:
    ConfigAdmin(const ConfigAdmin& ca);// : m_configsets(ca.m_configsets) {};
    ConfigAdmin& operator=(const ConfigAdmin& ca); //{return *this;};
    
    struct find_conf
    {
      std::string m_name;
      find_conf(const char* name) : m_name(name) {};
      bool operator()(ConfigBase* conf)
      {
        if (conf == 0) { return false; }
	return (m_name == conf->name);
      }
    };
    
    coil::Properties& m_configsets;
    coil::Properties  m_emptyconf;
    std::vector<ConfigBase*> m_params;
    std::string m_activeId;
    bool m_active;
    bool m_changed;
    std::vector<std::string> m_newConfig;

    OnUpdateCallback*                 m_updateCb;
    OnUpdateParamCallback*            m_updateParamCb;
    OnSetConfigurationSetCallback*    m_setConfigSetCb;
    OnAddConfigurationAddCallback*    m_addConfigSetCb;
    OnRemoveConfigurationSetCallback* m_removeConfigSetCb;
    OnActivateSetCallback*            m_activateSetCb;


  };
}; // namespace RTC
#endif // RTC_CONFIGADMIN_H
