// -*- C++ -*-
/*!
 * @file   ModuleManagerTests.cpp
 * @brief  ModuleManager test class
 * @date   $Date: 2008/05/01 03:42:41 $
 * @author Shinji Kurihara
 *         Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: ModuleManagerTests.cpp 1758 2010-01-22 14:04:54Z hakuta $
 *
 */

/*
 * $Log: ModuleManagerTests.cpp,v $
 * Revision 1.2  2008/05/01 03:42:41  arafune
 * Modified some tests.
 *
 * Revision 1.1  2007/12/20 07:50:17  arafune
 * *** empty log message ***
 *
 * Revision 1.1  2006/11/27 08:33:02  n-ando
 * TestSuites are devided into each directory.
 *
 * Revision 1.2  2006/10/26 01:34:57  kurihara
 * test program of class ModuelManager.
 *
 * Revision 1.1  2006/09/20 08:52:19  n-ando
 * The first commit.
 *
 */

#ifndef ModuleManager_cpp
#define ModuleManager_cpp

#include <cppunit/ui/text/TestRunner.h>
#include <cppunit/TextOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/extensions/HelperMacros.h>
#include <cppunit/TestAssert.h>
#include <fstream>
#include <rtm/ModuleManager.h>
#include <coil/Properties.h>

/*!
 * @class ModuleManagerTests class
 * @brief ModuleManager test
 */
namespace ModuleManager
{
  class ModuleManagerTests
    : public CppUnit::TestFixture
  {
    CPPUNIT_TEST_SUITE(ModuleManagerTests);
    CPPUNIT_TEST(test_load);
    CPPUNIT_TEST(test_load_inexistent_on_load_path);
    CPPUNIT_TEST(test_unload);
    CPPUNIT_TEST(test_unloadAll);
    CPPUNIT_TEST(test_symbol);
    CPPUNIT_TEST(test_setLoadpath_and_getLoadPath);
    CPPUNIT_TEST(test_addLoadpath);
    //CPPUNIT_TEST(test_getLoadedModules);
    //CPPUNIT_TEST(test_getLoadableModules);
    CPPUNIT_TEST(test_allowAbsolutePath_and_disallowAbsolutePath);
    //CPPUNIT_TEST(test_allowModuleDownload);
    //CPPUNIT_TEST(test_disallowModuleDownload);
    CPPUNIT_TEST(test_findFile);
    CPPUNIT_TEST(test_fileExist);
    CPPUNIT_TEST(test_getInitFuncName);
    CPPUNIT_TEST_SUITE_END();
		
  private:
    RTC::ModuleManager* m_pModMgr;
		
  private:
    bool isFound(const std::vector<coil::Properties>& path, const std::string& mod)
    {
      for (int i(0), len(path.size()); i < len; ++i)
        {
          if (mod == path[i]["file_path"]) return true;
        }
      return false;
    }
	
  public:
    /*!
     * @brief Constructor
     */
    ModuleManagerTests()
    {
    }
		
    /*!
     * @brief Destructor
     */
    ~ModuleManagerTests()
    {
    }
		
    /*!
     * @brief Test initialization
     */
    virtual void setUp()
    {
      const char* default_properties[] =
	{
	  "manager.modules.config_ext", "so",
	  "manager.modules.config_path", "/etc/rtc",
	  "manager.modules.detect_loadable", "Yes",
	  "manager.modules.load_path",
	  "/usr/lib, /usr/local/lib, /usr/local/lib/rtc, ../../.libs",
	  "manager.modules.init_func_suffix", "Init",
	  "manager.modules.init_func_prefix", "",
	  "manager.modules.abs_path_allowed", "Yes",
	  "manager.modules.download_allowed", "Yes",
	  "manager.modules.download_dir", "/tmp/rtc",
	  "manager.modules.download_cleanup", "Yes",
	  "manager.modules.preload", "",
	  ""
	};
			
      coil::Properties prop(default_properties);
      m_pModMgr = new RTC::ModuleManager(prop);
			
      // ɥѥlibRTC.so¸ߤ뤳ȤǧƤ
      m_pModMgr->load("libRTC.so");
			
      // ɥѥlibm.so¸ߤ뤳ȤǧƤ
      m_pModMgr->load("libm.so");
    }
		
    /*!
     * @brief Test finalization
     */
    virtual void tearDown()
    {
      m_pModMgr->unloadAll();
      delete m_pModMgr;
    }
		
    /* tests for string load(const string& file_name) */
    /*!
     * @brief load()᥽åɤΥƥ
     * 
     * - ե̾ꤷˡ˥ɤǤ뤫
     * - ե̾Хѥǻꤷˡ˥ɤǤ뤫
     * - ѥˡ//פ../פޤޤˡ˥ɤǤ뤫
     * - ɥѥ˥ե¸ߤ뤬ĥҤŬѳǤˡտޤɤɼԤ뤫
     * - ¸ߤʤեꤷˡտޤɤ˥ɼԤ뤫
     */
    void test_load()
    {
      try
	{
	  // Success case
	  // ե̾ꤷˡɤǤ뤫
	  //std::string str = m_pModMgr->load("libRTC.so");
	  //std::cout << "\nstr=(" << str << std::endl;
	  // 󥹥ȡĶˤäơեѥѲΤǥȤˤ
//	  CPPUNIT_ASSERT_EQUAL(
//			       std::string("/usr/local/lib/libRTC.so"),
//			       std::string("../../.libs/libRTC.so"),
//			       std::string("/usr/lib/libRTC.so"),
//			       m_pModMgr->load("libRTC.so"));
				
/*
	  // ե̾Хѥǻꤷˡ˥ɤǤ뤫
	  CPPUNIT_ASSERT_EQUAL(
//			       std::string("/usr/local/lib/libRTC.so"),
			       std::string("../../.libs/libRTC.so"),
			       m_pModMgr->load("/usr/local/lib/libRTC.so"));
				
	  // ǥ쥯ȥζڤʸ"//""../"
	  CPPUNIT_ASSERT_EQUAL(
//			       std::string("/usr/local//lib/../lib/libRTC.so"),
//			       m_pModMgr->load("/usr/local//lib/../lib/libRTC.so"));
			       std::string("../..//.libs/../libs/libRTC.so"),
			       m_pModMgr->load("../..//.libs/../libs/libRTC.so"));
*/
	}
      catch (RTC::ModuleManager::Error& e)
	{
	  CPPUNIT_FAIL("Error" + e.reason);
	}
      catch (RTC::ModuleManager::NotFound& e)
	{
	  CPPUNIT_FAIL("NotFound" + e.name);
	}
      catch (RTC::ModuleManager::NotAllowedOperation& e)
	{
	  CPPUNIT_FAIL("NotAllowedOperation");
	}
      catch (RTC::ModuleManager::InvalidArguments& e)
	{
	  CPPUNIT_FAIL("InvalidArguments");
	}
      catch (RTC::ModuleManager::FileNotFound& e)
	{
	  CPPUNIT_FAIL("FileNotFound" + e.name);
	}
      catch (...)
	{
	  CPPUNIT_FAIL("Other Exception");
	}
			
      // Failure case
      bool notThrown;

      // ɥѥ˥ե¸ߤ뤬ĥҤŬѳǤˡտޤɤɼԤ뤫
      notThrown = false;
      try
	{
	  m_pModMgr->load("libm.a");
	  notThrown = true;
	}
      catch (...) {}
      if (notThrown) CPPUNIT_FAIL("Exception not thrown.");

      // ¸ߤʤեξ
      notThrown = false;
      try
	{
	  m_pModMgr->load("inexist-file");
	  notThrown = true;
	}
      catch (...) {}
      if (notThrown) CPPUNIT_FAIL("Exception not thrown.");
    }
		
    /*!
     * @brief load()᥽åɤΥƥ
     * 
     * - ɥѥ¸ߤʤ⥸塼Υɤߤ硢տޤɤɼԤ뤫
     */
    void test_load_inexistent_on_load_path()
    {

      std::vector<std::string> loadPath = m_pModMgr->getLoadPath();
			
      // setLoadPath()Ѥơɥѥ򥯥ꥢ
      std::vector<std::string> emptyPath;
      m_pModMgr->setLoadpath(emptyPath);
			
      // ɥѥ¸ߤʤ⥸塼Υɤߤ硢տޤɤɼԤ뤫
      // ʥɥѥ򥯥ꥢ֤ʤΤǡ⥸塼ɤ˼ԤϤ
      bool notThrown = false;
      try
	{
	  m_pModMgr->load("libm.so");
	  notThrown = true;
	}
      catch (...) {}
      if (notThrown) CPPUNIT_FAIL("Exception not thrown.");
			
      // Υɥѥꤷʤ
      m_pModMgr->setLoadpath(loadPath);
			
      // ⥸塼ΥɤϤ
      std::string modName = m_pModMgr->load("libm.so");
      CPPUNIT_ASSERT(isFound(m_pModMgr->getLoadedModules(), modName));
    }
		
    /*!
     * @brief unload()᥽åɤΥƥ
     * 
     * - ɤƤ⥸塼ɤǤ뤫
     * - ɤƤʤ⥸塼ϡʤɤ줺˻ĤäƤ뤫
     * - Хѥꤻե̾ꤷˡտޤɤ˥ɼԤ뤫
     * - ɤƤʤ⥸塼Υɤߤ硢տޤɤ˼Ԥ뤫
     * - ɺѤߤΥ⥸塼򡢤˥ɤ褦Ȼߤ硢տޤɤ˼Ԥ뤫
     */
    void test_unload()
    {

      // ⥸塼ɤƤ
      std::string modName1 = m_pModMgr->load("libRTC.so");
      CPPUNIT_ASSERT(isFound(m_pModMgr->getLoadedModules(), modName1));
			
      std::string modName2 = m_pModMgr->load("libm.so");
      CPPUNIT_ASSERT(isFound(m_pModMgr->getLoadedModules(), modName2));
			
      // Success case
      // ɤƤ⥸塼ɤǤ뤫
      m_pModMgr->unload(modName1);
      CPPUNIT_ASSERT(! isFound(m_pModMgr->getLoadedModules(), modName1));
			
      // ɤƤʤ⥸塼ϡʤɤ줺˻ĤäƤ뤫
      CPPUNIT_ASSERT(isFound(m_pModMgr->getLoadedModules(), modName2));
			
			
      // Failure case
      // Хѥꤻե̾ꤷˡտޤɤ˥ɼԤ뤫
      try
	{
	  m_pModMgr->unload("libm.so");
	  CPPUNIT_FAIL("Exception not thrown.");
	}
      catch (RTC::ModuleManager::NotFound expected) {}
			
      // ɤƤʤ⥸塼Υɤߤ硢տޤɤ˼Ԥ뤫
      try
	{
	  m_pModMgr->unload("non-loaded-module.so");
	  CPPUNIT_FAIL("Exception not thrown.");
	}
      catch (RTC::ModuleManager::NotFound expected) {}
			
      // ɺѤߤΥ⥸塼򡢤˥ɤ褦Ȼߤ硢տޤɤ˼Ԥ뤫
      try
	{
	  m_pModMgr->unload(modName1);
	  CPPUNIT_FAIL("Exception not thrown.");
	}
      catch (RTC::ModuleManager::NotFound expected) {}
    }
		
    /*!
     * @brief unloadAll()᥽åɤΥƥ
     * 
     * - ɺѤߤΥ⥸塼뤬٤ƥɤ뤫
     */
    void test_unloadAll()
    {
      // ⥸塼ɤƤ
      std::string modName1 = m_pModMgr->load("libRTC.so");
      CPPUNIT_ASSERT(isFound(m_pModMgr->getLoadedModules(), modName1));
			
      std::string modName2 = m_pModMgr->load("libm.so");
      CPPUNIT_ASSERT(isFound(m_pModMgr->getLoadedModules(), modName2));
			
      // unloadAll()ˤäơɺѤߤΥ⥸塼뤬٤ƥɤ뤫
      m_pModMgr->unloadAll();
      CPPUNIT_ASSERT(! isFound(m_pModMgr->getLoadedModules(), modName1));
      CPPUNIT_ASSERT(! isFound(m_pModMgr->getLoadedModules(), modName2));
    }
		
    /*!
     * @brief symbol()᥽åɤΥƥ
     * 
     * - ⥸塼뤬ĥܥʴؿݥ󥿡ˤ˼Ǥ뤫
     * - ܥФƽФ˹Ԥ뤫
     */
    void test_symbol()
    {
      typedef double (*FUNCTION_COS)(double);
      typedef double (*FUNCTION_SIN)(double);

      // ⥸塼ɤƤ
      std::string modName = m_pModMgr->load("libm.so");
      CPPUNIT_ASSERT(isFound(m_pModMgr->getLoadedModules(), modName));
			
      // "cos"ؿؤΥܥ
      FUNCTION_COS func_cos = (FUNCTION_COS) m_pModMgr->symbol(modName, "cos");
      CPPUNIT_ASSERT(func_cos != NULL);
			
      // "sin"ؿؤΥܥ
      FUNCTION_SIN func_sin = (FUNCTION_SIN) m_pModMgr->symbol(modName, "sin");
      CPPUNIT_ASSERT(func_sin != NULL);
			
      // ܥؤθƽФ˹Ԥ뤫
      CPPUNIT_ASSERT_EQUAL(1.0, (*func_cos)(0.0));
      CPPUNIT_ASSERT_EQUAL(0.0, (*func_sin)(0.0));
    }
		
    /*!
     * @brief setLoadpath()getLoadPath()Υƥ
     * 
     * - ɥѥǤ뤫
     * - ɥѥǤ뤫
     */
    void test_setLoadpath_and_getLoadPath()
    {

      std::vector<std::string> loadPath;
      loadPath.push_back("/usr");
      loadPath.push_back("/usr/lib");
			
      // setLoadpath()ꤷѥgetLoadPath()ǼǤ뤫
      m_pModMgr->setLoadpath(loadPath);
      std::vector<std::string> loadPathRet = m_pModMgr->getLoadPath();
      CPPUNIT_ASSERT_EQUAL(2, (int) loadPathRet.size());
      CPPUNIT_ASSERT_EQUAL(std::string("/usr"), loadPathRet[0]);
      CPPUNIT_ASSERT_EQUAL(std::string("/usr/lib"), loadPathRet[1]);

    }
		
    /*!
     * @brief addLoadpath()᥽åɤΥƥ
     * 
     * - ɥѥɲäǤ뤫
     */
    void test_addLoadpath()
    {
      std::vector<std::string> loadPath1;
      loadPath1.push_back("/usr");
      loadPath1.push_back("/usr/lib");
      std::sort(loadPath1.begin(), loadPath1.end());

      std::vector<std::string> loadPath2;
      loadPath2.push_back("/usr/local/lib");
      loadPath2.push_back("/usr/local/lib/rtc");
      std::sort(loadPath2.begin(), loadPath2.end());
			
      std::vector<std::string> expectedPath;
      expectedPath.push_back("/usr");
      expectedPath.push_back("/usr/lib");
      expectedPath.push_back("/usr/local/lib");
      expectedPath.push_back("/usr/local/lib/rtc");
      std::sort(expectedPath.begin(), expectedPath.end());
			
      // ޤ֤ΥɥѥꤷƤ
      std::vector<std::string> loadPathRet;
      m_pModMgr->setLoadpath(loadPath1);
      loadPathRet = m_pModMgr->getLoadPath();
      CPPUNIT_ASSERT_EQUAL(2, (int) loadPathRet.size());
      for (int i = 0; i < 2; ++i) {
	CPPUNIT_ASSERT_EQUAL(loadPath1[i], loadPathRet[i]);
      }
			
      // ɥѥɲäǤ뤫
      m_pModMgr->addLoadpath(loadPath2);
      loadPathRet = m_pModMgr->getLoadPath();
      CPPUNIT_ASSERT_EQUAL(4, (int) loadPathRet.size());
      for (int i = 0; i < 4; ++i) {
	CPPUNIT_ASSERT_EQUAL(expectedPath[i], loadPathRet[i]);
      }
    }
		
    /*!
     * @brief getLoadedModules()᥽åɤΥƥ
     */
    void test_getLoadedModules()
    {
      // ¾ƥǻѤƤꡢǷͤΤȤ
    }
		
    /*!
     * @brief getLoadableModules()᥽åɤΥƥ
     */
    void test_getLoadableModules()
    {
      // ᥽åɤ̤ˤĤƥȤ̤
    }
		
    /*!
     * @brief allowAbsolutePath()᥽åɤdisallowAbsolutePath()᥽åɤΥƥ
     * 
     * - ХѥĤ֤ǡХѥǥ⥸塼ɤǤ뤫
     * - Хѥػߤ֤ǡХѥǥ⥸塼ɤߤơտޤɤ꼺Ԥ뤫
     */
    void test_allowAbsolutePath_and_disallowAbsolutePath()
    {
      // ХѥĤ֤ǡХѥǥ⥸塼ɤǤ뤫
      m_pModMgr->allowAbsolutePath();
      std::string modName = m_pModMgr->load("/usr/lib/libm.so");
      CPPUNIT_ASSERT(isFound(m_pModMgr->getLoadedModules(), modName));
			
      // Хѥػߤ֤ǡХѥǥ⥸塼ɤߤơտޤɤ꼺Ԥ뤫
      m_pModMgr->unloadAll(); // ä󥢥ɤƤ
      m_pModMgr->disallowAbsolutePath();
      try
	{
	  m_pModMgr->load("/usr/lib/libm.so");
	  CPPUNIT_FAIL("Exception not thrown.");
	}
      catch (RTC::ModuleManager::NotAllowedOperation expected) {}
    }
		
    /*!
     * @brief allowModuleDownload()᥽åɤΥƥ
     */
    void test_allowModuleDownload()
    {
      // ɤˤ⥸塼ɽ̤ΤᡢƥȤ̤
    }
		
    /*!
     * @brief disallowModuleDownload()᥽åɤΥƥ
     */
    void test_disallowModuleDownload()
    {
      // ɤˤ⥸塼ɽ̤ΤᡢƥȤ̤
    }
		
    /*!
     * @brief findFile()᥽åɤΥƥ
     * 
     * - ꤷѥ˥ե뤬¸ߤ硢̵ͭȽꤷѥǤ뤫
     * - ꤷѥ˥ե뤬¸ߤʤ硢̵ͭȽꤷʸǤ뤫
     */
    void test_findFile()
    {

      std::vector<std::string> path;
      path.push_back("/usr/lib");
			
      // ꤷѥ˥ե뤬¸ߤ硢̵ͭȽꤷѥǤ뤫
      CPPUNIT_ASSERT_EQUAL(std::string("/usr/lib/libm.so"),
			   m_pModMgr->findFile("libm.so", path));
			
      // ꤷѥ˥ե뤬¸ߤʤ硢̵ͭȽꤷʸǤ뤫
      CPPUNIT_ASSERT_EQUAL(std::string(""),
			   m_pModMgr->findFile("inexist-file.so", path));
    }
		
    /*!
     * @brief fileExist()᥽åɤΥƥ
     * 
     * - ¸ߤեꤷˡȽꤵ뤫
     * - ¸ߤʤեꤷˡȽꤵ뤫
     * - ¸ߤե//ס../פäѥɽǻꤷ硢Ƚꤵ뤫
     */
    void test_fileExist()
    {
      // ¸ߤեꤷˡȽꤵ뤫
      CPPUNIT_ASSERT(m_pModMgr->fileExist("/usr/lib/libm.so"));

      // ¸ߤʤեꤷˡȽꤵ뤫
      CPPUNIT_ASSERT(! m_pModMgr->fileExist("/usr/lib/inexistent-file.so"));
			
      // ¸ߤե//ס../פäѥɽǻꤷ硢Ƚꤵ뤫
      CPPUNIT_ASSERT(m_pModMgr->fileExist("/usr//lib/../lib/libm.so"));
    }
		
    /*!
     * @brief getInitFuncName()᥽åɤΥƥ
     * 
     * - ؿ̾Ǥ뤫
     */
    void test_getInitFuncName()
    {
      // ؿ̾Ǥ뤫
      CPPUNIT_ASSERT_EQUAL(std::string("ManipulatorInit"),
			   m_pModMgr->getInitFuncName("Manipulator"));
			
      CPPUNIT_ASSERT_EQUAL(std::string("PHANToMInit"),
			   m_pModMgr->getInitFuncName("PHANToM"));
    }
		
  };
}; // namespace ModuleManager

/*
 * Register test suite
 */
CPPUNIT_TEST_SUITE_REGISTRATION(ModuleManager::ModuleManagerTests);

#ifdef LOCAL_MAIN
int main(int argc, char* argv[])
{

  FORMAT format = TEXT_OUT;
  int target = 0;
  std::string xsl;
  std::string ns;
  std::string fname;
  std::ofstream ofs;

  int i(1);
  while (i < argc)
    {
      std::string arg(argv[i]);
      std::string next_arg;
      if (i + 1 < argc) next_arg = argv[i + 1];
      else              next_arg = "";

      if (arg == "--text") { format = TEXT_OUT; break; }
      if (arg == "--xml")
	{
	  if (next_arg == "")
	    {
	      fname = argv[0];
	      fname += ".xml";
	    }
	  else
	    {
	      fname = next_arg;
	    }
	  format = XML_OUT;
	  ofs.open(fname.c_str());
	}
      if ( arg == "--compiler"  ) { format = COMPILER_OUT; break; }
      if ( arg == "--cerr"      ) { target = 1; break; }
      if ( arg == "--xsl"       )
	{
	  if (next_arg == "") xsl = "default.xsl"; 
	  else                xsl = next_arg;
	}
      if ( arg == "--namespace" )
	{
	  if (next_arg == "")
	    {
	      std::cerr << "no namespace specified" << std::endl;
	      exit(1); 
	    }
	  else
	    {
	      xsl = next_arg;
	    }
	}
      ++i;
    }
  CppUnit::TextUi::TestRunner runner;
  if ( ns.empty() )
    runner.addTest(CppUnit::TestFactoryRegistry::getRegistry().makeTest());
  else
    runner.addTest(CppUnit::TestFactoryRegistry::getRegistry(ns).makeTest());
  CppUnit::Outputter* outputter = 0;
  std::ostream* stream = target ? &std::cerr : &std::cout;
  switch ( format )
    {
    case TEXT_OUT :
      outputter = new CppUnit::TextOutputter(&runner.result(),*stream);
      break;
    case XML_OUT :
      std::cout << "XML_OUT" << std::endl;
      outputter = new CppUnit::XmlOutputter(&runner.result(),
					    ofs, "shift_jis");
      static_cast<CppUnit::XmlOutputter*>(outputter)->setStyleSheet(xsl);
      break;
    case COMPILER_OUT :
      outputter = new CppUnit::CompilerOutputter(&runner.result(),*stream);
      break;
    }
  runner.setOutputter(outputter);
  runner.run();
  return 0; // runner.run() ? 0 : 1;
}
#endif // MAIN
#endif // ModuleManager_cpp
