/*
 * samma
 *
 * Copyright (C) 2006,2007,2008 DesigNET, INC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * $RCSfile: mailzip.c,v $
 * $Revision: 1.5 $
 * $Date: 2009/07/30 23:16:50 $
 */

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <pthread.h>
#include <libmilter/mfapi.h>
#include <libdgmail.h>
#include <gmime/gmime.h>
#include <signal.h>
#include <time.h>

#include "log.h"
#include "mailzip.h"
#include "mailzip_config.h"
#include "client_side.h"
#include "maildrop.h"
#include "mailsave.h"
#include "zipconv.h"
#include "global.h"

#define MLFIPRIV(ctx)        ((struct mlfiPriv *) smfi_getpriv(ctx))

extern pthread_mutex_t gmime_lock;

/*
 *  sfsistat mlfi_connect
 */
sfsistat
mlfi_connect(SMFICTX *ctx, char *hostname, _SOCK_ADDR *addr)
{
    struct mlfiPriv *priv;
    struct mailinfo *minfo;

    DEBUGLOG("mlfi_connect start");

    /* mail save */
    priv = malloc(sizeof *priv);
    if(priv == NULL) {
	log(ERR_MEMORY_ALLOCATE, "mlfi_connect", "priv", strerror(errno));
        return SMFIS_TEMPFAIL;
    }

    memset(priv, 0, sizeof *priv);

    /* read config */
    priv->mlfi_conf = config_init();

    /* create mailinfo structure */
    minfo = malloc(sizeof *minfo);
    if(minfo == NULL) {
	log(ERR_MEMORY_ALLOCATE, "mlfi_connect", "minfo", strerror(errno));
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }
    memset(minfo, 0, sizeof *minfo);

    /* allocate message buffer */
    minfo->ii_mbuf = (char *)malloc(MBSIZE + 1);
    if(minfo->ii_mbuf == NULL) {
	log(ERR_MEMORY_ALLOCATE, "mlfi_connect", "minfo->ii_mbuf", strerror(errno));
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }
    
    /* Initialize mailinfo structure */
    minfo->ii_bufsize = MBSIZE;

    priv->mlfi_minfo = minfo;

    /* copy buffer */
    smfi_setpriv(ctx, priv);

    return SMFIS_CONTINUE;
}

/*
 *  sfsistat mlfi_helo
 */
sfsistat
mlfi_helo(SMFICTX *ctx, char *helohost)
{
    DEBUGLOG("mlfi_helo start");

    return SMFIS_CONTINUE;
}

/*
 *  sfsistat mlfi_envfrom
 *  env
 */
sfsistat
mlfi_envfrom(SMFICTX *ctx, char **envfrom)
{
    char *fromaddr;
    struct mlfiPriv *priv;
    if ((priv = MLFIPRIV(ctx)) == NULL) {
	log(ERR_GET_PRIV, "mlfi_envfrom");
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }

    DEBUGLOG("mlfi_envfrom start");

    /* get mail addr */
    fromaddr = get_addrpart((unsigned char *)*envfrom);
    if (fromaddr == NULL) {
	if (fromaddr != NULL) {
            free(fromaddr);
	}
	mlfi_cleanup(ctx, MLFIABORT);
	log(ERR_MAIL_FIND_ADDRESS, "mlfi_envfrom");
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }

    /* save from addr temporally */
    priv->mlfi_savefrom = fromaddr;

    return SMFIS_CONTINUE;
}

/*
 *  sfsistat mlfi_envrcpt
 *  env
 */
sfsistat
mlfi_envrcpt(SMFICTX *ctx, char **rcptto)
{
    char *rcptaddr = NULL;
    struct mlfiPriv *priv;
    if ((priv = MLFIPRIV(ctx)) == NULL) {
	log(ERR_GET_PRIV, "mlfi_envfrom");
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }

    DEBUGLOG("mlfi_envrcpt start");

    /* get mail addr */
    rcptaddr = get_addrpart((unsigned char *)*rcptto);
    if (rcptaddr == NULL) {
	if (rcptaddr != NULL) {
            free(rcptaddr);
	}
	mlfi_cleanup(ctx, MLFIABORT);
	log(ERR_MAIL_FIND_ADDRESS, "mlfi_envrcpt");
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }

    /* push rcpt addr */
    if (push_rcptlist(&(priv->mlfi_savercpt), rcptaddr) != 0) {
	mlfi_cleanup(ctx, MLFIABORT);
	free(rcptaddr);
	return SMFIS_TEMPFAIL;
    }
    free(rcptaddr);

    return SMFIS_CONTINUE;
}

/*
 * sfsistat mlfi_header
 * header
 */
sfsistat
mlfi_header(SMFICTX *ctx, char *headerf, char *headerv)
{
    char *header;
    int len;
    struct mlfiPriv *priv;
    if ((priv = MLFIPRIV(ctx)) == NULL) {
	log(ERR_GET_PRIV, "mlfi_envfrom");
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }

    DEBUGLOG("mlfi_header start");

    /* create header line */
    len = strlen(headerf) + strlen(headerv) + 4;
    header = malloc(len + 1);
    if (!header) {
        log(ERR_MEMORY_ALLOCATE, "mlfi_header", "header", strerror(errno));
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }
    sprintf(header, "%s: %s\n", headerf, headerv);

    /* write header */
    if (mailsave_write(priv->mlfi_minfo, priv->mlfi_conf, header, len - 1) != 0 ) {
        /* write error */
        log(ERR_MAILSAVE_WRITE, "mlfi_header", header);
        free(header);
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }
    free(header);

    return SMFIS_CONTINUE;
}

/*
 * mlfi_eoh
 * header & body
 */
sfsistat
mlfi_eoh(SMFICTX *ctx)
{
    struct mlfiPriv *priv;
    if ((priv = MLFIPRIV(ctx)) == NULL) {
	log(ERR_GET_PRIV, "mlfi_envfrom");
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }

    DEBUGLOG("mlfi_eoh start");

    /* write new line between header and body */
    if (mailsave_write(priv->mlfi_minfo, priv->mlfi_conf, "\n", 1) == -1 ) {
        /* write error */
        log(ERR_MAILSAVE_WRITE, "mlfi_eoh", "blank line");
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }

    return SMFIS_CONTINUE;
}

/*
 * mlfi_body
 * write body
 */
sfsistat
mlfi_body(SMFICTX *ctx, u_char *bodyp, size_t bodylen)
{
    struct mlfiPriv *priv;
    if ((priv = MLFIPRIV(ctx)) == NULL) {
	log(ERR_GET_PRIV, "mlfi_envfrom");
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }

    DEBUGLOG("mlfi_body start");

    /* write body */
    if (mailsave_write(priv->mlfi_minfo, priv->mlfi_conf, (char *)bodyp, bodylen) == -1 ) {
        /* write error */
        log(ERR_MAILSAVE_WRITE, "mlfi_body", "body data");
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }

    return SMFIS_CONTINUE;
}

/*
 * mlfi_eom
 * end
 */
sfsistat
mlfi_eom(SMFICTX *ctx)
{
    int ret;
    struct mlfiPriv *priv;
    if ((priv = MLFIPRIV(ctx)) == NULL) {
	log(ERR_GET_PRIV, "mlfi_envfrom");
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    }

    DEBUGLOG("mlfi_eom start");

    ret = zip_convert_mail(ctx, priv->mlfi_minfo, priv->mlfi_conf, 
			   priv->mlfi_savefrom, priv->mlfi_savercpt);
    if (ret == PM_FAILED) {
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_TEMPFAIL;
    } else if (ret == ZIP_CONVERT_ACCEPT) {
        mlfi_cleanup(ctx, MLFIABORT);
        return SMFIS_ACCEPT;
    }

    return SMFIS_CONTINUE;
}

/*
 * mlfi_close
 * close
 */
sfsistat
mlfi_close(SMFICTX *ctx)
{
    DEBUGLOG("mlfi_close start");
    mlfi_cleanup(ctx, MLFICLOSE);
    return SMFIS_CONTINUE;
}

/*
 * mlfi_abort
 * abort
 */
sfsistat
mlfi_abort(SMFICTX *ctx)
{
    DEBUGLOG("mlfi_abort start");

    mlfi_cleanup(ctx, MLFIABORT);
    return SMFIS_CONTINUE;
}

/*
 * mlfi_cleanup
 * cleanup
 */
sfsistat
mlfi_cleanup(SMFICTX *ctx, int flag)
{
    sfsistat rstat = SMFIS_CONTINUE;
    struct mlfiPriv *priv;
    /* null data return */
    if ((priv = MLFIPRIV(ctx)) == NULL) {
        return rstat;
    }

    /* priv release */
    if (priv->mlfi_savefrom != NULL) {
        free(priv->mlfi_savefrom);
	priv->mlfi_savefrom = NULL;
    }

    if (priv->mlfi_savercpt != NULL) {
        free_rcptlist(priv->mlfi_savercpt);
	priv->mlfi_savercpt = NULL;
    }

    if (flag == MLFICLOSE) {
        /* config relase */
        if (priv->mlfi_conf != NULL) {
            config_release(priv->mlfi_conf);
	    priv->mlfi_conf = NULL;
        }

	if (priv->mlfi_minfo != NULL) {
	    mailsave_clean(priv->mlfi_minfo);
	    free(priv->mlfi_minfo);
	    priv->mlfi_minfo = NULL;
	}

	free(priv);

	/* set NULL */
	smfi_setpriv(ctx, NULL);

    }

    return rstat;
}

static void *
call_command(void *arg)
{
    struct thread_control *tcl;
    int ret = 0;

    tcl = (struct thread_control *)arg;

    if (accept_command(tcl->configname , tcl->addr, tcl->port) != 0) {
	log(ERR_RELOAD_SERVER, "call_command");
    }

    pthread_exit(&ret);
}

/*
 * usage
 * 
 */
void
usage(char *arg)
{
    fprintf(stderr, "usage: %s [-t timeout] [config file]\n", arg);
}

int
main(int argc, char *argv[])
{

    char  *args = "t:";
    char  *confname;
    char  *tmp;
    char   c;
    char  oconn[OCONN_LENGTH];
    int    timeout = DEFAULT_TIMEOUT;
    unsigned long tmp_val;
    struct thread_control tcl;
    struct config *cfg;

    /* set rand function */
    srand48(time(NULL));

    /* ignore sigpipe signal */
    signal(SIGPIPE, SIG_IGN); 

    /* set error output to stderr */
    init_log();

    /* arg */
    while ((c = getopt(argc, argv, args)) != -1) {

        switch (c) {

        /* arg time out */
        case 't':
            if((optarg == NULL) || (optarg[0] == '\0')) {
                usage(argv[0]);
                exit(1);
            }

            tmp_val = strtoul(optarg, &tmp, 10);
            if((*tmp != '\0') ||
               ((tmp_val == ULONG_MAX) && (errno == ERANGE)) ||
	       (tmp_val > INT_MAX)) {
                usage(argv[0]);
                exit(1);
            }

            timeout = (int) tmp_val;
            break;

        /* error */
        default:
            usage(argv[0]);
            exit(1);
        }
    }

    /* check arg */
    if (argc > MAX_ARGS) {
	usage(argv[0]);
	exit(1);
    } else if(optind + 1 == argc) {

	/* set config name */
	confname = argv[optind];
    } else if (optind == argc) {
	/* set default config name */
	confname = DEFAULT_CONFFILE;
    } else {
	usage(argv[0]);
	exit(1);
    }

    /* read config file */
    if (reload_config(confname) != 0) {
	fprintf(stderr, STDERR_CONFIG_READ, confname);
	fprintf(stderr, "\n");
        exit(2);
    }

    cfg = config_init();

    /* set thread data */
    tcl.configname = confname;
    tcl.port = cfg->cf_commandport;
    tcl.addr = strdup(cfg->cf_listenip);
    if (tcl.addr == NULL) {
	fprintf(stderr, STDERR_MEMORY_ALLOCATE, "tcl.addr", strerror(errno));
	fprintf(stderr, "\n");
	exit(2);
    }

    /* initialize GMime */
    pthread_mutex_lock(&gmime_lock);
    g_mime_init (GMIME_ENABLE_RFC2047_WORKAROUNDS);
    pthread_mutex_unlock(&gmime_lock);

    /* prepare socket string */
    sprintf(oconn, OCONN, cfg->cf_listenport, cfg->cf_listenip);

    /* config relase */
    config_release(cfg);

    /* command line thread */
    pthread_create(&child, NULL, call_command, &tcl);

    /* set socket */
    if(smfi_setconn(oconn) == MI_FAILURE) {
        fprintf(stderr, ERR_MILTER_SET_SOCKET, strerror(errno));
	fprintf(stderr, "\n");
	pthread_mutex_lock(&gmime_lock);
	g_mime_shutdown();
	pthread_mutex_unlock(&gmime_lock);
        exit(3);
    }

    /* set time out */
    if(smfi_settimeout(timeout) == MI_FAILURE) {
        fprintf(stderr, ERR_MILTER_SET_TIMEOUT, strerror(errno));
	fprintf(stderr, "\n");
	pthread_mutex_lock(&gmime_lock);
	g_mime_shutdown();
	pthread_mutex_unlock(&gmime_lock);
        exit(3);
    }

    /* set register */
    if(smfi_register(smfilter) == MI_FAILURE) {
        fprintf(stderr, ERR_MILTER_REGISTER, strerror(errno));
	fprintf(stderr, "\n");
	pthread_mutex_lock(&gmime_lock);
	g_mime_shutdown();
	pthread_mutex_unlock(&gmime_lock);
        exit(4);
    }

    /* run main */
    if(smfi_main() == MI_FAILURE) {
        fprintf(stderr, ERR_MILTER_START, strerror(errno));
	fprintf(stderr, "\n");
	pthread_mutex_lock(&gmime_lock);
	g_mime_shutdown();
	pthread_mutex_unlock(&gmime_lock);
        exit(5);
    }

    pthread_mutex_lock(&gmime_lock);
    g_mime_shutdown();
    pthread_mutex_unlock(&gmime_lock);
    exit(0);
}
