/*
 * samma
 *
 * Copyright (C) 2006,2007,2008 DesigNET, INC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * $RCSfile: mailzip_config.c,v $
 * $Revision: 1.3 $
 * $Date: 2009/07/30 00:52:16 $
 */

#include <stdio.h>
#include <errno.h>
#include <limits.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/stat.h>
#include <syslog.h>
#include <pthread.h>
#include <libdgconfig.h>
#include <gmime/gmime.h>

#define _MAILZIP_CONFIG_C_
#include "mailzip_config.h"

#include "log.h"
#include "mailzip_db.h"
#include "mailzip_tmpl.h"

static pthread_mutex_t config_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t config_ref_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t config_ref_old_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t config_reload_lock = PTHREAD_MUTEX_INITIALIZER;
static int config_ref = 0;
static int config_ref_old = 0;

struct config *cur_cfg = NULL;
struct config *new_cfg = NULL;
struct config *old_cfg = NULL;

static pthread_mutex_t tmpl_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t tmpl_ref_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t tmpl_ref_old_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t tmpl_reload_lock = PTHREAD_MUTEX_INITIALIZER;
static int tmpl_ref = 0;
static int tmpl_ref_old = 0;

char *cur_tmpl = NULL;
char *new_tmpl = NULL;
char *old_tmpl = NULL;

/*
 * setup_config()
 *
 * Setup config structure.
 *
 * args: (void)
 *
 * Return value:
 *  struct config *	Success
 *  NULL		error
 */
static struct config *
setup_config()
{
    struct config *cf = NULL;

    // allocate memory
    cf = (struct config *)malloc(sizeof(struct config));
    if (cf == NULL) {
	log(ERR_MEMORY_ALLOCATE, "setup_config", "cf", strerror(errno));
	return NULL;
    }

    // set all members to NULL
    memset(cf, 0, sizeof(struct config));
    return cf;
}

static void
free_config(struct config *cfg)
{
    if (cfg == NULL) {
	return;
    }
    if (cfg->cf_syslogfacility != NULL) {
	free(cfg->cf_syslogfacility);
    }
    if (cfg->cf_listenip != NULL) {
	free(cfg->cf_listenip);
    }
    if (cfg->cf_encryptiontmpdir != NULL) {
	free(cfg->cf_encryptiontmpdir);
    }
    if (cfg->cf_zipcommand != NULL) {
	free(cfg->cf_zipcommand);
    }
    if (cfg->cf_zipcommandopt != NULL) {
	free(cfg->cf_zipcommandopt);
    }
    if (cfg->cf_senderdb != NULL) {
	free(cfg->cf_senderdb);
    }
    if (cfg->cf_rcptdb != NULL) {
	free(cfg->cf_rcptdb);
    }
    if (cfg->cf_templatepath != NULL) {
	free(cfg->cf_templatepath);
    }
    if (cfg->cf_senderdbpath != NULL) {
	free(cfg->cf_senderdbpath);
    }
    if (cfg->cf_rcptdbpath != NULL) {
	free(cfg->cf_rcptdbpath);
    }
    free(cfg);
    return;
}

static void *
cfree_handler(void *arg)
{
    int ret = 0;
    pthread_mutex_lock(&config_reload_lock);
    free_config(old_cfg);
    old_cfg = NULL;
    pthread_mutex_unlock(&config_reload_lock);
    pthread_exit(&ret);
    return NULL;
}

/*
 * config_init()
 *
 * Get current config structure pointer,
 * and countup the reference counter.
 *
 * Args: (void)
 *
 * Return value:
 *  struct config *		config structure pointer
 */
struct config *
config_init()
{
    struct config *ret_ptr;

    // countup reference counter
    pthread_mutex_lock(&config_lock);
    pthread_mutex_lock(&config_ref_lock);
    config_ref ++;

    // get config structure pointer
    ret_ptr = cur_cfg;

    pthread_mutex_unlock(&config_ref_lock);
    pthread_mutex_unlock(&config_lock);
    return ret_ptr;
}

/*
 * config_release()
 * 
 * Countdown config reference counter.
 *
 * Args:
 *  struct config *cfg		To release pointer.
 *
 * Return value:
 *  (void)
 */
void
config_release(struct config *cfg)
{
    pthread_mutex_lock(&config_lock);
    if (old_cfg == cfg) {
	// case to release old config
	pthread_mutex_lock(&config_ref_old_lock);
	config_ref_old --;
	if (config_ref_old == 0) {
	    pthread_mutex_unlock(&config_reload_lock);
	}
	pthread_mutex_unlock(&config_ref_old_lock);
    } else {
	// case to release cur config
	pthread_mutex_lock(&config_ref_lock);
	config_ref --;
	pthread_mutex_unlock(&config_ref_lock);
    }
    pthread_mutex_unlock(&config_lock);
}

/*
 * reload_config()
 *
 * Reload configuration file
 *
 * Args:
 *  char *file		Configuration file name
 *
 * Return value:
 *  0			Success
 *  CFG_NG		System error
 *  1			Temporaly error (during reloading)
 *  2			Temporaly error (because of config file)
 */
int
reload_config(char *file)
{
    struct config *cfg = NULL;
    int ret;
    pthread_t cfree;
    int i;

    pthread_mutex_lock(&config_ref_old_lock);
    ret = config_ref_old;
    pthread_mutex_unlock(&config_ref_old_lock);
    if (ret > 0) {
	// case reloading
	log(ERR_CONFIG_RELOADING, "reload_config", file);
	return 1;
    }

    // setup config structure
    cfg = setup_config();
    if (cfg == NULL) {
	return CFG_NG;
    }

    // read config file
    ret = read_config(file, cfe, NCONFIG, cfg);

    if (ret > 0) {
	// case config file error
	free_config(cfg);
	return 2;
    }
    if (ret < 0) {
	// case system error
	free_config(cfg);
	return CFG_NG;
    }

    // set db infomation
    for (i = 0; i < DBNUM; i++) {
        ret = strncmp(cfg->cf_senderdb, db_kind[i].db_name, db_kind[i].db_len);
        if (ret == 0) {
            cfg->cf_senderdbpath = strdup(cfg->cf_senderdb + db_kind[i].db_len);
	    cfg->cf_senderdbtype = db_kind[i].db_type;
        }
        ret = strncmp(cfg->cf_rcptdb, db_kind[i].db_name, db_kind[i].db_len);
        if (ret == 0) {
            cfg->cf_rcptdbpath = strdup(cfg->cf_rcptdb + db_kind[i].db_len);
	    cfg->cf_rcptdbtype = db_kind[i].db_type;
        }
	if ((cfg->cf_senderdbpath != NULL) && (cfg->cf_rcptdbpath != NULL)) {
	    break;
	}
    }

    // template file reload 
    if (reload_tmpl(cfg->cf_templatepath) != 0) {
	free_config(cfg);
	return CFG_NG;
    }

    // change config structure
    pthread_mutex_lock(&config_lock);
    pthread_mutex_lock(&config_ref_lock);
    pthread_mutex_lock(&config_ref_old_lock);
    config_ref_old = config_ref;
    config_ref = 0;
    if (config_ref_old > 0) {
	pthread_mutex_lock(&config_reload_lock);
    }
    pthread_mutex_unlock(&config_ref_old_lock);
    pthread_mutex_unlock(&config_ref_lock);

    // create config free thread
    ret = pthread_create(&cfree, NULL, cfree_handler, NULL);
    if (ret != 0) {
	free_config(cfg);
	pthread_mutex_unlock(&config_lock);
	log(ERR_THREAD_CREATE, "reload_config",
	    "cfree_handler", strerror(errno));
	return CFG_NG;
    }
    ret = pthread_detach(cfree);
    if (ret != 0) {
	free_config(cfg);
	pthread_mutex_unlock(&config_lock);
	log(ERR_THREAD_DETACH, "reload_config",
	    "cfree_handler", strerror(errno));
	return CFG_NG;
    }

    // switch pointer
    old_cfg = cur_cfg;
    cur_cfg = cfg;
    switch_log(cfg->cf_syslogfacility);
    pthread_mutex_unlock(&config_lock);

    return 0;
}

static void *
tfree_handler(void *arg)
{
    int ret = 0;
    pthread_mutex_lock(&tmpl_reload_lock);
    free(old_tmpl);
    old_tmpl = NULL;
    pthread_mutex_unlock(&tmpl_reload_lock);
    pthread_exit(&ret);
    return NULL;
}

/*
 * tmpl_init()
 *
 * Get current template data pointer,
 * and countup the reference counter.
 *
 * Args: (void)
 *
 * Return value:
 *  char *		template data pointer
 */
char *
tmpl_init()
{
    char *ret_ptr;

    // countup reference counter
    pthread_mutex_lock(&tmpl_lock);
    pthread_mutex_lock(&tmpl_ref_lock);
    tmpl_ref ++;

    // get template data pointer
    ret_ptr = cur_tmpl;

    pthread_mutex_unlock(&tmpl_ref_lock);
    pthread_mutex_unlock(&tmpl_lock);
    return ret_ptr;
}

/*
 * tmpl_release()
 * 
 * Countdown template reference counter.
 *
 * Args:
 *  char *tmpdata		To release pointer.
 *
 * Return value:
 *  (void)
 */
void
tmpl_release(char *tmpdata)
{
    pthread_mutex_lock(&tmpl_lock);
    if (old_tmpl == tmpdata) {
	// case to release old template data
	pthread_mutex_lock(&tmpl_ref_old_lock);
	tmpl_ref_old --;
	if (tmpl_ref_old == 0) {
	    pthread_mutex_unlock(&tmpl_reload_lock);
	}
	pthread_mutex_unlock(&tmpl_ref_old_lock);
    } else {
	// case to release cur template data
	pthread_mutex_lock(&tmpl_ref_lock);
	tmpl_ref --;
	pthread_mutex_unlock(&tmpl_ref_lock);
    }
    pthread_mutex_unlock(&tmpl_lock);
}

/*
 * reload_tmpl()
 *
 * Reload template file
 *
 * Args:
 *  char *file		Configuration file name
 *
 * Return value:
 *  0			Success
 *  CFG_NG		System error
 *  1			Temporaly error (during reloading)
 *  2			Temporaly error (because of template file)
 */
int
reload_tmpl(char *file)
{
    int ret;
    pthread_t tfree;
    char *tmpl_data = NULL;

    pthread_mutex_lock(&tmpl_ref_old_lock);
    ret = tmpl_ref_old;
    pthread_mutex_unlock(&tmpl_ref_old_lock);
    if (ret > 0) {
	// case reloading
	log(ERR_TEMPLATE_RELOADING, "reload_tmpl", file);
	return 1;
    }

    // read tmplate file
    if (tmpl_read(&(tmpl_data), file) != 0) {
	return 2;
    }

    // change template data
    pthread_mutex_lock(&tmpl_lock);
    pthread_mutex_lock(&tmpl_ref_lock);
    pthread_mutex_lock(&tmpl_ref_old_lock);
    tmpl_ref_old = tmpl_ref;
    tmpl_ref = 0;
    if (tmpl_ref_old > 0) {
	pthread_mutex_lock(&tmpl_reload_lock);
    }
    pthread_mutex_unlock(&tmpl_ref_old_lock);
    pthread_mutex_unlock(&tmpl_ref_lock);

    // create template free thread
    ret = pthread_create(&tfree, NULL, tfree_handler, NULL);
    if (ret != 0) {
	free(tmpl_data);
	pthread_mutex_unlock(&tmpl_lock);
	log(ERR_THREAD_CREATE, "reload_tmpl",
	    "tfree_handler", strerror(errno));
	return CFG_NG;
    }
    ret = pthread_detach(tfree);
    if (ret != 0) {
	free(tmpl_data);
	pthread_mutex_unlock(&tmpl_lock);
	log(ERR_THREAD_DETACH, "reload_tmpl",
	    "tfree_handler", strerror(errno));
	return CFG_NG;
    }

    // switch pointer
    old_tmpl = cur_tmpl;
    cur_tmpl = tmpl_data;
    pthread_mutex_unlock(&tmpl_lock);

    return 0;
}

/*
 * is_dbpath()
 *
 * Check if the string is valid DB type.
 * Check if the string is valid path.
 *
 * Args:
 *  char *path			string
 *
 * Return value:
 *  NULL			Success
 *  Invalid argument		Error
 */
static char *
is_dbpath(char *path)
{
    static char errbuf[MAX_CONFIG_LINE];
    char *retp, *file = NULL;
    int   ret, i, n;

    for (i = 0; i < DBNUM; i++) {
        ret = strncmp(path, db_kind[i].db_name, db_kind[i].db_len);
        if (ret == 0) {
            file = path + db_kind[i].db_len;
            break;
        }
    }
    if (file == NULL || *file == '\0') {
        n = sizeof(ERR_CONF_DBPATH) + strlen(path);
        snprintf(errbuf, n, ERR_CONF_DBPATH, path);
        return(errbuf);
    }
    retp = is_readable_file(file);
    if (retp != NULL) {
        n = sizeof(ERR_CONF_DBPATH) + strlen(path);
        snprintf(errbuf, n, ERR_CONF_DBPATH, path);
        return(errbuf);
    }
    return (NULL);
}

char *
is_executable_file(char *str)
{
    static char errbuf[MAX_CONFIG_LINE];
    struct stat st;
    int n = 0;

    if (stat(str, &st) < 0) {
        n = sizeof(ERR_CONF_FILEDIR) + strlen(str) + strlen(strerror(errno));
        snprintf(errbuf, n, ERR_CONF_FILEDIR, str, strerror(errno));
        return (errbuf);
    }

    if (S_ISDIR(st.st_mode)) {
        errno = EISDIR;
        n = sizeof(ERR_CONF_FILEDIR) + strlen(str) + strlen(strerror(errno));
        snprintf(errbuf, n, ERR_CONF_FILEDIR, str, strerror(errno));
        return (errbuf);
    }

    if (access(str, X_OK) != 0) {
        n = sizeof(ERR_CONF_FILEDIR) + strlen(str) + strlen(strerror(errno));
        snprintf(errbuf, n, ERR_CONF_FILEDIR, str, strerror(errno));
        return (errbuf);
    }
    return (NULL);
}

char *
is_passwd_length(int len)
{
    if ((len < PASSMIN) || (PASSMAX < len)) {
        return (ERR_CONF_PASSLEN);
    }
    return (NULL);
}

char *
is_strcode(char *str)
{
    int i;

    for (i = 0; i < STRCODENUM; i++) {
        if (strncmp(str, str_code[i].code_name, str_code[i].code_len) == 0) {
	    return (NULL);
        }
    }
    return (ERR_STR_CODE);
}

char *
is_notnull(char *str)
{
    if (str != NULL) {
	if (strlen(str) > 0) {
	    return (NULL);
	}
    }
    return (ERR_NULL_VALUE);
}


