#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <libdgstr.h>
#include <gmime/gmime.h>
#include "zipconv.h"
#include "log.h"
#include "mailzip_tmpl.h"
#include "maildrop.h"

/*
 * tmpl_read()
 *
 * Read the template file.
 *
 * Args:
 *   char **tmp      pointer
 *   char *tmp_file  file path
 *
 * Return value:
 *   ERR_TMP   -1    error
 *   READ_TMP   0    success
 *
 */
int
tmpl_read(char **tmp, char *tmpl_file) 
{
    int fd, ret = 1;
    char buf[BUFSIZE];
    char *tmpr = NULL;
    int total = 0;

    // file open
    fd = open(tmpl_file, O_RDONLY);
    if (fd == -1) {
        log(ERR_FILE_OPEN, "tmpl_read", "template file open error");
        return(ERR_TMP);
    }
    // file read
    while (ret != 0) {
        ret = read(fd, buf, BUFSIZE - 1);
        if (ret == 0) {
            break;
        }
        // error
        if (ret == -1) {
            log(ERR_FILE_GET_INFO, "tmpl_read", "fd", "template file read error");
	    if (*tmp != NULL) {
		free(*tmp);
	    }
            return(ERR_TMP);
        }
        buf[ret] = '\0';

        tmpr = (char *)realloc(*tmp, (total + ret + 1));
        if (tmpr == NULL) {
            if (*tmp != NULL) {
                free(*tmp);
            } 
            log(ERR_MEMORY_ALLOCATE, "tmpl_read", "tmpr", strerror(errno));
            close(fd);
            return(ERR_TMP);
        }
        *tmp = tmpr;

        strcpy((*tmp + total), buf);
        total = total + ret;

    }
    //file close
    close(fd);

    if (*tmp == NULL) {
        log(ERR_TEMPLATE_READ, "tmpl_read", "file is empty");
        return(ERR_TMP);
    }

    return(READ_TMP);
}


#define NOSUBJECT	""
/*
 * tmpl_tag_replace()
 *
 * Args:
 *   char *           read file data 
 *   struct mailzip   reference data
 *   struct rcptinfo  reference data
 *   char *           recipient address 
 *   char **          replaced date
 *
 * Return:
 *   ERR_FUNC    -1   error
 *   ERR_VALUE    1   error
 *   REPLACED_TAG 0   success
 */
int 
tmpl_tag_replace(char *tmp, struct mailzip mz, struct rcptinfo *other, char *from, char **retp)
{

    int i;
    char *start_tag = "<@@";
    char *end_tag = "@@>";
    char *subject = NULL;
    char *filename = NULL;
    char *tmpl = NULL;

    struct strtag tmpl_tag[] = {
        {"SUBJECT", 7, NULL},
        {"DATE", 4, NULL},
        {"PASSWORD", 8, NULL},
        {"RCPTLIST", 8, NULL},
        {"FILENAME", 8, NULL},
        {"TOADDR", 6, NULL}
    };
    char *rcptaddr, *tmpp = NULL, *tmptagp = NULL;
    int ret = 0, total = 0;
    int rcptaddr_len = 0;
    char *name, *tmpf = NULL;
    int name_len = 0, ret_len = 0, total_len = 0;

    // subject
    if (mz.subject != NULL) {
	if (dg_str2code((char *)mz.subject, &subject, "UTF8", "ISO2022JP") != 0) {
            log(ERR_CHARCODE_CONVERT, "tmpl_tag_replace");
	    return(ERR_FUNC);
	}
        tmpl_tag[SUBJECT].st_str = subject;
    } else {
        tmpl_tag[SUBJECT].st_str = NOSUBJECT;
    }
    // date
    tmpl_tag[DATE].st_str = mz.date;
    // passwd
    tmpl_tag[PASSWORD].st_str = other->passwd;

    // rcptaddr
    // measure total length
    for (i = 0; (other->rcptlist + i)->rcpt_addr != NULL; i++) {
        rcptaddr_len = (other->rcptlist + i)->rcpt_addr_len;
        ret = ret + rcptaddr_len + 1;
    }
    // allocate memory
    tmpp = (char *)malloc(ret + 1);
    if (tmpp == NULL) {
        log(ERR_MEMORY_ALLOCATE, "tmpl_tag_replace", "tmpp", strerror(errno));
	if (subject != NULL) {
	    free(subject);
	}
        return(ERR_FUNC);
    }
    // reset memory
    *tmpp = '\0';

     // write memory
    for (i = 0; (other->rcptlist + i)->rcpt_addr != NULL; i++) {
        rcptaddr = (other->rcptlist + i)->rcpt_addr;
        rcptaddr_len = (other->rcptlist + i)->rcpt_addr_len;

        strcat(tmpp, rcptaddr);
        *(tmpp + total + rcptaddr_len) = '\n';
        *(tmpp + total + rcptaddr_len + 1) = '\0';
        total = total + rcptaddr_len + 1;
    }
    // last '\n' change into '\0'
    *(tmpp + total - 1) = '\0';
    // store all rcptaddr 
    tmpl_tag[RCPTLIST].st_str = tmpp;


    //file_name
    if (mz.namelist != NULL) {
        for (i = 0; (mz.namelist + i)->attach_name != NULL; i++) {
            // measure total length
            ret_len = ret_len + (mz.namelist + i)->attach_name_len + 1;
        }
        // allocate memory
        tmpf = (char *)malloc(ret_len + 1);
        if (tmpf == NULL) {
            log(ERR_MEMORY_ALLOCATE, "tmpl_tag_replace", "tmpf", strerror(errno));
	    free(tmpp);
	    if (subject != NULL) {
	        free(subject);
	    }
            return(ERR_FUNC);
        }
        // reset memory
        *tmpf = '\0';
    } else {
        log(ERR_NULL_FILE_NAME, "tmpl_tag_replace");    
	free(tmpp);
	if (subject != NULL) {
	    free(subject);
	}
        return(ERR_VALUE);
    }

    // write memory
    for (i = 0; (mz.namelist + i)->attach_name != NULL; i++) {
        name = (mz.namelist + i)->attach_name;
        name_len = (mz.namelist + i)->attach_name_len;

        strcat(tmpf, name);
        *(tmpf + total_len + name_len) = '\n';
        *(tmpf + total_len + name_len + 1) = '\0';
        total_len = total_len + name_len + 1;
    }

    // last '\n' change into '\0'
    *(tmpf + total_len - 1) = '\0';
    // store all filename 
    if (dg_str2code(tmpf, &filename, "UTF8", "ISO2022JP") != 0) {
        log(ERR_CHARCODE_CONVERT, "tmpl_tag_replace");
	free(tmpp);
	if (subject != NULL) {
	    free(subject);
	}
	free(tmpf);
	return(ERR_FUNC);
    }
    tmpl_tag[FILENAME].st_str = filename;
    free(tmpf);

    // toaddr
    tmpl_tag[TOADDR].st_str = from;

    //replace tag
    if (dg_str2code(tmp, &tmpl, "UTF8", "ISO2022JP") != 0) {
        log(ERR_CHARCODE_CONVERT, "tmpl_tag_replace");
	free(tmpp);
	if (subject != NULL) {
	    free(subject);
	}
	free(filename);
	return(ERR_FUNC);
    }
    tmptagp = str_replace_tag(tmpl, start_tag, end_tag, tmpl_tag, 6);
    if (tmptagp == NULL) {
	free(tmpp);
	if (subject != NULL) {
	    free(subject);
	}
	free(filename);
	free(tmpl);
        log(ERR_MEMORY_ALLOCATE, "tmpl_tag_replace", "tmptagp", strerror(errno));
        return(ERR_FUNC);
    }
    free(tmpp);
    if (subject != NULL) {
	free(subject);
    }
    free(filename);
    free(tmpl);

    //convert str code
    if (dg_str2code(tmptagp, retp, "UTF8", "ISO2022JP") != 0) {
        log(ERR_CHARCODE_CONVERT, "tmpl_tag_replace");
	free(tmptagp);
	return(ERR_FUNC);
    }
    free(tmptagp);

    return(REPLACED_TAG);
}

