<?
/**
 * Samurai_Yaml
 * 
 * Yaml解析用のクラス。
 * 独自に実装してもいいのだが、基本的にはSpycへ依存。
 * 
 * @package    Samurai
 * @subpackage Component.Samurai
 * @copyright  Befool,Inc.
 * @author     Satoshi Kiuchi <satoshi.kiuchi@befool.co.jp>
 * @license    http://opensource.org/licenses/bsd-license.php  The modified BSD License
 */
class Samurai_Yaml
{
    /**
     * コンストラクタ。
     * @access     private
     */
    private function __construct()
    {
    }
    
    
    
    
    
    /**
     * ロード。
     * @access     public
     * @param      string  $yaml_file   YAMLファイルパス
     * @return     array   パース結果
     */
    public static function load($yaml_file)
    {
        $result = array();
        $result = self::_merge($result, self::_load($yaml_file));
        if(defined('SAMURAI_APPLICATION_NAME')){
            $app_yaml_file = self::_replaceYamlName($yaml_file, SAMURAI_APPLICATION_NAME);
            $result = self::_merge($result, self::_load($app_yaml_file));
        }
        if(defined('SAMURAI_ENVIRONMENT')){
            $env_yaml_file = self::_replaceYamlName($yaml_file, SAMURAI_ENVIRONMENT);
            $result = self::_merge($result, self::_load($env_yaml_file));
        }
        return $result;
    }
    
    
    /**
     * ロード処理実体。
     * 一度ロードしたファイルはキャッシュします。
     * @access     private
     * @param      string  $yaml_file   YAMLファイルパス
     * @return     array   パース結果
     */
    private static function _load($yaml_file)
    {
        if($cache_file = self::hasCache($yaml_file)){
            return self::loadByCache($cache_file);
        } elseif(self::enableSyck()){
            return self::loadBySyck($yaml_file);
        } elseif(self::enableSpyc()){
            return self::loadBySpyc($yaml_file);
        } else {
            throw new Samurai_Exception('YAML parser (example Syck or Spyc) is not found...');
        }
    }
    
    
    /**
     * キャッシュからロードする。
     * @access     public
     * @param      string  $cache_file   キャッシュパス
     * @return     array   パース結果
     */
    public static function loadByCache($cache_file)
    {
        $result = unserialize(file_get_contents($cache_file));
        return $result;
    }
    
    
    /**
     * Syckを使用してロードする。
     * @access     public
     * @param      string  $yaml_file   YAMLファイルパス
     * @return     array   パース結果
     */
    public static function loadBySyck($yaml_file)
    {
        $contents = self::_includeYaml($yaml_file);
        $result = syck_load($contents);
        if(!$result) $result = array();
        self::_saveCache($yaml_file, $result);
        return $result;
    }
    
    
    /**
     * Spycを使用してロードする。
     * @access     public
     * @param      string  $yaml_file   YAMLファイルパス
     * @return     array   
     */
    public static function loadBySpyc($yaml_file)
    {
        self::_loadSpyc();
        $Spyc = new Spyc();
        $contents = self::_includeYaml($yaml_file);
        $result = $Spyc->load($contents);
        if(!$result) $result = array();
        self::_saveCache($yaml_file, $result);
        return $result;
    }
    
    
    
    
    
    /**
     * キャッシュが存在するかどうか。
     * @access     public
     * @param      string  $yaml_file
     * @return     boolean キャッシュが存在するかどうか
     */
    public static function hasCache($yaml_file)
    {
        $cache_file = Samurai_Loader::getPath(sprintf('temp/yaml/%s', urlencode($yaml_file)));
        $yaml_file = Samurai_Loader::getPath($yaml_file);
        if(Samurai_Loader::isReadable($cache_file) && Samurai_Loader::isReadable($yaml_file)){
            return filemtime($cache_file) > filemtime($yaml_file) ? $cache_file : false ;
        }
        return false;
    }
    /**
     * キャッシュを保存する
     * @access     private
     */
    private function _saveCache($yaml_file, $data=array())
    {
        if(!Samurai_Config::get('enable.yaml_cache')) return false;
        $cache_dir = Samurai_Loader::getPath('temp' . '/yaml', true);
        if(!file_exists($cache_dir)) mkdir($cache_dir) && @chmod($cache_dir, 0777) ;
        $cache_file = sprintf('%s/%s', $cache_dir, urlencode($yaml_file));
        file_put_contents($cache_file, serialize($data)) && @chmod($cache_file, 0777);
    }
    
    
    /**
     * Syackでの解析が可能かどうか。
     * つまり、syack_load関数が存在するかどうかである。
     * @access     public
     * @return     boolean Syacでの解析が可能かどうか
     */
    public static function enableSyck()
    {
        return function_exists('syck_load');
    }
    
    
    /**
     * Spycでの解析が可能かどうか。
     * つまり、Spycライブラリが存在し、かつ読み込めて、利用できるかどうかである。
     * @access     public
     * @return     boolean Spycでの解析が可能かどうか
     */
    public static function enableSpyc()
    {
        return Samurai_Loader::isReadable(Samurai_Loader::getPath('library/Spyc/spyc.php'))
                    || Samurai_Loader::isReadable(Samurai_Loader::getPath('spyc/spyc.php', false, explode(PS, get_include_path())));
    }
    /**
     * Spycライブラリをロードする。
     * @access     private
     */
    private static function _loadSpyc()
    {
        $spyc = Samurai_Loader::getPath('library/Spyc/spyc.php');
        if(Samurai_Loader::isReadable($spyc)){
            include_once($spyc);
            return true;
        }
        $spyc = Samurai_Loader::getPath('spyc/spyc.php', false, explode(PS, get_include_path()));
        if(Samurai_Loader::isReadable($spyc)){
            include_once($spyc);
            return true;
        }
    }
    
    
    /**
     * YamlファイルにPHPコードを記述できるように一工夫。
     * ただし、キャッシュされるファイルに関してはPHPコード解釈後の内容がキャッシュされるので、
     * 元の情報を変更した後は、キャッシュを削除することをおすすめします。
     * @access     private
     * @param      string  $yaml_file   YAMLファイルパス
     * @return     string  YAMLのPHPコード解釈後の結果
     */
    private static function _includeYaml($yaml_file)
    {
        $yaml_file = Samurai_Loader::getPath($yaml_file);
        if(Samurai_Loader::isReadable($yaml_file)){
            if(class_exists('Samurai_Logger', false)) Samurai_Logger::debug('YAML loaded. -> %s', $yaml_file);
            ob_start();
            include($yaml_file);
            $contents = ob_get_clean();
            $contents = $contents === NULL ? '' : preg_replace_callback('/%([a-z0-9\._]*?)%/i', 'Samurai_Yaml::_tag2Entity', $contents) ;
            return $contents;
        }
        return '[]';
    }
    /**
     * YAMLファイルに「%...%」形式で書かれた記述を展開する。
     * %の中身が大文字のみで構成されていると、定数として解釈しようとします。
     * その他の場合は、全てSamurai_Configから取得しようとします。
     * @access     private
     * @param      array   $matches   ヒットした箇所の情報
     * @return     string  適宜情報を修正したもの
     */
    private static function _tag2Entity($matches)
    {
        $value = $matches[1];
        switch(true){
            //大文字のみ(定数として解釈)
            case preg_match('/[A-Z_]+/', $value) && defined($value):
                $value = constant($value);
                break;
            //Samurai_Config検索
            case is_scalar(Samurai_Config::get($value)):
                $value = Samurai_Config::get($value);
                break;
            //元に戻す
            default:
                $value = '%'.$value.'%';
                break;
        }
        return $value;
    }
    
    
    /**
     * マージ。
     * @access     private
     */
    private static function _merge($array1, $array2)
    {
        try {
            $Utility = Samurai::getContainer()->getComponent('Utility');
            $result = $Utility->array_merge($array1, $array2);
        } catch(Samurai_Exception $E){
            $result = $array1;
            if(is_array($array2)){
                foreach($array2 as $_key => $_val){
                    if(!isset($result[$_key])){
                        $result[$_key] = $_val;
                    } else {
                        $result[$_key] = self::_merge($result[$_key], $_val);
                    }
                }
            } else {
                $result = $array2;
            }
        }
        return $result;
    }
    
    
    /**
     * yamlの名前を置き換える。
     * @access     private
     * @param      string  $yaml_file   元のファイル名
     * @param      string  $postfix     差し込み文字列
     * @return     string  差し込まれたファイル名
     */
    private static function _replaceYamlName($yaml_file, $postfix)
    {
        $info = pathinfo($yaml_file);
        if(isset($info['extension'])){
            $filename = preg_replace(sprintf('/\.%s$/', $info['extension']), '', $info['basename']);
            $filename = sprintf('%s.%s.%s', $filename, $postfix, $info['extension']);
        } else {
            $filename = sprintf('%s.%s', $info['basename'], $postfix);
        }
        return $info['dirname'].'/'.$filename;
    }
}
