<?
/**
 * Samurai_Request_Session
 * 
 * Session管理を行うクラス。
 * DBセッションを利用する場合は、ActiveGateway必須。
 * 
 * @package    Samurai
 * @subpackage Samurai.Request
 * @copyright  Befool, Inc
 * @author     Satoshi Kiuchi <satoshi.kiuchi@befool.co.jp>
 * @license    http://opensource.org/licenses/bsd-license.php  The modified BSD License
 */
Samurai_Loader::loadByClass('Samurai_Request_Parameter');
class Samurai_Request_Session extends Samurai_Request_Parameter
{
    private
        /** @var        string  SESSIONタイプ(file|db) */
        $_type  = 'file',
        /** @var        string  ActiveGateway識別子 */
        $_dsn   = 'session',
        /** @var        string  ActiveGatewayセッションテーブル名エイリアス */
        $_alias = 'session',
        /** @var        object  ActiveGatewayインスタンス */
        $AG;
    
    
    /**
     * コンストラクタ。
     * @access    public
     */
    public function __construct()
    {
        
    }
    
    
    /**
     * 値のセット。
     * @access     public
     */
    public function setParameter($key, $value)
    {
        parent::setParameter($key, $value);
        $keys = explode('.', $key);
        $key_str = '';
        foreach($keys as $key){
            $key_str .= (is_numeric($key) || !$key) ? "[{$key}]" : "['{$key}']" ;
        }
        $script = "\$_SESSION{$key_str} = \$value;";
        eval($script);
    }
    
    
    /**
     * 値の削除。
     * @access     public
     */
    public function delParameter($key)
    {
        parent::delParameter($key);
        $keys = explode('.', $key);
        $key_str = '';
        foreach($keys as $key){
            if($key == '') return false;
            $key_str .= (is_numeric($key)) ? "[{$key}]" : "['{$key}']" ;
        }
        $script = "unset(\$_SESSION{$key_str});";
        eval($script);
    }
    
    
    /**
     * 値の取得と削除を同時に行う
     * @access     public
     * @param      string  $key
     */
    public function getAndDel($key)
    {
        $value = $this->getParameter($key);
        $this->delParameter($key);
        return $value;
    }
    
    
    
    
    
    /**
     * セッション処理を開始する。
     * @access    public
     */
    public function start()
    {
        if(!isset($_SESSION)){
            $this->_setHandler();
            session_start();
            $this->import($_SESSION);
        }
    }
    
    
    /**
     * セッション処理を終了
     * name_spaceが指定された場合は、そのname_spaceの範囲だけクリア。
     * @access     public
     * @param      string  $name_space   名前空間
     */
    public function close($name_space='')
    {
        if($name_space){
            $this->delParameter($name_space);
        } else {
            $_SESSION = array();
            $this->_parameters = array();
            session_destroy();
        }
    }
    
    
    /**
     * セッション名を設定/返却。
     * @access     public
     * @param      string  $name
     * @return     string   セッション名
     */
    public function name($name=NULL)
    {
        return $name === NULL ? session_name() : session_name($name) ;
    }
    
    
    /**
     * セッションIDを設定/返却。
     * @access     public
     * @param      string   $id
     * @return     string   セッションID
     */
    public function id($id=NULL)
    {
        return $id === NULL ? session_id() : session_id($id) ;
    }
    
    
    /**
     * save_pathをセット。
     * @access    public
     * @param     string  $save_path   save_path
     */
    public function savePath($save_path=NULL)
    {
        return $save_path === NULL ? session_save_path() : session_save_path($save_path) ;
    }
    
    
    /**
     * cache_limiterをセット。
     * @access    public
     * @param     string  $cache_limiter   cache_limiter
     */
    public function cacheLimiter($cache_limiter=NULL)
    {
        return $cache_limiter === NULL ? session_cache_limiter() : session_cache_limiter($cache_limiter) ;
    }
    
    
    /**
     * cache_expireをセット。
     * @access    public
     * @param     int     $cache_expire   cache_expire
     */
    public function cacheExpire($cache_expire=NULL)
    {
        return $cache_expire === NULL ? session_cache_expire() : session_cache_expire($cache_expire) ;
    }
    
    
    /**
     * use_cookieをセット。
     * @access    public
     * @param     boolean $use_cookies   クッキーを使うかどうか
     */
    public function setUseCookies($use_cookies)
    {
        ini_set('session.use_cookies', $use_cookies ? 1 : 0 );
    }
    
    
    /**
     * cookie_lifetimeをセット。
     * @access    public
     * @param     int     $cookie_lifetime   cookie_lifetime
     */
    public function setCookieLifetime($cookie_lifetime)
    {
        $cookie_params = session_get_cookie_params();
        session_set_cookie_params($cookie_lifetime, $cookie_params['path'], $cookie_params['domain'], $cookie_params['secure']);
    }
    
    
    /**
     * cookie_pathをセット。
     * @access    public
     * @param     string  $cookie_path   cookie_path
     */
    public function setCookiePath($cookie_path)
    {
        $cookie_params = session_get_cookie_params();
        session_set_cookie_params($cookie_params['lifetime'], $cookie_path, $cookie_params['domain'], $cookie_params['secure']);
    }
    
    
    /**
     * cookie_domainをセット。
     * @access    public
     * @param     string  $cookie_domain   cookie_domain
     */
    public function setCookieDomain($cookie_domain)
    {
        $cookie_params = session_get_cookie_params();
        session_set_cookie_params($cookie_params['lifetime'], $cookie_params['path'], $cookie_domain, $cookie_params['secure']);
    }
    
    
    /**
     * cookie_secureをセット。
     * @access    public
     * @param     boolean $cookie_secure   cookie_secure
     */
    public function setCookieSecure($cookie_secure)
    {
        if(preg_match('/^true$/i', $cookie_secure) || preg_match('/^secure$/i', $cookie_secure) ||
            preg_match('/^on$/i', $cookie_secure) || $cookie_secure === 1 || $cookie_secure === '1'){
            $cookie_secure = 1;
        } else {
            $cookie_secure = 0;
        }
        $cookie_params = session_get_cookie_params();
        session_set_cookie_params($cookie_params['lifetime'], $cookie_params['path'], $cookie_params['domain'], $cookie_secure);
    }
    
    
    /**
     * gc_*をセットする
     * @access     public
     */
    public function setGCConfig($key, $value)
    {
        ini_set('session.'.$key, $value);
    }
    
    
    
    
    
    /**
     * セッションタイプを設定する。
     * @access     public
     */
    public function setType($type)
    {
        $this->_type = $type;
    }
    
    
    /**
     * セッションタイプを取得する。
     * @access     public
     */
    public function getType()
    {
        return $this->_type;
    }
    
    
    /**
     * DSNをセットする。
     * @access     public
     */
    public function setDsn($dsn)
    {
        $this->_dsn = $dsn;
    }
    
    
    /**
     * DSNを取得する。
     * @access     public
     */
    public function getDsn()
    {
        return $this->_dsn;
    }
    
    
    
    
    
    /**
     * session_handlerを設定に従って設定する。
     * @access     private
     */
    private function _setHandler()
    {
        if($this->_type == 'db'){
            session_set_save_handler(array($this, 'handler_db_open'), array($this, 'handler_db_close'),
                                        array($this, 'handler_db_read'), array($this, 'handler_db_write'),
                                        array($this, 'handler_db_destroy'), array($this, 'handler_db_gc'));
        }
    }
    
    
    /**
     * DB用ハンダラー。
     * セッションをオープンする(DBに接続する)。
     * @access     public
     */
    public function handler_db_open($save_path, $session_name)
    {
        $ActiveGatewayManager = ActiveGatewayManager::singleton();
        $this->AG = $ActiveGatewayManager->getActiveGateway($this->_dsn);
        return true;
    }
    
    
    /**
     * DB用ハンダラー。
     * セッションをクローズする(DBを切断する)。
     * @access     public
     */
    public function handler_db_close()
    {
        //PDOに切断処理はない
        return true;
    }
    
    
    /**
     * DB用ハンダラー。
     * データを読み込む。
     * @access     public
     */
    public function handler_db_read($id)
    {
        $session = $this->AG->findBy($this->_alias, 'session_id', $id);
        $data = $session ? $session->session_data : serialize(array());
        return $data;
    }
    
    
    /**
     * DB用ハンダラー。
     * データを書き込む。
     * @access     public
     */
    public function handler_db_write($id, $data)
    {
        //検索
        $session = $this->AG->findBy($this->_alias, 'session_id', $id);
        //作成
        if(!$session){
            $dto = new stdClass();
            $dto->session_id   = $id;
            $dto->session_data = $data;
            $this->AG->create($this->_alias, $dto);
        } else {
            $session->session_data = $data;
            $this->AG->save($session);
        }
        return true;
    }
    
    
    /**
     * DB用ハンダラー。
     * デストロイ処理。
     * @access     public
     */
    public function handler_db_destroy($id)
    {
        $this->AG->executeQuery("DELETE FROM `{$this->_alias}` WHERE `id` = :id", array(':id' => $id));
        return true;
    }
    
    
    /**
     * DB用ハンダラー。
     * ガベージコレクション処理。
     * @access     public
     */
    public function handler_db_gc($maxlifetime)
    {
        $maxlifetime = $maxlifetime * 60;
        $this->AG->executeQuery("DELETE FROM `{$this->_alias}` WHERE `updated_at` < :updated_at",
                                                            array(':updated_at' => time() - $maxlifetime));
        return true;
    }
}
