package com.sanpudo.formula;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Hashtable;
import java.util.Map;

/**
 * 指定された数式を評価するクラス。
 * 
 * @author Sanpudo.
 */
public class FormulaEvaluator {
	private Environments environments;

	/** コンストラクタ。 */
	public FormulaEvaluator() {
		environments = new Environments();
	}

	/** BigDecimalの小数点以下桁数を指定したコンストラクタ。
	 * @param scale BigDecimalの小数点以下桁数
	 *  */
	public FormulaEvaluator(int scale) {
		this();
		environments.setScale(scale);
	}

	/** BigDecimalの小数点以下桁数と端数処理を指定したコンストラクタ。 
	 * @param roundPoint BigDecimalの端数処理時点
	 * @param scale BigDecimalの小数点以下桁数
	 * @param rMode BigDecimalの端数処理方法
	 * */
	public FormulaEvaluator(RoundPoint rondPoint, int scale, RoundingMode rMode)
			throws FormulaEvaluatorException {
		this();
		environments.setRounding(rondPoint, scale, rMode);
	}

	/** コンストラクタ。 */
	FormulaEvaluator(Environments environments) {
		this.environments = environments;
	}

	/**
	 * 指定された数式の構文をチェックする。不正であればFormulaEvaluatorExceptionをスローする。 正しければ何もしない。
	 * @param formula 式
	 * @exception FormulaEvaluatorException 式が不正である
	 */
	public void parse(String formula) throws FormulaEvaluatorException {
		new Parser(formula).parse(new Lex(environments).analyze(formula));
	}

	/** 指定された数式を評価したdoubleの値を返す。 
	 * @param formula 式
	 * @return 評価結果
	 * @exception FormulaEvaluatorException 式が不正である
	 * */
	public double dEvaluate(String formula) throws FormulaEvaluatorException {
		return new Parser(formula)
				.parse(new Lex(environments).analyze(formula)).value();
	}

	/** 指定された数式を評価したdoubleの値を返す。%0～%9のパラメータを許す。 */
	double dEvaluateParam(String formula) throws FormulaEvaluatorException {
		return new Parser(formula).parse(
				new Lex(environments).analyze(formula, true)).value();
	}

	/** 指定された数式を評価したBigDecimalの値を返す。 
	 * @param formula 式
	 * @return 評価結果
	 * @exception FormulaEvaluatorException 式が不正である
	 * */
	public BigDecimal bdEvaluate(String formula)
			throws FormulaEvaluatorException {
		BigDecimal bd = new Parser(formula).parse(
				new Lex(environments).analyze(formula)).value(
				environments.rounding);
		// 最終端数処理
		return environments.rounding.round(bd);
	}

	/** 指定された数式を評価したBigDecimalの値を返す。%0～%9のパラメータを許す。 */
	BigDecimal bdEvaluateParam(String formula) throws FormulaEvaluatorException {
		BigDecimal bd = new Parser(formula).parse(
				new Lex(environments).analyze(formula, true)).value(
				environments.rounding);
		// 最終端数処理
		return environments.rounding.round(bd);
	}

	/** BigDecimal計算における小数点以下の桁数と端数処理方法を変更する。 
	 * @param scale BigDecimalの小数点以下桁数
	 * @param rMode BigDecimalの端数処理方法
	 * */
	public void setRounding(int scale, RoundingMode rMode) {
		environments.setRounding(scale, rMode);
	}

	/** BigDecimal計算における小数点以下の桁数を変更する。 
	 * @param scale BigDecimalの小数点以下桁数
	 * */
	public void setScale(int scale) {
		environments.setScale(scale);
	}

	/** BigDecimal計算における端数処理のコンテクスト(端数処理タイミング、小数点以下の桁数および端数処理方法)を変更する。 
	 * @param roundPoint BigDecimalの端数処理時点
	 * @param scale BigDecimalの小数点以下桁数
	 * @param rMode BigDecimalの端数処理方法
	 * */
	public void setRounding(RoundPoint roundPoint, int scale, RoundingMode rMode)
			throws FormulaEvaluatorException {
		environments.setRounding(roundPoint, scale, rMode);
	}

	/**
	 * ユーザ定義定数を登録する。overrideがfalseの場合はシステム定義済み定数と同名の定数を登録しようとすると
	 * FormulaEvaluatorExceptionをスローする。
	 * @param name 定数名
	 * @param value 定数の値
	 * @param override trueの場合、システム定義定数と同名の定数の登録を許可する
	 * @exception FormulaEvaluatorException overrideがfalseでシステム定義定数と同名の定数の登録
	 */
	public void defineUserConstant(String name, String value, boolean override)
			throws FormulaEvaluatorException {
		environments.defineUserConstant(name, value, false, override);

	}

	/**
	 * ユーザ定義定数を登録する。システム定義済み定数と同名の定数を登録しようとすると FormulaEvaluatorExceptionをスローする。
	 * @param name 定数名
	 * @param value 定数の値
	 * @exception FormulaEvaluatorException システム定義定数と同名の定数の登録
	 */
	public void defineUserConstant(String name, String value)
			throws FormulaEvaluatorException {
		environments.defineUserConstant(name, value);
	}

	/**
	 * 定義済みの定数(システム定義およびユーザ定義）の一覧をマップで返す。 マップのキーは変数名、値は変数の値の文字列表現である。
	 * @return 定義済みの定数(システム定義およびユーザ定義）の一覧
	 */
	public Map<String, String> getConstants() {
		Hashtable<String, String> h = new Hashtable<String, String>(
				Environments.predefine);
		h.putAll(environments.userDefine);
		return h;
	}

	/**
	 * ユーザ定義関数を定義したオブジェクトを登録する。 システム定義済み関数と同名の定数を登録しようとすると
	 * FormulaEvaluatorExceptionをスローする。不正な関数名を定義するとFunctionExceptionをスローする。
	 * @param func ユーザ定義関数を定義したオブジェクト
	 * @exception FormulaEvaluatorException システム定義済み関数と同盟の関数を登録
	 * @exception FunctionException 関数の名が不正
	 */
	public void defineUserFunction(FunctionImplementation func)
			throws FormulaEvaluatorException, FunctionException {
		environments.defineUserFunction(func);
	}

	/**
	 * 指定した式をユーザ定義関数として登録可能かチェックする。
	 * 不正の場合FormulaEvaluatorExceptionをスローする。正しい場合何もしない。
	 * @param funcName 関数名
	 * @param numberOfArgs 引数の個数
	 * @param formula 関数の式
	 * @exception FormulaEvaluatorException 不正(ユーザ定義関数として登録できない)
	 */
	public void checkUserFunction(String funcName, int numberOfArgs,
			String formula) throws FormulaEvaluatorException {
		FunctionInterpreter.checkFormula(funcName, numberOfArgs, formula,
				this.environments);
	}

	/**
	 * 指定した式をユーザ定義関数として登録する。 不正な式、あるいはシステム定義済み関数と同名の定数を登録しようとすると
	 * FormulaEvaluatorExceptionをスローする。
	 * @param funcName 関数名
	 * @param numberOfArgs 引数の個数
	 * @param formula 関数の式
	 * @exception FormulaEvaluatorException 不正(ユーザ定義関数として登録できない)
	 */
	public void defineUserFunction(String funcName, int numberOfArgs,
			String formula) throws FormulaEvaluatorException {
		defineUserFunction(new FunctionInterpreter(funcName, numberOfArgs,
				formula, this.environments));
	}

	/** 指定した名前のクラスのインスタンスを生成しユーザ定義関数として登録する。 
	 * @param className ユーザ定義関数を定義したクラスのクラス名
	 * @exception FormulaEvaluatorException 不正(ユーザ定義関数として登録できない)
	 * */
	public void loadUserFunction(String className)
			throws FormulaEvaluatorException {
		environments.loadUserFunction(className);
	}

	/** 現在の小数点以下桁数の設定を返す。 
	 * @return 現在のBigDecimalの小数点以下桁数
	 * */
	public int getScale() {
		return environments.rounding.scale;
	}

	/** 現在の端数処理方法の設定を返す。 
	 * @return 現在のBigDecimalの端数処理方法
	 * */
	public RoundingMode getRoundingMode() {
		return environments.rounding.rMode;
	}

	/** 現在の端数処理時点の設定を返す。 
	 * @return 現在のBigDecimalの端数処理時点
	 * */
	public RoundPoint getRoundPoint() {
		return environments.rounding.roundPoint;

	}
}
