package com.sanpudo.formula;

import java.util.Vector;
import java.util.Iterator;

/**
 * パーサー。
 * 
 * @author Sanpudo.
 */
class Parser {

	private String input;

	/** コンストラクタ。 */
	Parser(String input) {
		this.input = input;
	}

	/** トークン列をパースし、ファクターの構造にする。 */
	Token parse(Vector<Token> tokens) throws FormulaEvaluatorException {
		Token current = null;
		Iterator<Token> it = tokens.iterator();
		while (it.hasNext()) {
			Token next = it.next();
			// 開始直後
			if (current == null) {
				if (next.isNonValue()) {
					error(next, Messages.VALUE_EXPECT);
				}
			}
			// 即値、右括弧の後
			else if (current.isValue() || current.isRPar()) {
				if (next.isValue() || next.isContainer()) {
					error(next, Messages.OPERATOR_EXPECT);
				}
				// 演算子
				else if (next.isOperator()) {
					Operator op = (Operator) next;
					Token f1 = current.getF1(op);
					if (f1.parent == null) {
						op.parent = null;
					} else if (f1.parent.isLPar()) {
						((LPar) (f1.parent)).bind(op);
					} else if (f1.parent.isFunction()) {
						((Function) (f1.parent)).reBind(op);
					} else if (f1.parent.isOperator()) {
						((Operator) (f1.parent)).bindF2(op);
					}
					op.bindF1(f1);
				}
				// 右括弧
				else if (next.isRPar()) {
					// 対応する左括弧を完結
					Container container = current.recentOpenContainer();
					if (container == null) {
						error(next, Messages.NO_CORR_PAREN);
					}
					if (!container.filled()) {
						if (container.isLPar()) {
							error(next, Messages.VALUE_EXPECT);
						} else {
							error(next, Messages.NEXT_VALUE_EXPECT);
						}
					}
					container.close((RPar) next);
				}
				// 区切り文字
				else if (next.isDelimiter()) {
					Function func = current.recentOpenFunc();
					if (func == null) {
						error(next, Messages.INVALID_DELIM);
					}
					if (func.filled()) {
						error(next, Messages.RPAR_EXPECT);
					}
					((Delimiter) next).bindToParent(current);
				}
			}
			// 左括弧の後
			else if (current.isLPar()) {
				if (next.isNonValue()) {
					error(next, Messages.VALUE_EXPECT);
				}
				((LPar) current).bind(next);
			}
			// 関数の後
			else if (current.isFunction()) {
				if (next.isOperator() || next.isDelimiter()) {
					error(next, Messages.VALUE_EXPECT);
				}
				// 右括弧
				if (next.isRPar()) {
					if (!((Function) current).filled()) {
						error(next, Messages.VALUE_EXPECT);
					}
					((Function) current).close((RPar) next);
					// Value
				} else {
					if (((Function) current).filled()) {
						error(next, Messages.RPAR_EXPECT);
					}
					((Function) current).bind(next);
				}
			}
			// 演算子の後
			else if (current.isOperator()) {
				if (next.isNonValue()) {
					error(next, Messages.VALUE_EXPECT);
				}
				((Operator) current).bindF2(next);
			}
			// 区切り文字の後
			else if (current.isDelimiter()) {
				if (next.isNonValue()) {
					error(next, Messages.VALUE_EXPECT);
				}
				Function func = current.recentOpenFunc();
				func.bind(next);
			}
			current = next;
		}
		// nullチェック
		if (current == null) {
			throw new FormulaEvaluatorException(Messages.NULL_INPUT, input,
					input.length());
		}
		// 完全であるかチェック
		Token root = current.root();
		if (!root.isComplete()) {
			throw new FormulaEvaluatorException(root.imcompleteMessage(),
					input, input.length());
		}
		return root;
	}

	private void error(Token token, String msg)
			throws FormulaEvaluatorException {
		throw new FormulaEvaluatorException(msg, input, token.location);
	}
}
