/************************************************
*                                               *
*       Back Propergation Simurator(BPS)        *
*             subroutine package                *
*               Ver                             *
*         coded         in Nov.5 1990           *
*         coded by      K.Kuroda                *
*************************************************
*                                               *
*       filename sigmoid.c                      *
*                                               *
************************************************/
#include        "BPS.h"
#include        "BPStrace.h"

#define	PLOT		'P'
#define	HISTGRAM	'H'

static char          plot_mode;
static int           lay_no, unit_no, plot_n;
static float        *net_data, *active_data;

/************************************************
 *	set parameters
 ************************************************/
void
set_param()
{
  char *str;

  lay_no    =   (int)GetScalar(0);
  unit_no   =   (int)GetScalar(1);
  str       =        GetString(2);
  
  if ((lay_no < 0) || ((NumOfLayer - 1) < lay_no))    exit(20);
  if (FuncBias[lay_no][0] != SIGMOID)                 exit(20);

  if ((unit_no < 1) || (NumOfCell[lay_no] < unit_no)) exit(24);

  plot_mode = toupper(str[0]);
  switch (plot_mode) {
  case PLOT:     plot_n =   (int)GetScalar(3);  break;
  case HISTGRAM: plot_n = (float)GetScalar(3);  break;
  default:
    exit(22);
  }
}

/************************************************
 * allocate to array
 ************************************************/
void
alloc_data()
{
  net_data    = (float*)malloc(NumOfPtrn * sizeof(float));
  if (net_data == NULL)    exit(23);

  active_data = (float*)malloc(NumOfPtrn * sizeof(float));
  if (active_data == NULL) exit(23);
}

/***********************************************
 * network test
 ***********************************************/
void
net_test()
{
  int     ptrn;
  cel_t  *cur_cel;
  
  for (ptrn = 0; ptrn < NumOfPtrn; ptrn++) {
    test_network(InputData[ptrn]);
    cur_cel = BPNet[lay_no][unit_no].CellNode;
    net_data[ptrn] = cur_cel->Net;
    active_data[ptrn] = cur_cel->Active;
  }
}

/************************************************
 * draw sigmoid curve
 ************************************************/
void  SigmoidPlot()
{
  char     title[2][10];
  int      i, j, ptrn, datapoint, nn, jj;
  float    min_net, max_net, min_leng; 
  float    char_size, scale_width;
  float    act_sum, act_mean, act_vari, work, act_stand;
  float    delta, dat, hist_max, act_area;
  float   *data, *sigmoid_data;
  float   *histdata1, *histdata2;
  double   x;
  
  gpen(GpmCont.fLineWidth,GpmCont.gLineType, 0);
  gnewpen(GpmCont.fColor);
  gorigin(GpmCont.xOrigin, GpmCont.yOrigin);
  gfactor(GpmCont.factor);
  gfont(GpmCont.fontType);

  /* SET MIMIMUM NET VALUE AND MAXIMUM NET VALUE */
  max_net = 5.0;
  for (ptrn = 0; ptrn < NumOfPtrn; ptrn++) {
    if (max_net < net_data[ptrn])
      max_net = net_data[ptrn];
    if (max_net < -net_data[ptrn])
      max_net = -net_data[ptrn];
  }
  max_net *= 10;
  max_net = (int) (max_net + 0.5);
  max_net /= 10;
  min_net = -max_net;
  
  /* SET SIGMOID DATA */
  datapoint = (max_net - min_net) / 0.1;

  sigmoid_data = (float *) malloc((datapoint + 1) * sizeof(float));
  if (sigmoid_data == NULL)
    exit(46);

  data = (float *) malloc((datapoint + 1) * sizeof(float));
  if (data == NULL)
    exit(46);
  
  for (i = 0; i <= datapoint; i++) {
    x = min_net + i * 0.1;
    data[i] = x;
    sigmoid_data[i] = Sigmoid(x);
  }
  
  /* DRAW AXIS */
  strcpy(title[0], "NET");
  strcpy(title[1], "ACTIVATION");
  if (GpmCont.xSize > GpmCont.ySize)
    min_leng = GpmCont.ySize;
  else
    min_leng = GpmCont.xSize;
  char_size = min_leng * 0.04;
  scale_width = (int) ((-min_net < max_net) ? -min_net : max_net);
  
  gaxis(0.0, 0.0, GpmCont.xSize, 0.0, min_net, max_net, scale_width,
	1, 1, title[0], char_size, 0, GpmCont.ySize);
  gaxis(0.0, 0.0, GpmCont.ySize, (double)M_PI/2.0, 0.0, 1.0, 0.5,
	1, 0, title[1], char_size, 0, GpmCont.ySize);
  
  gfsize(GpmCont.xSize, GpmCont.ySize);
  gframe();
  
  /* DRAW SIGMOID CURVE */
  glimit(0.0, -0.1, GpmCont.xSize, GpmCont.ySize + 0.1);
  gpen(GpmCont.gLineWidth,GpmCont.gLineType, 0);
  gnewpen(GpmCont.gColor);
  gscale(min_net, 0.0, max_net, 1.0);
  graph(data, sigmoid_data, datapoint + 1, 1, 0, 0, 0.0);
  
  glimit(0.0, 0.0, 0.0, 0.0);
  gnewpen(GpmCont.fColor);
  gpen(GpmCont.fLineWidth, 1, 0);

  act_area = 2.2 * GpmCont.xSize / (max_net * 2.0);
  act_area = GpmCont.xSize / 2.0 - act_area;
  gplot(act_area, 0.0, 0);
  gplot(act_area, GpmCont.ySize, 2);

  act_area = GpmCont.xSize - act_area;
  gplot(act_area, 0.0, 0);
  gplot(act_area, GpmCont.ySize, 2);
  
  /* SET MEAN VALUE, VARIANCE AND STANDARD DEVIATION */
  act_sum = 0.0;
  for (i = 0; i < NumOfPtrn; i++)
    act_sum += active_data[i];
  act_mean = act_sum / (float) NumOfPtrn;
  
  act_sum = 0.0;
  for (i = 0; i < NumOfPtrn; i++) {
    work = active_data[i] - act_mean;
    act_sum += work * work;
  }
  
  act_vari = act_sum / (float) NumOfPtrn;
  act_stand = (float) sqrt((double) act_vari);
  
  printf("\n*** MEAN VALUE         = %15.5g\n", act_mean);
  printf("*** VARIANCE           = %15.5g\n", act_vari);
  printf("*** STANDARD DEVIATION = %15.5g\n", act_stand);
  
  if (plot_mode == PLOT) {
    /* PLOT */
    gnewpen(GpmCont.fColor);
    glimit(0.0, -0.1, GpmCont.xSize, GpmCont.ySize + 0.1);
    gscale(min_net, 0.0, max_net, 1.0);
    graph(net_data, active_data, NumOfPtrn, 1, 1, plot_n, char_size / 2);
  } else {
    /* HISTOGRAM */
    delta = (max_net - min_net) / (float) plot_n;
    nn = plot_n * 2 + 2;
    plot_n++;
    
    histdata1 = (float *) malloc(nn * sizeof(float));
    if (histdata1 == NULL)
      exit(46);
    histdata2 = (float *) malloc(nn * sizeof(float));
    if (histdata2 == NULL)
      exit(46);
    
    for (i = 0; i < plot_n; i++) {
      j = i * 2;
      dat = delta * (float) i + min_net;
      histdata1[j] = dat;
      histdata1[j + 1] = dat;
    }
    
    for (i = 0; i < nn; histdata2[i++] = 0.0);
    
    for (ptrn = 0; ptrn < NumOfPtrn; ptrn++) {
      if (net_data[ptrn] < min_net || net_data[ptrn] > max_net)
	continue;
      jj = (net_data[ptrn] - min_net) / delta;
      jj = jj * 2 + 1;
      histdata2[jj]++;
      histdata2[jj + 1]++;
    }
    
    hist_max = 0;
    for (i = 0; i < nn; i = i + 2)
      hist_max = (hist_max > histdata2[i]) ? hist_max : histdata2[i];
    hist_max = (int) (hist_max * 1.5 + 0.5);
    
    gnewpen(GpmCont.fColor);
    gpen(GpmCont.fLineWidth, GpmCont.fLineType, 0);
    strcpy(title[1], "FREQUENCY");
    gscale(min_net, 0.0, max_net, hist_max);
    scale_width = (int) (hist_max / 2.0);
    gaxis(GpmCont.xSize, 0.0, GpmCont.ySize, (double)M_PI/2.0, 0.0,
	  hist_max, scale_width, 2, 1, title[1],
	  char_size, 0, GpmCont.ySize);
    graph(histdata1, histdata2, nn, 1, 0, 0, 0);
  }
  
  free(sigmoid_data);
  glimit(0.0, 0.0, 0.0, 0.0);
  gflush();
}


/************************************************
  sigmoid plot
  ************************************************/
int
main()
{
  /* GET PARAMETERS */
  read_syscom();
  rebps();
  regpm();
  
  GetStructureParameters();
  GetTestingParameters();
  set_param();
  
  /* SYSTEM INITIALIZE */
  rec_system_initialize();
  MakeNetwork();

  ReadWeight2(TestWgtFile, WgtHistoryNo);

  alloc_data();
  
  /* NETWORK TEST */
  net_test();
  
  /* DRAW SIGMOID CURVE AND PLOT */
  if (checktodvi())
    writegpm();
  gopen(GpmCont.paper, GpmCont.orientation, GpmCont.device,
	GpmCont.winNum);
  gnewpen(GpmCont.fColor);
  gpen(GpmCont.fLineWidth,GpmCont.fLineType, 0);
  gorigin(GpmCont.xOrigin, GpmCont.yOrigin);
  gfactor(GpmCont.factor);
  
  SigmoidPlot();
  
  gflush();
  gclose();
  
  /* SYSTEM END */
  BreakNetwork();
  rec_system_end();

  wrbps();
  write_syscom();
  return 0;
}
