/*
 * Copyright (c) 2000, Toshihiro Aoyama All rights reserved.
 * 
 * $Id: abf2satellite.c,v 1.1.1.1 2004/03/26 14:57:15 orrisroot Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/stat.h>

#include "SL_macro.h"
#include "SL_cmd.h"

#include "abf2satellite.h"

static char    *
rename_filename(char *in, char *out, char *suffix)
{
	int             i, j;

	/* generate save file name */
	i = 0;
	while (in[i] != '.') {
		out[i] = in[i];
		i++;
	}
	out[i] = '.';
	for (j = 0; j < 3; j++) {
		out[i + j + 1] = suffix[j];
	}
	out[i + j + 1] = '\0';

	return out;
}

int
main(int argc, char **argv)
{
	struct stat     stat_buf;
	int             data_fd;
	int             n, i, j = 0;
	char           *text;
	char           *p;
	FILE           *head_fp;
	char            *out_filename;
	char            *cnf_filename;
	char            *in_filename;

	int             sl_index[2];
	Buffer         *sl_buffer;
	int             k = 0;

	int             NumSamplesPerEpisode;
	int             EpisodesPerRun;

	/* header structure */
	FileID_Size_Info *id_size_info;
	File_Structure *file_structure;
	Display_Parameters *disp_param;
	Trial_Hierarchy_Info *tr_info;
	Environmental_Info *env_info;
	Multi_Channel_Info *mc_info;
	DAC_Output_File *DAC_out_file;
	Hardware_Info  *hw_info;

	/* initialize for SATELLITE */
	read_syscom();
	in_filename=GetString(0);
	if(in_filename==NULL){
	  exit(1);
	}

	/* initialize filename */
	out_filename = (char *) malloc(sizeof(char)*ABF_FILENAMELEN);
	cnf_filename = (char *) malloc(sizeof(char)*ABF_FILENAMELEN);
	rename_filename(in_filename, out_filename, "dat");
	rename_filename(in_filename, cnf_filename, "cnf");

	/* open header output file */
	if ((head_fp = fopen(cnf_filename, "w")) == NULL) {
		fprintf(stderr, "%s file open failed\n", cnf_filename);
		exit(1);
	}
	/* open ABF file */
	data_fd = open(in_filename, O_RDONLY);
	if (data_fd == -1) {
		fprintf(stderr, "%s file open failed\n", in_filename);
		exit(1);
	}
	if (fstat(data_fd, &stat_buf) == -1) {
		fprintf(stderr, "stat failed\n");
		exit(1);
	}
	n = stat_buf.st_size;
	text = (char *) mmap(NULL, n + 1, PROT_READ | PROT_WRITE, MAP_PRIVATE, data_fd, 0);
	if ((int) text == -1) {
		fprintf(stderr, "mmap failed\n");
		exit(1);
	}
	text[n] = 0;

	/* header structure initialization */
	id_size_info = (FileID_Size_Info *) malloc(sizeof(FileID_Size_Info));
	file_structure = (File_Structure *) malloc(sizeof(File_Structure));
	disp_param = (Display_Parameters *) malloc(sizeof(Display_Parameters));
	hw_info = (Hardware_Info *) malloc(sizeof(Hardware_Info));
	tr_info = (Trial_Hierarchy_Info *) malloc(sizeof(Trial_Hierarchy_Info));
	env_info = (Environmental_Info *) malloc(sizeof(Environmental_Info));
	mc_info = (Multi_Channel_Info *) malloc(sizeof(Multi_Channel_Info));
	DAC_out_file = (DAC_Output_File *) malloc(sizeof(DAC_Output_File));

	/* read File ID and Size information */
	Read_FileID_Size_Info(text, id_size_info);
	Print_FileID_Size_Info(head_fp, id_size_info);

	/* read File Structure */
	Read_File_Structure(text, file_structure);
	Print_File_Structure(head_fp, file_structure);

	/* read Trial_Hierarchy_Info */
	Read_Trial_Hierarchy_Info(text, tr_info);
	Print_Trial_Hierarchy_Info(head_fp, tr_info);

	/* read Hulti_Channel_Info */
	Read_Multi_Channel_Info(text, mc_info);
	Print_Multi_Channel_Info(head_fp, mc_info);

	/* read Display Parameters */
	Read_Display_Parameters(text, disp_param);
	Print_Display_Parameters(head_fp, disp_param);

	/* read Hardware Information */
	Read_Hardware_Info(text, hw_info);
	Print_Hardware_Info(head_fp, hw_info);

	/* read Environmental_Information */
	Read_Environmental_Info(text, env_info);
	Print_Environmental_Info(head_fp, env_info);

	/* read DAC Output File */
	Read_DAC_Output_File(text, DAC_out_file);
	Print_DAC_Output_File(head_fp, DAC_out_file);

	fclose(head_fp);

	NumSamplesPerEpisode = tr_info->NumSamplesPerEpisode;
	/* EpisodesPerRun=tr_info->EpisodesPerRun */
	EpisodesPerRun = id_size_info->ActualEpisodes;

	/* allocate SATELLITE buffer */
	sl_buffer =
		(Buffer *) malloc(sizeof(Buffer) * NumSamplesPerEpisode * EpisodesPerRun);
	if (sl_buffer == NULL) {
		fprintf(stderr, "out of memory\n");
		exit(1);
	}
	/* Read DATA */
	p = text + ABF_BLOCKSIZE * file_structure->DataSectionPtr;
	for (j = 0; j < EpisodesPerRun; j++) {
		for (i = 0; i < NumSamplesPerEpisode; i++) {
			switch (file_structure->DataFormat) {
			case ABF_DATAFORMAT_INT:
				sl_buffer[k] =
					*(short *) (p + (i + j * NumSamplesPerEpisode) * 2)
					* (hw_info->ADCRange)
					/ (hw_info->ADCResolution)
					/ mc_info->InstrumentScaleFactor[0];
				break;
			case ABF_DATAFORMAT_FLOAT:
				sl_buffer[k] =
					*(float *) (p + (i + j * NumSamplesPerEpisode) * 4);
				break;
			default:
				exit(1);
			}
			k++;
		}
	}

	/* save satellite buffer file */
	sl_index[0] = EpisodesPerRun;
	sl_index[1] = NumSamplesPerEpisode;

	ChangeDataSize(8);
	WriteFile(out_filename, 2, sl_index, (char *) sl_buffer);

	return 0;
}
