#include <stdio.h>
#include <string.h>
#include "SL_macro.h"
#include "SL_cmd.h"

typedef struct {
  int type;    /* Computer Type */
  int rows;    /* Number of Data rows */
  int cols;    /* Number of Data columns */
  int imgf;   /* Flag Complex ( 0 = REAL , 1 = COMPLEX ) */
  int namelen; /* Length of Name for Data */
} MAT_Header ;  /* Matlab Version 4 format */

typedef struct {
  char fname[64];
  int dim,index[10],length;
  Buffer *data;
} Satellite_Data;


int isBigEndian(int type)
{
  if (type > 9999 || type < 0)
    return 1;
  else return 0;
}

void swap_data( char *data, int nbyte)
{
  int i;
  char temp;
  for(i=0;i<nbyte/2;i++){
    temp=data[i];
    data[i]=data[nbyte-i-1];
    data[nbyte-i-1]=temp;
  }
}

int read_mat_file_header(FILE *fp,MAT_Header *mat,int *swap)
{
  *swap=0;
  fread(&(mat->type),4,1,fp);
  fread(&(mat->rows),4,1,fp);
  fread(&(mat->cols),4,1,fp);
  fread(&(mat->imgf),4,1,fp);
  fread(&(mat->namelen),4,1,fp);
  if(feof(fp))  return 1;
  if ( isBigEndian(mat->type) && mat->type != 0 ){
    *swap=1;
    swap_data((char *) &(mat->type),4);
    swap_data((char *) &(mat->rows),4);
    swap_data((char *) &(mat->cols),4);
    swap_data((char *) &(mat->imgf),4);
    swap_data((char *) &(mat->namelen),4);
  }
#ifdef DEBUG
  fprintf(stdout,"type : %d\n",mat->type);
  fprintf(stdout,"rows : %d\n",mat->rows);
  fprintf(stdout,"cols : %d\n",mat->cols);
  fprintf(stdout,"imgf : %d\n",mat->imgf);
  fprintf(stdout,"len  : %d\n",mat->namelen);
#endif
  return 0;
}

int main()
{
  register int i,j,k;
  int swap,type,prec,order,mach,n,m;
  char *fname;
  FILE *fp;
  char tmp[64];
  union {
    double d;
    int  i;
    float  f;
    short  sh;
    unsigned short  us;
    unsigned char c;
  } temp ;

  MAT_Header mat;
  Satellite_Data sd;

  read_syscom();

  /* Get File Name */
  fname=GetString(0);
  if(fname==NULL)
    exit(1);

  /* Input Matlab File Open */
  fp=fopen(fname,"r");
  if(fp==NULL)
    exit(2);

  while(!read_mat_file_header(fp,&mat,&swap)){
    type = mat.type % 10;  /* Full, sparse, etc */
                           /* 1=string */
    mat.type /= 10;
    prec = mat.type % 10;  /* double, float, int, short, ushort, uchar */
    mat.type /= 10;
    order = mat.type % 10; /* Row or column major ordering */
    mat.type /= 10;
    mach = mat.type % 10;  /* Endian type */
                           /* 0=little, 1=big, 2=vax_d, 3=vax_g, 4=cray */
#ifdef DEBUG
  fprintf(stdout,"data : %d\n",type);
  fprintf(stdout,"prec : %d\n",prec);
  fprintf(stdout,"order : %d\n",order);
  fprintf(stdout,"mach : %d\n",mach);
#endif
    if ( prec == 5 )
      exit(4); /* No Supported String Data */
    if ( type!=0 && type!=1 )
      exit(4); /* can't read sparse matrices */
    if ( mach!=0 && mach!=1 )
      exit(4); /* No Supported Endian type */
    if ( order == 0 ){
      m=mat.cols;
      n=mat.rows;
    } else {
      n=mat.cols;
      m=mat.rows;
    }

    fread(tmp,mat.namelen,1,fp);

    sd.length=mat.rows*mat.cols;
    sd.dim=2;
    sd.index[0]=mat.rows;
    sd.index[1]=mat.cols;

    for(k=0;k<mat.imgf+1;k++){
      sd.data=AllocBuffer(sd.length);
      if(mat.imgf){ /* If Complex Data */
	if(k==0){
	  sprintf(sd.fname,"mat_%s_r.dat",tmp);
	} else {
	  sprintf(sd.fname,"mat_%s_i.dat",tmp);
	}
      } else {
	sprintf(sd.fname,"mat_%s.dat",tmp);
      }
      for(i=0;i<n;i++){
	for(j=0;j<m;j++){
	  switch(prec){
	  case 0: /* double 8 */
	    fread(&temp.d,sizeof(double),1,fp);
	    if(swap)swap_data((char *)&temp.d,8);
	    sd.data[i*m+j]=temp.d;
	    break;
	  case 1: /* float 4 */
	    fread(&temp.f,sizeof(float),1,fp);
	    if(swap)swap_data((char *)&temp.f,4);
	    sd.data[i*m+j]=temp.f;
	    break;
	  case 2: /* int 4 */
	    fread(&temp.i,sizeof(int),1,fp);
	    if(swap)swap_data((char *)&temp.i,4);
	    sd.data[i*m+j]=temp.i;
	    break;
	  case 3: /* short 2 */
	    fread(&temp.sh,sizeof(short),1,fp);
	    if(swap)swap_data((char *)&temp.sh,2);
	    sd.data[i*m+j]=temp.sh;
	    break;
	  case 4: /* unsigned short 2 */
	    fread(&temp.us,sizeof(unsigned short),1,fp);
	    if(swap)swap_data((char *)&temp.us,2);
	    sd.data[i*m+j]=temp.us;
	    break;
	  }
#ifdef DEBUG
	  fprintf(stdout,"sd.data[i*m+j]=%f\n",sd.data[i*m+j]);
#endif
	}
      }
      /* Satellite File Object Data Output */
      ChangeDataSize(8);
      fprintf(stdout,"Output Satellite File Object : \"%s\"\n",sd.fname);
      WriteFile(sd.fname, sd.dim, sd.index, (char *) sd.data);
      FreeBuffer(sd.data);
    }
#ifdef DEBUG
  fprintf(stdout,"OK!\n");
#endif
  }
  fclose(fp);
  return 0;
}
