  /*****************************************************
   *      Calculate EIGEN VECTOR & EIGEN VALUE
   *
   *      EIGEN  X,VEC,VAL
   *            X : MATRIX
   *          VEC : OUTPUT EIGEN VECTOR
   *          VAL : OUTPUT EIGEN VALUE
   *
   *          Jul 24, 1997 by Dora(T.Hayasaka)
   *          Nov  8, 1999 by Dora(T.Hayasaka)
   *          Aug  1, 2001 by Dora(T.Hayasaka)
   *****************************************************/


#include <stdio.h>
#include <math.h>
#include "SL_macro.h"
#include "SL_cmd.h"

#define PI 3.141592654
#define MAX_ITERATION 50
#define MINIMA 1e-8

double    jacobi  _ANSI_ARGS_((double **inputMat, double **eigenMat, 
			       double *eigenVal, int dim));
double**  alloc2D _ANSI_ARGS_((int c1, int c2));
void      free2D  _ANSI_ARGS_((double **buf, int c));


int
main()
{
  Buffer  *a, *val, *vec, code, **inputMat, **eigenMat, *eigenVal;
  int     index[MAX_INDEX], index3[MAX_INDEX];
  int     dim, dim3, n, i, j, val_id, vec_id;
  int     ct, ct2;

  read_syscom();
  a = GetSeries( 0, &dim, index );
  if ( a == NULL )
    exit(4);

  if ( dim != 2 || index[0] != index[1] )
    exit(7);

  vec_id = GetBufferID(1);
  val_id = GetBufferID(2);
  if ( vec_id <= 0 || val_id <= 0 )  exit(17);
  n = index[0] ; 
  val = AllocBuffer( n );
  vec = AllocBuffer( n * n );
  if ( val == NULL || vec == NULL )
    exit(8);

  dim3 = 1;
  index3[0] = n;

  eigenVal = AllocBuffer( n );
  inputMat = alloc2D(n,n);
  eigenMat = alloc2D(n,n);

  ct = 0;
  for( i=0; i<n; i++ )
    for ( j=0; j<n; j++ )
      inputMat[i][j] = a[ct++];

  for ( i=0; i<n; i++ ) {
    eigenVal[i] = inputMat[i][i];
    for ( j=i; j<n; j++ ) {
      eigenMat[i][j] = eigenMat[j][i] = 0.0;
      if ( i == j ) eigenMat[i][j] = 1.0;
    }
  }

  code = jacobi( inputMat, eigenMat, eigenVal, n );

  ct = ct2 = 0;
  for( i=0; i<n; i++ ){
    val[ct2++] = eigenVal[i];
    for( j=0; j<n; j++ ){
      vec[ct++] = eigenMat[i][j];
    }
  }

  if ( WriteBuffer( vec_id, 2, index, vec ) == -1 ) exit(3);
  if ( WriteBuffer( val_id, 1, index3, val ) == -1 ) exit(3);

  FreeBuffer( a );
  FreeBuffer( vec );
  FreeBuffer( val );
  free2D(inputMat,dim);
  free2D(eigenMat,dim);
  free(eigenVal);

  ReturnScalar( code );
  write_syscom();

  return 0;
}

double
jacobi(inputMat,eigenMat,eigenVal,dim)
     double **inputMat;
     double **eigenMat;
     double *eigenVal;
     int    dim;
{
  int i,j,k,loop;
  int idx1, idx2;
  double tmp, theta;
  double **rotateMat, **tmp1Mat;


  rotateMat = alloc2D(dim,dim);
  tmp1Mat   = alloc2D(dim,dim);

  for( loop=0; loop<MAX_ITERATION; loop++ ){

    /* Initializing RotateMat[dim][dim] */
    for ( i=0; i<dim; i++ ) {
      for ( j=i; j<dim; j++ ) {
	rotateMat[i][j] = rotateMat[j][i] = 0.0;
	if ( i == j ) rotateMat[i][j] = 1.0;
      }
    }

    /* Finding maximum of elements in inputMat[dim][dim] */
    tmp = 0.0;
    idx1 = 0;
    idx2 = 0;
    for ( i=0; i<dim-1; i++ ){
      for ( j=i+1; j<dim; j++ ){
	if ( tmp < fabs(inputMat[i][j]) ) {
	  tmp = fabs(inputMat[i][j]);
	  idx1 = i;
	  idx2 = j;
	}
      }
    }
    if ( tmp < MINIMA ) {
      printf("LOOP = %d\n",loop+1);
      free2D(rotateMat,dim);
      free2D(tmp1Mat,dim);
      return 0.0; 
    }

    /* Calculating Rotating Angle Theta */
    if ( (inputMat[idx1][idx1]-inputMat[idx2][idx2]) < MINIMA ) {
      theta = PI / 4.;
      if (inputMat[idx1][idx2] < 0) theta *= -1.;
    }else{
      theta = 0.5 * atan(2.*inputMat[idx1][idx2] / 
			 (inputMat[idx1][idx1]-inputMat[idx2][idx2])); 
    }

    /* Reproducing RotateMat[dim][dim] */   
    rotateMat[idx1][idx1] =  cos(theta);
    rotateMat[idx1][idx2] = -sin(theta);
    rotateMat[idx2][idx1] =  sin(theta);
    rotateMat[idx2][idx2] =  cos(theta);

    /* Calculating rotateMat * inputMat * rotateMat */
    for ( i=0; i<dim; i++ )
      for ( j=0; j<dim; j++ )
	tmp1Mat[i][j] = inputMat[i][j];

    tmp1Mat[idx1][idx1] = inputMat[idx1][idx1] * pow(cos(theta),2.)
      + 2.*inputMat[idx1][idx2] * cos(theta) * sin(theta)
	+ inputMat[idx2][idx2] * pow(sin(theta),2.);
    tmp1Mat[idx2][idx2] = inputMat[idx1][idx1] * pow(sin(theta),2.)
      - 2.*inputMat[idx1][idx2] * cos(theta) * sin(theta)
	+ inputMat[idx2][idx2] * pow(cos(theta),2.);
    tmp1Mat[idx1][idx2] = tmp1Mat[idx2][idx1] = 0.;

    for ( k=0; k<dim; k++ ) {
      if ( (k != idx1) && (k != idx2) ) {
	tmp1Mat[idx1][k] = tmp1Mat[k][idx1] = inputMat[idx1][k] * cos(theta)
	  + inputMat[idx2][k] * sin(theta);
	tmp1Mat[idx2][k] = tmp1Mat[k][idx2] = -inputMat[idx1][k] * sin(theta)
	  + inputMat[idx2][k] * cos(theta);
      }
    }
    for ( i=0; i<dim; i++ )
      for ( j=0; j<dim; j++ ) 
	  inputMat[i][j] = tmp1Mat[i][j];


    /* Calculating Eigen Values & Vectors */
    for ( i=0; i<dim; i++ )
      for ( j=0; j<dim; j++ )
	tmp1Mat[i][j] = eigenMat[i][j];

    for ( i=0; i<dim; i++ ) {
      eigenVal[i] = inputMat[i][i];
      for ( j=0; j<dim; j++ ) {
	tmp1Mat[i][j] = 0.0;
	for ( k=0; k<dim; k++ )
	  tmp1Mat[i][j] += eigenMat[i][k] * rotateMat[k][j];
      }
    }

    for ( i=0; i<dim; i++ )
      for ( j=0; j<dim; j++ )
	  eigenMat[i][j] = tmp1Mat[i][j];
  }

  free2D(rotateMat,dim);
  free2D(tmp1Mat,dim);

  return -1.0;
}

double** alloc2D(c1, c2)
     int c1, c2;
{
  double **buf, *tmp;
  int i;

  buf = (double **)malloc(sizeof(double *) * c1);
  for (i = 0; i < c1; i++){
    tmp = (double *)malloc(sizeof(double) * c2);
    buf[i] = tmp;
  }
  return buf;
}

void free2D(buf, c)
     double **buf;
     int   c;
{
  int i;

  for (i = 0; i < c; i++)
    free(buf[i]);

  return;
}

