#include <stdio.h>
#include <math.h>
#include "SL_macro.h"
#include "SL_cmd.h"
  
#define PAI    3.14159265358979 
#define MAX_ORDER 101
  
  /*****************************************************
   *      POWER SPECTRUM BY AR MODEL
   *       ---  LEVINSON DURBIN ALGORITHM  ---
   *
   *      LEVIN IBIN , IBO1 , IBO2 , [ A , X ] , IODR ,
   *            IDPT , TOLA
   *            IBIN : INPUT BUFFER
   *            IBO1 : POWER SPECTRUM
   *            IBO2 : AIC
   *            A , X : AIC MIN. OR FIXED ORDER
   *            IODR : MAXIMUM ORDER OF AR
   *            IDPT : POINTS OF POWER SPECTRUM
   *            TOLA : TOLERANCE OF AIC
   *            IBO3 : ERROR SERIES
   *            IBO4 : AR PARAMETER
   *        Translate F to C at Jul,11,1989
   *                                  T.Kobayashi
   ******************************************************
   *        PUT AR PARAMETER TO BUFFER VERSION
   *            17th Mar 1991 modified by K.Hagiwara
   *****************************************************/
  

int
main()
{
  Buffer *work, *o_work1, *o_work2, *o_work3;
  static double	a[MAX_ORDER-1][MAX_ORDER-1] ;
  double p[MAX_ORDER] ;
  Buffer parc[MAX_ORDER] ;
  int	ip, ipp, i, ib2, ib3, ib4, ib5, ndpt, iodr ;
  int	nn, iodr1, i1, kmax, k, ki, l, l1, j, lj, lj1, iaic, ip1; 
  int   dim, idx[MAX_INDEX];
  char  *ialpha;
  Buffer tola, s, aw, val, aicmin, adif, fdpt, s1, s2, sv ;
  Buffer aic[MAX_ORDER-1], r[MAX_ORDER], awork[MAX_ORDER];
  double sf;

  read_syscom() ;

  sf = get_sampling();

  if( sf < 0.0 )
    exit(9);

  if (( work = GetSeries(0, &dim, idx ) ) == NULL )
    exit(4);

  if ( dim != 1 )
    exit(24);

  ib2    = GetBufferID(1);
  ib3    = GetBufferID(2);
  ialpha = GetString(3);

  iodr = (int)GetScalar(4);
  if( iodr < 1 || iodr > 60 )
    exit(22);
  
  if (( ndpt = (int)GetScalar(5) ) < 1 )
    exit(10);
  
  tola = (int)GetScalar(6); 
  if( tola < 0 || tola > 3 )
    exit(2);
  
  ib4 = GetBufferID(7);
  ib5 = GetBufferID(8);

  if ( ib2 <= 0 || ib3 <= 0 || ib4 <= 0 || ib5 <= 0 )
    exit(3);

  nn = IndexSize( dim, idx );

  o_work1 = AllocBuffer( ndpt );
  o_work2 = AllocBuffer( MAX_ORDER-1 );
  o_work3 = AllocBuffer( nn );

  if ( o_work1 == NULL || o_work2 ==  NULL || o_work3 == NULL )
    exit(8);

  /*------< Auto Correlation >----- */
  iodr1 = iodr + 1 ;
  for( i = 1 ; i <= iodr1 ; i ++ ) {
    s = 0.0 ;
    i1 = i -1 ;
    kmax = nn - i1 ;

    for( k = 0 ; k < kmax ; k ++ ) {
      ki = k + i1 ;
      s += work[k]*work[ki];
    }
    r[ i - 1 ] = s / (double)( nn - i ) ;
  }
  
  /*---< Levinson - Durbin Argorithm >---*/
  parc[0] = - r[1] / r[0] ;
  p[0]    = (double)r[0] ;
  a[0][0] = - r[1] / (double)p[0] ;
  p[1]    = (double)( 1.0 - a[0][0] * a[0][0] ) * p[0] ;
  aic[0]  = (Buffer)nn * (Buffer)log( p[1] ) + 2.0 ;
  
  ip = iodr ;
  
  for( l = 2 ; l <= iodr ; l ++ ) {
    s = 0.0 ;
    l1 = l - 1 ;
    for( j = 1 ; j <= l1 ; j ++ ) {
      lj1 = l + 1 - j ;
      s += a[ l1 - 1 ][ j - 1 ] * r[ lj1 - 1 ] ;
    }
    parc[ l - 1 ] = -( r[ l ] + s ) / p[ l - 1 ] ;
    a[ l - 1 ][ l - 1 ] = - ( r[l] + s ) / p[ l - 1 ] ;
    
    aw = fabs( a[ l - 1 ][ l - 1 ] ) ;
    if( aw >= 1.0 ) {
      ip = l1 ;
      printf( "Ill-Conditioned Order < Order(%3d)\n", ip ) ;
      break ;
    }
    for( j = 1 ; j <= l1 ; j ++ ) {
      lj = l - j ;
      a[ l - 1 ][ j - 1 ] = a[ l1 - 1 ][ j - 1 ] +
	a[ l - 1 ][ l - 1 ] * a[ l1 - 1 ][ lj - 1 ] ;
    }
    p[l] = (double)( 1.0 - a[l-1][l-1] * a[l-1][l-1] ) * p[l-1] ;
    aic[l-1] = (double)nn * log( p[l] ) + 2.0 * (double)l ;
  }
  
  /*-------<METHOD FOR SELECTION OF NUMBER OF PARAMETER >-----*/
  switch( ialpha[0] ) {
  case 'A' : /*---< AIC Minimum >---*/
    aicmin = aic[0] ;
    iaic = 1 ;
/*    ipw  = ip ; */
    for( i = 1 ; i <= ip ; i ++ ) {
      val = aic[ i - 1 ] ;
      o_work2[i-1] = val;
      if( aic[ i - 1 ] >= aicmin )
	continue ;
      aicmin = aic[ i - 1 ] ;
      iaic = i ;
    }
    ip = iaic ;
    ip1 = iaic - 1 ;
    for( i = 1 ; i <= ip1 ; i ++ ) {
      j = ip - i ;
      adif = fabs( aic[j-1]  - aicmin ) ;
      if( adif > tola )
	break ;
      iaic = j ;
      aicmin = aic[j-1] ;
    }
    
    ip = iaic ;
    printf( "Order =%5d     Min. AIC =%15.5g\n", iaic, aicmin ) ;
    
    for( i = 1 ; i <= ip ; i ++)
      awork[ i - 1 ] = a[ ip - 1 ][ i - 1 ] ;
    break ;
    
  case 'E' : /*----< Fixed Order & Square Error >---*/
    for( i = 1 ; i <= ip ; i ++ ) {
      val = p[ i - 1 ] ;
      o_work2[i-1] = val;
      awork[ i - 1 ] = a[ ip - 1 ][ i - 1 ] ;
    }
    printf( "Order =%5d\n", ip ) ;
    break ;
    
  default: /*----< Fixed Order >-------------------*/
    printf( "Order =%5d     AIC =%15.5g\n", ip, aic[ ip - 1 ] ) ;
    for( i = 1 ; i <= ip ; i ++ ) {
      val = aic[ i - 1 ] ;
      o_work2[i-1] = val;
      awork[ i - 1 ]= a[ ip - 1 ][ i - 1 ] ;
    }
    break ;
  }
  
  /* end of switch */
  
  /*-----< POWER SPECTR >-------*/
  fdpt = (double)( ndpt - 1 ) ;
  for( i = 1 ; i <= ndpt ; i ++ ) {
    s1 = 1.0 ;
    s2 = 0.0 ;
    i1 = i - 1 ;
/*
    for( l = 1 ; l <= ip ; l ++ ) {
      s1 += ( a[ip-1][l-1]*cos( PAI* (double)( i1*l )/ fdpt ) );
      s2 += ( a[ip-1][l-1]*sin( PAI* (double)( i1*l )/ fdpt ) );
    }
*/
    for( l = 0 ; l < ip ; l ++ ) {
      s1 += ( a[ip-1][l]*cos( PAI* (double)( i1*(l+1) )/ fdpt ) );
      s2 += ( a[ip-1][l]*sin( PAI* (double)( i1*(l+1) )/ fdpt ) );
    }
    s = sf * ( s1 * s1 + s2 * s2 ) ;
/*
    val = 2.0 * p[ip-1]/s ;
*/
    val = 2.0 * p[ip]/s ;
    o_work1[i-1] = val;
  }
  for( j = 1 ; j <= ip ; j ++ )
    printf( " AR Coefficient(%3d)=%15.7g,%15.7g\n",
	   j, a[ ip - 1 ][ j - 1 ], parc[ j - 1 ] ) ;
  
  /*-----< Error Handling >-------*/
  for( i = 1 ; i <= nn ; i ++ ) {
    sv = 0.0 ;
    for( j = 1 ; j <= ip ; j ++ ) {
      if( i - j > 0 )
	sv += ( a[ ip - 1 ][ j - 1 ] * work[i-j-1] ) ; 
    }
    sv += work[i-1] ; 
    o_work3[i-1] = sv ; 
  }
  ipp = ip ;
  if(ipp > 100){
    printf("\n Upper order of AR parameter is 100");
    exit(2);
  }

  if ( WriteBuffer( ib2, 1, &ndpt, o_work1 ) <= 0 ) exit(3);
  if ( WriteBuffer( ib3, 1, &ip,   o_work2 ) <= 0 ) exit(3);
  if ( WriteBuffer( ib4, 1, &nn,   o_work3 ) <= 0 ) exit(3);
  if ( WriteBuffer( ib5, 1, &ipp,  awork   ) <= 0 ) exit(3);

  FreeBuffer( work );
  FreeBuffer( o_work1 );
  FreeBuffer( o_work2 );
  FreeBuffer( o_work3 );
  
  write_syscom();
  return 0;
}
