#include <stdio.h>
#include <math.h>
#include "SL_macro.h"
#include "SL_cmd.h"

/******************************************************
*           normalize                                 *
*---------------------------------------------------- *
*     norm ibuf1,typ4                                 *
*          ibuf1 : input data buffer                  *
*          typ4  :                                    *
*            = 1 : normalize at MAX                   *
*            = 2 : normalize at MIN                   *
*            = 3 : normalize at abusolute MAX         *
*            = 4 : normalize at abusolute MIN         *
*            = 5 : normalize at standard deviation    *
*-----------------------------------------------------*
*                   1986 , 7 , 29                     *
*                   1993 , 4 , 15                     *
*                   1996 , 4 ,  5 (added typ4 = 5)    *
******************************************************/
static int    srch   _ANSI_ARGS_((Buffer *data, int leng, int iflag));
static double absol  _ANSI_ARGS_((double inp));


int
main()
{
  double xnorm, xmean, tmp;
  Buffer *data;
  int    indx, iaddr, i;
  int    dim, index[MAX_INDEX], leng;
  
  read_syscom();

  data   = GetSeries( 0, &dim, index );

  if ( data == NULL )
    exit(4);

  iaddr  = (int)GetScalar( 1 );
  leng = IndexSize( dim, index );

  if ( ( iaddr < 1) || ( iaddr > 5) ) {
    printf("bud type of normalization\n");
    exit(2);
  }

  if ( ( iaddr <= 4) ){
    indx = srch( data, leng, iaddr );

    xnorm = data[ indx ];
    if ( xnorm == 0.0 ) {
      printf("cannot normalize by zero\n");
      exit(2);
    }
  
    for (i = 0; i < leng; i++)
      data[ i ] = data[ i ] / xnorm;

  }else{ 	/* 1996.4.5. by Dora */
    tmp = 0.0;
    for (i = 0; i < leng; i++)
      tmp = tmp + data[ i ];
    xmean = tmp / (double)leng;	
/*    printf("MEAN = %f\n", xmean); */
    
    tmp = 0.0;
    for (i = 0; i < leng; i++)
      tmp = tmp + pow( (data[ i ] - xmean), 2.0 );
    xnorm = sqrt( tmp / (double)leng );
    printf("Standard Deviation = %f\n", xnorm);

    for (i = 0; i < leng; i++)
      data[ i ] = ( data[ i ] - xmean ) / xnorm;
  }

  ReturnSeries( data, dim, index );
  FreeBuffer( data );
  return 0;
}

/*****************************************************
 *                                                    *
 *         srch ( data , iflag  )                     *
 *                                                    *
 *----------------------------------------------------*
 *    search max , min , absmax , absmin              *
 *****************************************************/
static int srch(Buffer *data, int leng, int iflag){
  double xmax, xmin, abxmax, abxmin, abx;
  int    i, indx;

  indx = 0;
  switch (iflag) {

    /* .....max search */
  case 1:
    xmax = data[0];
    for (i = 1; i < leng; i++) {
      if ( data[i] > xmax ) {
	xmax = data[i];
	indx = i;
      }
    }
    break;
    
    /* .....min search */
  case 2:
    xmin = data[0];
    for (i = 1; i < leng; i++) {
      if (data[i] < xmin ) {
	xmin = data[i];
	indx = i;
      }
    }
    break;
    
    /* .....abs max serch */
  case 3:
    abxmax = absol(data[0]);
    for (i = 1; i < leng; i++) {
      abx = absol(data[i]);
      if ( abx > abxmax) {
	abxmax = abx;
	indx = i;
      }
    }
    break;
    
    /* .....abs min search */
  case 4:
    abxmin = absol(data[0]);
    for (i = 1; i < leng; i++) {
      abx = absol(data[i]);
      if ( abx < abxmin ) {
	abxmin = abx;
	indx = i;
      }
    }
    break;
  }
  return( indx );
}

static double absol(double inp){
  if ( inp < 0 )
    inp = -inp;
  return( inp );
}

