#include <stdio.h>
#include <math.h>
#include "SL_macro.h"
#include "SL_cmd.h"
#include "Complex.h"

/*****************************************************
*                                                    *
*   POLE  PBUF , IBX , IBY                           *
*                                                    *
*         PBUF : AR PARAMETER BUFFER                 *
*         IBX  : REAL                                *
*         IBY  : IMAGINARY                           *
*                                                    *
*----------------------------------------------------*
*              CODED BY nishi       Aug 01, 2001     *
*****************************************************/

#define      PI    3.1415925358979323846

ComplexValue pole_cexp  _ANSI_ARGS_((ComplexValue i));
double       pole_cabsl _ANSI_ARGS_((ComplexValue i));

int          pole_dka   _ANSI_ARGS_((ComplexValue *c, ComplexValue *roots, int n, double eps));

int main(){
  Buffer       *coef, *data1, *data2;
  int          dim, index[MAX_INDEX];
  int          ibx, iby;
  int          i, m;
  double       eps=1.0e-06;
  ComplexValue a[101],roots[101];

  read_syscom();

  if ((coef = GetSeries( 0, &dim, index )) == NULL ) 
    exit(4);

  ibx    = GetBufferID(1);
  iby    = GetBufferID(2);

  m = index[0];
  if ((data1 = AllocBuffer(m)) == NULL) exit(8);
  if ((data2 = AllocBuffer(m)) == NULL) exit(8);

  a[0] = cplx(1.0,0.0);
  for (i=1; i<=m; i++)
    a[i] = cplx(coef[i-1],0.0);
  /*
  a[m] = cplx(1.0,0.0);
  for (i=1; i<=m; i++)
    a[m-i] = cplx(coef[i-1],0.0);
  */

  pole_dka(a, roots, m, eps);

  for (i=1; i<=m; i++){
    data1[i-1] = roots[i].real;
    data2[i-1] = roots[i].imag;
  }

  if ( WriteBuffer( ibx, dim, index, data1 ) == -1 ) exit(3);
  if ( WriteBuffer( iby, dim, index, data2 ) == -1 ) exit(3);
  write_syscom();

  FreeBuffer(data1);
  FreeBuffer(data2);
  return 0;
}


int pole_dka(ComplexValue *c, ComplexValue *roots, int n, double eps)
     /* Durand-Kerner-Aberth Method */
{
  int          m=50;  /* limit of trial number */
  int          i, j, k, flag;
  double       rmax, cf, p0, p1;
  ComplexValue f1, f2, ri, b, z, zmax, *ratio;

  if( (ratio=(ComplexValue*)malloc((n+1)*sizeof(ComplexValue))) == NULL ) {
    exit(8);
  }

  /* normalize coefficients */
  /*
  for( i=1 ; i<=n ; i++ ) {
    c[i] = cdiv(c[i], c[0]);
  }
  c[0] = cplx(1.0, 0.0);
  */

  /* get initial roots */
  rmax = 0.0;
  b = cdiv(c[1], cplx((double)n, 0.0));
  for( i=2 ; i<=n ; i++ ) {
    cf = (double)n * pow(pole_cabsl(c[i]),1.0/(double)i);
    if( rmax < cf ) {
      rmax = cf;
    }
  }

  p0 = 3.0 / (double)(2*n);
  p1 = 2*PI / (double)n;
  zmax = cplx(rmax, 0.0);
  for( i=1 ; i<=n ; i++ ) {
    z = pole_cexp(cplx(0.0, p0+(double)(i-1)*p1));
    roots[i] = csub(cmpl(z, zmax), b);
  }

  k = 0;
  while( k++ < m ) {
    for( i=1 ; i<=n ; i++ ) {
      f1 = c[0];
      f2 = cplx(1.0, 0.0);
      ri = roots[i];
      for( j=1 ; j<=n ; j++ ) {
	f1 = cadd(cmpl(f1, ri), c[j]);
	if( j != i ) {
	  f2 = cmpl(f2, csub(ri, roots[j]));
	}
      }
      ratio[i] = cdiv(f1, f2);
      roots[i] = csub(ri, ratio[i]);
    }

    flag = 1;
    for( i=1 ; i<=n ; i++ ) {
      if( pole_cabsl(ratio[i]) > eps ) {
	flag = 0;
	break;
      }
    }
    if( flag ) {
      free((char*)ratio);
      return( 0 );
    }
  }

  free((char*) ratio);
  return( -1 );
}


ComplexValue pole_cexp(ComplexValue i)
{
  ComplexValue k;

  k.real = exp(i.real)*cos(i.imag);
  k.imag = exp(i.real)*sin(i.imag);

  return( k );
}

double pole_cabsl(ComplexValue i)
{
  return sqrt(i.real*i.real+i.imag*i.imag);
}

