#include <stdio.h>
#include <math.h>
#include "SL_macro.h"
#include "SL_cmd.h"

/******************************************************
 *   "WINDOW   BF1 , BF2 , TYPE , MODE"               *
 *                                                    *
 *	BF1  : Input Buffer No.                       *
 *	BF2  : Output Buffer No.		      *
 *	TYPE : WINDOW TYPE			      *
 *----------------------------------------------------*
 *             1:HAMMING                              *
 * 	       2:HANNING                              *
 *	       3:BLACKMAN                             *
 *             4:Triangle                             *
 *----------------------------------------------------*
 *	MODE : Compensation Flag 0 or 1	              *
 *						      *
 * Revised at 19th Nov 1988                           *
 *****************************************************/
#define PAI     3.14159265358979
#define PAI2    (PAI*2.0)

typedef struct window {
  int Wtype ;
  int CmpFlag ;
} WINDOW ;

/*****************************************************
 *                                                    *
 *   WINDOW                                           *
 *                                                    *
 *****************************************************/
int
Exec_Window( WINDOW Win, Buffer *data1, int dpt )
{
  int	 i, m;
  double omg, rev, gain = 0.5, cons = 0.5;
  
  rev = 1.0 ;
  switch ( Win.Wtype ) {
  case 1:			/* Hamming  Window */
    gain = 0.46;
    cons = 0.54;
    break;
  case 2:			/* Hanning  Window */
    gain = cons = 0.5;
    break;
  case 3:			/* Blackman Window */
    gain = 0.5;
    cons = 0.42;
    break;
  case 4:			/* Bartlet  Window */
    cons = 0.5;
    gain = 2.0;
    break;
  }
  if ( Win.CmpFlag == 1 )
    rev = cons;
  
/*  omg = PAI2 / (double)dpt; */
  omg = PAI2 / (double)(dpt - 1); /* found by H.Sakai on 24/05/99 */
  
  if ( Win.Wtype != 4 )
    for ( i = 0 ; i < dpt ; i ++ )
      data1[ i ] *= ( cons - gain * cos( omg * i ) ) / rev ;
  else {
    m = dpt / 2 ;
    for( i = 0 ; i < m ; i ++ )
      data1[ i ] *= gain * (double)i / (double)dpt / rev ;
    
    for ( i = m ; i < dpt ; i ++ )
      data1[ i ] *= gain * ( 1.0 - (double)i / (double)dpt ) / rev ;
  }
  return 0;
}

int
Exec_Window2( WINDOW Win, Buffer *data1, int dim, int idx[MAX_INDEX] )
{
  int	 i, j, dpt, p, q;
  double rev, gain = 0.5, cons = 0.5;
  double p2_0, p2_1, mmx, mmy;
  
  /****** Window *****/
  switch ( Win.Wtype ) {
  case 1:			/* Hamming  Window */
    gain = 0.46;
    cons = 0.54;
    break;
  case 2:			/* Hanning  Window */
    gain = cons = 0.5;
    break;
  case 3:			/* Blackman Window */
    gain = 0.5;
    cons = 0.42;
    break;
  case 4:			/* Bartlet  Window */
    cons = 0.5;
    gain = 2.0;
    break;
  }
  rev = ( Win.CmpFlag == 1 ) ? cons : 1.0;

  p2_0 = PAI2/(double)idx[0];
  p2_1 = PAI2/(double)idx[1];
  p   = idx[0]/2;
  q   = idx[1]/2;
  
  if ( Win.Wtype != 4 )
    for ( i = 0 ; i < idx[0] ; i++ ) {
      dpt = i*idx[1];
      mmy = (cons + gain*cos( (double)(i-p)*p2_0 ))/rev;
      for ( j = 0 ; j < idx[1] ; j++ ) {
	mmx = (cons + gain*cos( (double)(j-q)*p2_1 ))/rev;
	data1[dpt+j] *= (mmx*mmy);
      }
    }
  else {
    for ( i = 0 ; i < p ; i++ ) {
      dpt = i*idx[1];
      mmy = gain*(double)i/idx[0]/rev;
      for ( j = 0 ; j < q ; j++ ) {
	mmx = gain*(double)j/idx[1]/rev;
	data1[dpt+j] *= (mmx*mmy);
      }
      for ( j = q ; j < idx[1]; j++ ) {
	mmx = gain*(1.0-(double)j/idx[1])/rev;
	data1[dpt+j] *= (mmx*mmy);
      }
    }
    for ( i = p ; i < idx[0] ; i++ ) {
      dpt = i*idx[1];
      mmy = gain*(1.0 - (double)i/idx[0])/rev;
      for ( j = 0; j < q; j++ ) {
	mmx = gain*(double)j/idx[1]/rev;
	data1[dpt+j] *= (mmx*mmy);
      }
      for ( j = q; j < idx[1] ; j++ ) {
	mmx = gain*(1.0-(double)j/idx[1])/rev;
	data1[dpt+j] *= (mmx*mmy);
      }
    }
  }
  return 0;
}

int
main()
{
  Buffer *data1;
  int   dim, idx[MAX_INDEX];
  WINDOW Win ;

  read_syscom() ;
  
  if ( (data1 = GetSeries(0,&dim,idx)) == NULL ) {
    printf("failed to read buffer\n");
  }
  Win.Wtype   = (int)GetScalar(1);
  Win.CmpFlag = (int)GetScalar(2);
  
  if ( Win.Wtype > 4 || Win.Wtype < 1 )
    exit( 2 ) ;

  if ( Win.CmpFlag > 1 || Win.CmpFlag < 0 )
    exit( 2 ) ;
    
  if ( dim == 1 )
    Exec_Window( Win, data1, idx[0] );
  else if ( dim == 2 )
    Exec_Window2( Win, data1, dim, idx );

  ReturnSeries( data1, dim, idx );
  FreeBuffer( data1 );
  return 0;
}

