/******************************************************************
 **                                                               **
 **      File Name : buffer.c                                     **
 **                                                               **
 **           Data Buffer Controll Routine                        **
 **                  for SATELLITE Basic Library                  **
 **                                                               **
 **                                      Coded by S.Hitomi        **
 **                                                               **
 ******************************************************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "SL_macro.h"

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#else
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#endif

#include <stdio.h>
#include <string.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif

#include <signal.h>
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_FILE_H
#include <sys/file.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif


/*
   #define BUF_NAME	"buf."
   */
#define BUF_NAME	"BUFFER"

/* Prototypes of Private Functions */
static void     BufferMon    _ANSI_ARGS_((char *fname));
static int      NameOfBuffer _ANSI_ARGS_((int n, char *fname));
static int      GetHeader    _ANSI_ARGS_((int fd, int *index));
static int      PutHeader    _ANSI_ARGS_((int fd, int dim, int *index));
static Buffer  *GetBuffer    _ANSI_ARGS_((int fd, int *dim, int *index));
static int      PutBuffer    _ANSI_ARGS_((int fd, int dim, int *index,
					  Buffer *area));
static Buffer  *GetSubBuffer _ANSI_ARGS_((int fs, int time,
					  int *sub_dim, int *sub_index));
static int      PutSubBuffer _ANSI_ARGS_((int fd, int time,
					  int dim, int *index,
					  int sub_dim, int *sub_index,
					  Buffer *area));

enum {
  SUCCESS, OUT_MEM, OPEN_ER, READ_DT, WRITE_DT, READ_HD, WRITE_HD,
  OUT_INDEX, ILL_INDEX, ILL_DIM, DIM_MIS, NO_READY, BSIZ_MIS, UNLINK
};

static int      ErrorNo = 0;

Buffer *
AllocBuffer(buf_size)
     unsigned  buf_size;
{
  ErrorNo = SUCCESS;

#ifdef DEBUG
  fprintf(stderr,"AllocBuffer():\n");
#endif

  return ((Buffer *) emalloc(buf_size * sizeof(Buffer)));
}


Buffer	*
CAllocBuffer(buf_size)
     unsigned	buf_size;
{
  unsigned   siz = buf_size * sizeof(Buffer);
  Buffer    *buf = (Buffer *) emalloc(siz);
  ErrorNo = SUCCESS;

#ifdef DEBUG
  fprintf(stderr,"CAllocBuffer():\n");
#endif

  if(buf != NULL)
    Bzero(buf, siz);
  return buf;
}


int
FreeBuffer(area)
     Buffer *area;
{
#ifdef DEBUG
  fprintf(stderr,"FreeBuffer():\n");
#endif

  if (area != NULL) {
    free((char *) area);
    area = NULL;
  }
  return 0;
}


int
DestroyBuffer(n)
     int  n;
{
  char  buff_file[FILE_LENGTH];
  int   status;

  ErrorNo = SUCCESS;
  if (NameOfBuffer(n, buff_file) == -1)
    return -1;
  status = unlink(buff_file);
  
  strcat(buff_file, "M");
#if 0
  if (access(buff_file, 0) == 0)
#endif
    unlink(buff_file);

  if(status == -1)
    ErrorNo = UNLINK;

  return status;
}


int
InitBuffer(n, dim, index)
     int  n;
     int  dim;
     int *index;
{
  int   fd, msize, length;
  char *area, buff_file[FILE_LENGTH];
  
#ifdef DEBUG
  fprintf(stderr,"InitBuffer():\n");
#endif

  ErrorNo = SUCCESS;
  if (dim < 1 ) {
    ErrorNo = ILL_DIM;
    return(-1);
  }
  if ( index == NULL) {
    ErrorNo = ILL_INDEX;
    return (-1);
  }
  
  /* Open Buffer file */
  NameOfBuffer(n, buff_file);
  if ((fd = open(buff_file, O_WRONLY | O_CREAT | O_TRUNC, 0644)) == -1) {
    ErrorNo = OPEN_ER;
    return (-1);
  }
  msize = IndexSize(dim, index) * sizeof(Buffer);
  area  = emalloc(msize);
  Bzero(area, msize);
  length = PutBuffer(fd, dim, index, (Buffer*)area);
  close(fd);
  if (length == -1)
    return -1;
  return 0;
}

Buffer *
ReadBuffer(n, dim, index)
     int  n;	 /* buffer ID                  */
     int *dim;	 /* dimension of read buffer   */
     int *index; /* index range of read buffer */
{
  int      fd;
  Buffer  *buf;
  char     buff_file[FILE_LENGTH];
  
  ErrorNo = SUCCESS;
  if (index == NULL) {
    ErrorNo = ILL_INDEX;
    return NULL;
  }
  if (NameOfBuffer(n, buff_file) == -1) 
    return NULL;
  if ((fd = open(buff_file, O_RDONLY, 0)) < 0) {
    ErrorNo = OPEN_ER;
    return NULL;
  }
  buf = GetBuffer(fd, dim, index);
  close(fd);
  return buf;
}


int
WriteBuffer(n, dim, index, area)
     int     n;
     int     dim;
     int    *index;
     Buffer *area;
{
  char  buff_file[FILE_LENGTH];
  int   fd, length;
  
  ErrorNo = SUCCESS;
  /* if index == NULL then abort */
  if (index == NULL) {
    ErrorNo = ILL_INDEX;
    return (-1);
  }
  
  NameOfBuffer(n, buff_file);
  if ((fd = open(buff_file, O_WRONLY | O_CREAT, 0644)) < 0) {
    ErrorNo = OPEN_ER;
    return (-1);
  }
  
  length = PutBuffer(fd, dim, index, area);
  close(fd);
  if (length == -1)
    return (-1);
  
  BufferMon(buff_file);
  return IndexSize(dim, index);
}



int
AppendBuffer(n, sub_dim, sub_index, area)
     int     n;	        /* buffer ID               */
     int     sub_dim;    /* sub dimension of buffer */
     int    *sub_index;  /* sub index of buffer     */
     Buffer *area;       /* sub buffer area         */
{
  char  fname[FILE_LENGTH];
  int   fd, time, wsiz, dim, index[MAX_INDEX];

  ErrorNo = SUCCESS;
  if (NameOfBuffer(n, fname) == -1)
    return (-1);
  if ((fd = open(fname, O_RDWR)) == -1) {
    ErrorNo = OPEN_ER;
    return (-1);
  }
  if ((dim = GetHeader(fd, index)) == -1)
    return (-1);
  time = index[0];
  wsiz = PutSubBuffer(fd, time, dim, index, sub_dim, sub_index, area);
  if (wsiz != -1) {
    index[0] += 1;
    PutHeader(fd, dim, index);
  }
  close(fd);
  return index[0];
}

Buffer *
ReadSubBuffer(n, time, sub_dim, sub_index)
     int  n;	        /* buffer ID               */
     int  time;	        /* time                    */
     int *sub_dim;       /* sub dimension of buffer */
     int *sub_index;     /* sub index of buffer     */
{
  int      fd;
  char     fname[FILE_LENGTH];
  Buffer  *area;

  ErrorNo = SUCCESS;
  if (NameOfBuffer(n, fname) == -1)
    return NULL;
  if ((fd = open(fname, O_RDONLY)) == -1) {
    ErrorNo = OPEN_ER;
    return NULL;
  }
  area = GetSubBuffer(fd, time, sub_dim, sub_index);
  close(fd);
  return area;
}


int
WriteSubBuffer(n, time, sub_dim, sub_index, area)
     int     n;          /* buffer ID               */
     int     time;       /* time                    */
     int     sub_dim;    /* sub dimension of buffer */
     int    *sub_index;  /* sub index of buffer     */
     Buffer *area;       /* sub buffer area         */
{
  int     fd;
  char    fname[FILE_LENGTH];
  int     dim, index[MAX_INDEX];
  int     wsiz;

  ErrorNo = SUCCESS;
  if (NameOfBuffer(n, fname) == -1)
    return (-1);
  if ((fd = open(fname, O_RDWR)) == -1) {
    ErrorNo = OPEN_ER;
    return (-1);
  }
  if ((dim = GetHeader(fd, index)) == -1)
    return (-1);
  wsiz = PutSubBuffer(fd, time, dim, index, sub_dim, sub_index, area);
  if (wsiz != -1) {
    if(time >= index[0]) 
      index[0] = time + 1;
    PutHeader(fd, dim, index);
  }
  close(fd);
  return (wsiz == -1) ? -1 : index[0];
}


Buffer *
ReadTimeSeries(n, sub_dim, sub_index, length)
     int  n;	    /* buffer ID                  */
     int  sub_dim;   /* size of index              */
     int *sub_index; /* spatial location on buffer */
     int *length;
{
  register int    i;
  char            fname[FILE_LENGTH];
  int             fd;
  int             dim, index[MAX_INDEX];
  int             sub_siz, aBlockSize, zero_point;
  long            offset, h_siz;
  Buffer         *area;
  
  ErrorNo = SUCCESS;
  if (NameOfBuffer(n, fname) == -1)
    return NULL;
  if ((fd = open(fname, O_RDONLY)) == -1) {
    ErrorNo = OPEN_ER;
    return NULL;
  }
  if ((dim = GetHeader(fd, index)) == -1)
    return NULL;
  
  if (dim - 1 != sub_dim) {
    ErrorNo = DIM_MIS;
    return NULL;
  }
  if (!RegularIndex(sub_index, SubIndex(index), sub_dim)) {
    ErrorNo = OUT_INDEX;
    return NULL;
  }
  
  sub_siz = (sub_dim == 0) ? 1 : IndexSize(dim - 1, SubIndex(index));
  aBlockSize = sub_siz * sizeof(Buffer);

  if((area = CAllocBuffer((unsigned)index[0])) == NULL)
    return NULL;
  
  /* set the seek pointer at a point of zero [time] */
  zero_point = Index(sub_index, dim - 1, SubIndex(index));
  h_siz = sizeof(int) + dim * sizeof(int); /* dim + index[dim] */
  offset = h_siz + zero_point * sizeof(Buffer);
  lseek(fd, offset, 0);
  
  for (i = 0; i < index[0]; i++) {
    if (read(fd, &area[i], sizeof(Buffer)) != sizeof(Buffer))
      break;
    offset += aBlockSize;
    lseek(fd, offset, 0);	/* increment the seek pointer */
  }
  close(fd);
  *length = i;
  return area;
}


int
WriteTimeSeries(n, sub_dim, sub_index, area, length)
     int       n;	  /* buffer ID                  */
     int       sub_dim;   /* size of index              */
     int      *sub_index; /* spatial location on buffer */
     Buffer   *area;	  /* buffer area (one dimension) */
     int       length;    /* length of buffer           */
{
  register int    i;
  char            fname[FILE_LENGTH];
  int             fd, dim, index[MAX_INDEX];
  int             sub_siz, aBlockSize, zero_point, dpt;
  long            offset, h_siz;
  Buffer	*zbuf;

  ErrorNo = SUCCESS;
  if (NameOfBuffer(n, fname) == -1)
    return -1;
  if ((fd = open(fname, O_RDWR)) == -1) {
    ErrorNo = OPEN_ER;
    return -1;
  }
  if ((dim = GetHeader(fd, index)) == -1)
    return -1;

  if (dim - 1 != sub_dim) {
    ErrorNo = DIM_MIS;
    return -1;
  }
  if (sub_index == NULL)  {
    ErrorNo = ILL_INDEX;
    return -1;
  }

  sub_siz = (sub_dim == 0) ? 1 : IndexSize(dim - 1, SubIndex(index));
  aBlockSize = sub_siz * sizeof(Buffer);

  /* set the seek pointer at a point of zero [time] */
  zero_point = Index(sub_index, dim - 1, SubIndex(index));
  h_siz = sizeof(int) + dim * sizeof(int); /* dim + index[dim] */
  offset = h_siz + zero_point * sizeof(Buffer);
  lseek(fd, offset, 0);

  dpt = Min(index[0], length);
  for (i = 0; i < dpt; i++) {
    if (write(fd, &area[i], sizeof(Buffer)) != sizeof(Buffer))
      break;
    offset += aBlockSize;
    lseek(fd, offset, 0);	/* increment the seek pointer */
  }
  if(index[0] >= length) {
    close(fd);
    return i;
  }

  /****Need Check : Append Data ******/
  if((zbuf = CAllocBuffer((unsigned)sub_siz)) == NULL) 
    return i;
  lseek(fd, 0L, 2);		/* set seek pointer at end of file */
  for(i = dpt; i < length; i++) {
    zbuf[zero_point] = area[i];
    if(write(fd, zbuf, aBlockSize) != aBlockSize)
      break;
  }
  index[0] = length;
  PutHeader(fd, dim, index);
	
  free(zbuf);
  close(fd);
  return i;
}



int
GetBufferInfo(n, index)
     int   n;	   /* buffer ID       */
     int  *index;  /* index of buffer */
{
  int             fd, dim;
  char            buff_file[FILE_LENGTH];
  ErrorNo = SUCCESS;
  if (NameOfBuffer(n, buff_file) == -1)
    return (-1);
  if ((fd = open(buff_file, O_RDONLY, 0)) < 0) {
    ErrorNo = OPEN_ER;
    return (-1);
  }
  dim = GetHeader(fd, index);
  close(fd);
  return dim;			/* return Buffer dimension */
}


/******************************************
 **                                       **
 **   Local Functions for Buffer I/O      **
 **                                       **
 ******************************************/

static int
PutSubBuffer(fd, time, dim, index, sub_dim, sub_index, area)
     int      fd;	 /* file descriptor         */
     int      time;	 /* time                    */
     int      dim;	 /* dimension of buffer     */
     int     *index;	 /* index of buffer         */
     int      sub_dim;   /* dimension of sub buffer */
     int     *sub_index; /* index of sub buffer     */
     Buffer  *area;	 /* sub buffer area         */
{
  int     sub_siz, siz, aBlockSize;
  long    h_siz, offset;

  siz = IndexSize(dim - 1, SubIndex(index));
  if(siz == 0) siz = 1;
  sub_siz = IndexSize(sub_dim, sub_index);
  if (siz != sub_siz) {
    ErrorNo = BSIZ_MIS;
    return (-1);
  }
  aBlockSize = sub_siz * sizeof(Buffer);
  h_siz = sizeof(int) + dim * sizeof(int); /* dim + index[dim] */
  offset = h_siz + (long) aBlockSize *(long) time;
  lseek(fd, offset, 0);		/* set the seek pointer */

  if (write(fd, (char *) area, aBlockSize) != aBlockSize) {
    ErrorNo = WRITE_DT;
    return -1;
  }
  return aBlockSize;
}

static Buffer  *
GetSubBuffer(fd, time, sub_dim, sub_index)
     int  fd;	     /* file descriptor         */
     int  time;      /* time                    */
     int *sub_dim;   /* dimension of sub buffer */
     int *sub_index; /* index of sub buffer     */
{
  int      sub_siz, aBlockSize;
  int      dim, index[MAX_INDEX];
  long     h_siz, offset;
  Buffer  *area;

  if( (dim = GetHeader(fd, index)) < 0)
    return NULL;
  if( index[0] <= time)
    return NULL;

  *sub_dim = dim - 1;
  CopyIndex(sub_index, SubIndex(index), *sub_dim);
  sub_siz = (*sub_dim == 0) ? 1 : IndexSize(*sub_dim, sub_index);
  aBlockSize = sub_siz * sizeof(Buffer);
  h_siz = sizeof(int) + dim * sizeof(int); /* dim + index[dim] */
  offset = h_siz + (long) aBlockSize *(long) time;
  lseek(fd, offset, 0);		/* set the seek pointer */

  if((area = AllocBuffer((unsigned)aBlockSize)) == NULL)
    return NULL;
  if (read(fd, (char *) area, aBlockSize) != aBlockSize) {
    ErrorNo = READ_DT;
    return NULL;
  }
  return area;
}


static int
PutHeader(fd, dim, index)
     int   fd;
     int   dim;
     int  *index;
{
  int   wsize = dim * sizeof(int);
  ErrorNo = WRITE_HD;
  lseek(fd, 0L, 0);
  if (write(fd, &dim, sizeof(int)) != sizeof(int)) 
    return -1;
  if (write(fd, index, wsize) != wsize)
    return -1;
  return ErrorNo = 0;
}


static int
GetHeader(fd, index)
     int   fd;
     int  *index;
{
  int       dim;
  unsigned  index_size;

  lseek(fd, 0L, 0);
  if (read(fd, &dim, sizeof(int)) == -1) {
    ErrorNo = READ_HD;
    return (-1);
  }
  if (dim < 1) {
    ErrorNo = ILL_DIM;
    return (-1);
  }

  index_size = (unsigned) (dim * sizeof(int));
  if (read(fd, index, index_size) != index_size) {
    ErrorNo = READ_HD;
    return (-1);
  }
  return dim;
}


static Buffer*
GetBuffer(fd, dim, index)
     int   fd;	  /* file descriptor       */
     int  *dim;	  /* dimension of buffer   */
     int  *index; /* index range of buffer */
{
  int           length;
  unsigned int  rsize;
  Buffer       *area;

  if((*dim = GetHeader(fd, index)) < 0)
    return NULL;

  /* buffer area allocation */
  length = IndexSize(*dim, index);
  area = (Buffer *) AllocBuffer((unsigned) length);

  rsize = (unsigned) length *sizeof(Buffer);
  if (read(fd, (char *) area, rsize) != rsize) {
    ErrorNo = READ_DT;
    return NULL;
  }
  return area;
}


static int
PutBuffer(fd, dim, index, area)
     int     fd;	/* file descriptor       */
     int     dim;	/* dimension of buffer   */
     int    *index;	/* index range of buffer */
     Buffer *area;	/* area of buffer        */
{
  int       length;
  unsigned  wsize;

  if (PutHeader(fd, dim, index) == -1)
    return -1;

  length = IndexSize(dim, index);
  wsize = (unsigned) length *sizeof(Buffer);
  if (write(fd, (char *) area, wsize) != wsize) {
    ErrorNo = WRITE_DT;
    return -1;
  }
  return length;
}


static void
BufferMon(fname)
     char  *fname;
{
  int  pid, fd;

  strcat(fname, "M");
  if (access(fname, 0) == 0) {
    fd = open(fname, O_RDONLY, 0);
    while (read(fd, (char *) &pid, sizeof(int)) == sizeof(int)) {
      kill(pid, SIGUSR2);
    }
    close(fd);
  }
}


static int
NameOfBuffer(n, fname)
     int    n;	        /* buffer ID */
     char  *fname;	/* buffer name */
{
  sprintf(fname, "%s%s%d", get_tmpdir(), BUF_NAME, n);
#if 0
  if (access(fname, 0) == -1) {	/* Not Ready Buffer File */
    ErrorNo = NO_READY;
    return -1;
  }
#endif
  return 0;
}


int
BufferErrorNo()
{
  return ErrorNo;
}

char *
BufferErrorMessage()
{
  static char    *ErrorMessage[] = {
    "",                                  /*  0 */ 
    "out of memory",                     /*  1 */ 
    "failed to open internal buffer",    /*  2 */ 
    "failed to read data",               /*  3 */ 
    "failed to write data",              /*  4 */ 
    "failed to read header",             /*  5 */ 
    "failed to write header",            /*  6 */
    "index, out of range",               /*  7 */
    "illegal index",                     /*  8 */ 
    "illegal dimension",                 /*  9 */ 
    "dimension mismatch",                /* 10 */ 
    "not ready internal buffer",         /* 11 */ 
    "block size mismatch",               /* 12 */
    "failed to destroy internal buffer", /* 13 */ 
  };
  return ErrorMessage[ErrorNo];
}

