/******************************************************************
**                                                               **
**      File Name : rwfile.c                                     **
**                                                               **
**              Data File Read / Write Routines                  **
**                for SATELLITE Basic Library                    **
**                                                               **
**                                      Coded by S.Hitomi        **
**                                                               **
******************************************************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "SL_macro.h"
#include "SL_cmd.h"

#include <stdio.h>
#include <string.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif

#ifdef HAVE_SYS_FILE_H
#include <sys/file.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#else
#ifdef HAVE_MALLLOC_H
#include <malloc.h>
#endif
#endif

#ifndef F_OK
#define F_OK    0		/* test for presence of file */
#endif

/*************  CPU type  **************
 BigEndian    - 680x0 or SPARC (EWS)
 LittleEndian - 80x86 (PC)
****************************************/

static int      DataType = BigEndian;	/* SPARC, LUNA88K, TITAN */

static int      aDataSize = 4;	/* sizeof(float) */

/* rwfile.old.o */
static int      write_data     _ANSI_ARGS_((int fd, char *data, int elsize,
					    int siz, int flag));
static char    *read_data      _ANSI_ARGS_((int fd, int elsize, int siz,
					    int flag));

/* private functions */
static int      openfile       _ANSI_ARGS_((char *f_name, Header *head));
static int      fillgap        _ANSI_ARGS_((int fd, int bsize));
static int      _LoadHeader    _ANSI_ARGS_((int fd, Header *head));
static int      _StoreHeader   _ANSI_ARGS_((int fd, Header *head));
static int      TotalDataSize  _ANSI_ARGS_((Header *head));
static int      cpu_type       _ANSI_ARGS_((/* no args */));
static void     ReverseData    _ANSI_ARGS_((char *data, int siz, int length));
static void     ReverseHead    _ANSI_ARGS_((Header *head));


int
ChangeDataType(type)
    int             type;
{
  switch(type) {
  case 0:
  case 1:
    DataType = type + BigEndian;
    break;
  case BigEndian:
  case LittleEndian:
    DataType = type;
    break;
  default:
    break;
  }
  return DataType;
}

int
ChangeDataSize(siz)
    int             siz;
{
  if (siz == 2 || siz == 4 || siz == 8)
    aDataSize = siz;
  return aDataSize;
}

/**************************************************************
  --------- Public Functions for Data File Operation  -----------
  ReadFile()  : read data file
  WriteFIle() : write data fIle
  LoadData()  : read series-block from data file
  StoreData() : write series-block to data file
  ---------------------------------------------------------------
  ***************************************************************/

char *ReadFile(f_name, head)
    char   *f_name;	/* data file name */
    Header *head;	/* header of data */
{
  char           *area;
  int             siz, fd = open(f_name, O_RDONLY);
  
  if (fd < 0)
    return NULL;
  
  if (_LoadHeader(fd, head) == -1)
    return NULL;
  
  siz = TotalDataSize(head);
  area = read_data(fd, head->data_size, siz, head->type_flag);
  if (area == NULL)
    return NULL;
  
  close(fd);
  return area;
}



int
WriteFile(f_name, dim, index, data)
    char           *f_name;	/* data file name    */
    int		dim;
    int		*index;
    char           *data;	/* data              */
{
  int             fd, siz;
  Header		head;
  InitHeader(&head);
  
  if(dim == 1) {
    head.dim = 2;
    head.index[0] = index[0];
    head.index[1] = 1;
  } else {
    head.dim = dim;
    CopyIndex(head.index, index, head.dim);
  }
  
  /* file create */
  fd = open(f_name, O_RDWR | O_CREAT | O_TRUNC, 0644);
  if (fd == -1)
    return -1;
  
  siz = TotalDataSize(&head);
  
  if (_StoreHeader(fd, &head) == -1)
    return -1;
  
  if (write_data(fd, data, head.data_size, siz, head.type_flag) == -1)
    return (-1);
  close(fd);
  return 0;
}

char  *
LoadData(f_name, numb, head)
    char    *f_name;	/* data file name        */
    int      numb;	/* number of block       */
    Header  *head;	/* header of read data   */
{
  char           *area;
  long            offset;
  int             aBlockSize = 0;
  int             fd = open(f_name, O_RDONLY);
  if (fd == -1)
    return NULL;
  if (_LoadHeader(fd, head) == -1)
    return NULL;
  
#if 0
  PrintHeader(head);
#endif
  if (head->dim == 0)
    return NULL;
  if (head->index[0] < numb)
    return NULL;
  
  aBlockSize = TotalDataSize(head) / head->index[0];	/* [byte] */
  offset = sizeof(Header) + (long) aBlockSize *(long) numb;
  lseek(fd, offset, 0);	/* set the seek pointer */
  
  area = read_data(fd, head->data_size, aBlockSize, head->type_flag);
  if (area == NULL)
    return NULL;
  
  close(fd);
  return area;
}


int
StoreData(f_name, numb, dim, index, data)
    char           *f_name;	/* data file name         */
    int             numb;	/* number of block        */
    int		dim;
    int		*index;
    char           *data;	/* write data (aDataSize) */
{
  int             aBlockSize;
  Header		head;
  int		bsiz, fsiz, old_numb;
  int		fd = openfile(f_name, &head);
  
  if(fd == -1)
    return -1;
  
  if(head.data_size != aDataSize) {
    fprintf(stderr, "%s : a data size mismatch", f_name);
    return -1;
  }
  
  if(head.dim == 0) {
    head.dim = dim + 1;
    CopyIndex(&head.index[1], index, dim);
  }
  old_numb = head.index[0];
  head.index[0] = Max(old_numb, numb + 1);
  
  bsiz = IndexSize(dim, index);
  fsiz = IndexSize(head.dim - 1, SubIndex(head.index));
  if(bsiz != fsiz) {
    fprintf(stderr, "%s : block size mismatch", f_name);
    return -1;
  }
  
  aBlockSize = bsiz * aDataSize; /* [byte] */
  
  if (_StoreHeader(fd, &head) == -1)
    return -1;
  if (numb > head.index[0]) {
    fillgap(fd, aBlockSize * (numb - old_numb));
  } else {
    long            offset;
    offset = sizeof(Header) + (long) aBlockSize *(long) numb;
    lseek(fd, offset, 0);	/* set the seek pointer */
  }
  
  if (write_data(fd, data, head.data_size, aBlockSize, head.type_flag) == -1)
    return -1;
  
  close(fd);
  return 0;
}

/****************************************************************
  --------- Private Functions for Data File Operation  ----------
  openfile()  : open exist file
  fillgap()   : fill a gap ( zero set )
  write_data(): write data block
  read_data() : read data block
  ---------------------------------------------------------------
  ***************************************************************/

static int
openfile(f_name, head)
    char   *f_name;
    Header *head;
{
  int             fd;
  if (access(f_name, F_OK) != 0) {
    if((fd = open(f_name, O_RDWR | O_CREAT | O_TRUNC, 0644)) == -1)
      return(-1);
    InitHeader(head);
  } else {
    if((fd = open(f_name, O_RDWR)) == -1)
      return (-1);
    if (_LoadHeader(fd, head) == -1)
      return (-1);
  }
  return fd;
}

static int
fillgap(fd, bsize)
    int  fd;	/* file descriptor  */
    int  bsize;	/* block size (gap) */
{
  char           *area = (char *) malloc(bsize);
  if (area == NULL)
    return -1;
  Bzero(area, bsize);
  lseek(fd, 0L, 2);	/* extend the file size */
  if (write(fd, area, bsize) != bsize)
    return -1;
  free(area);
  return 0;
}


static int
write_data(fd, data, elsize, siz, flag)
    int   fd;	  /* file descriptor     */
    char  *data;  /* write  data         */
    int   elsize; /* element size [byte] */
    int   siz;	  /* total reading size  */
    int   flag;	  /* cpu type flag       */
{
  if (flag != cpu_type())
    ReverseData((char *) data, elsize, siz / elsize);
  if (write(fd, data, siz) != siz)
    return (-1);
  if (flag != cpu_type())
    ReverseData((char *) data, elsize, siz / elsize);
  return (0);
}

static char *
read_data(fd, elsize, siz, flag)
    int fd;	/* file descriptor     */
    int elsize;	/* element size [byte] */
    int siz;	/* total reading size  */
    int flag;	/* cpu type flag       */
{
  char           *area = (char *) malloc(siz);
  if (area == NULL)
    return NULL;
  if (read(fd, area, siz) != siz)
    return NULL;
  if (flag != cpu_type())
    ReverseData((char *) area, elsize, siz / elsize);
  return area;
}


/**************************************************************
  ---------- Public Functions for Header Operation  -------------
  StoreHeader(): Store Header //include ReverseHeader()
  LoadHeader() : Load Header  //include ReverseHeader()
  ---------------------------------------------------------------
  ***************************************************************/

void
PrintHeader(head)
    Header		*head;
{
  int	i;
  printf("type_flag = %d\n", (int)head->type_flag);
  printf("data_size = %d\n", (int)head->data_size);
  printf("opr_name  = [%s]\n", head->opr_name);
  printf("comment   = [%s]\n", head->comment);
  printf("date      = [%d:%d:%d]\n", head->date[0], head->date[1], head->date[2]);
  printf("dim       = %d\n", (int)head->dim);
  printf("index     = ");
  for(i = 0; i < 10; i++)
    printf("[%d]", head->index[i]);
  puts("");
  printf("samf      = %f\n", head->samf);
}


void
InitHeader(head)
    Header         *head;
{
  Header          tmp;	/* for sizeof */
  int             y, m, d;
  
  Bzero((char *)head, sizeof(Header));
  head->type_flag = (char) DataType;	/* Default = 0 */
  head->data_size = (char) aDataSize;	/* Default = 4 byte */
  
  getusername(head->opr_name, sizeof(tmp.opr_name));
  
  get_date(&y, &m, &d);
  head->date[0] = (char) y;
  head->date[1] = (char) m;
  head->date[2] = (char) d;
}


int
LoadHeader(fname, head)
    char    *fname;
    Header  *head;
{
  int fd = open(fname, O_RDONLY);

  if (fd == -1)
    return -1;
  _LoadHeader(fd, head);
  close(fd);
  return 0;
}

int
StoreHeader(fname, head)
    char           *fname;
    Header         *head;
{
  int fd = open(fname, O_RDWR);

  if (fd == -1)
    return -1;
  _StoreHeader(fd, head);
  close(fd);
  return 0;
}

static int
_LoadHeader(fd, head)
    int    fd;
    Header *head;
{
  lseek(fd, 0L, 0);	/* begin of file */
  if (read(fd, head, sizeof(Header)) != sizeof(Header))
    return (-1);
  
  if (head->type_flag != cpu_type())
    ReverseHead(head);
  
  return 0;
}

static int
_StoreHeader(fd, head)
    int    fd;
    Header *head;
{
  int             status = 0;

  if (head->type_flag != cpu_type())
    ReverseHead(head);
  
  lseek(fd, 0L, 0);	/* begin of file */
  if (write(fd, head, sizeof(Header)) != sizeof(Header))
    status = -1;
  
  if (head->type_flag != cpu_type())
    ReverseHead(head);
  return status;
}

/**************************************************************
  -----------------------  Private Functions --------------------
  TotalDataSize()	: Total Data Size
  ReverseData()   : Reverse the byte order (Data)
  ReverseHead()   : Reverse the byte order (Header)
  ---------------------------------------------------------------
  local function
  reverse() : reverse the byte order
  (2, 4, and 8 byte data)
  ***************************************************************/

static int
TotalDataSize(head)
    Header   *head;
{
  int	isiz = IndexSize(head->dim, head->index);
  return isiz * head->data_size; /* [byte] */
}

static int
cpu_type()
{
  short i = 0x0001;
  char *c = (char *)&i;
  return ((int)c[0] == 1) ? LittleEndian : BigEndian;
}

static void
reverse(s, siz)
    unsigned char  *s;
    int            siz;
{
  register int      i;
  register unsigned char c;
  int               half = siz / 2, j = siz - 1;
  
  for (i = 0; i < half; i++) {
    c = s[i];
    s[i] = s[j - i];
    s[j - i] = c;
  }
}

static void
ReverseData(data, siz, length)
    char       *data;	/* pointer of data area */
    int         siz;	/* size of a data       */
    int         length;	/* number of data       */
{
  register int    i;
  for (i = 0; i < length; i++)
    reverse((unsigned char*)&(data[i * siz]), siz);
}


static void
ReverseHead(head)
    Header  *head;
{
  register int    i;
  
  for (i = 0; i < 10; i++)
    reverse((unsigned char*)&(head->index[i]), sizeof(int));
  reverse((unsigned char*)&(head->samf), sizeof(float));
}
