/******************************************************************
**                                                               **
**      File Name : syscom.c                                     **
**                                                               **
**          System Common Area Read / Write Routine              **
**                                                               **
**                                      Coded by Shigeru Hitomi  **
**                                                               **
******************************************************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "macro.h"
#define _LIB_SYSCOM_C_
#include "syscom.h"

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
 
#include <stdio.h>
#include <memory.h>
#include <signal.h>

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef UseSHARE
#include <sys/ipc.h>
#include <sys/shm.h>
#else
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#endif

#ifndef RETSIGTYPE
#define RETSIGTYPE  int
#endif

SystemCommon syscom;

/* dangerous choice to use shared memory!
 * #define UseSHARE	TRUE
 */

#define SC_SIZ		sizeof(SystemCommon)


/***** System Common Data File Pointer *****/
static int   syscom_fd = 0;

/***** Environment Variable Stack      *****/
static char  my_pid[ENV_VAR_LENGTH]     = {""};
static char  system_common[FILE_LENGTH] = {""};


int
make_syscom()
     /******************************
      * Making System Common Area  *
      ******************************/
{
 
  /***** Create System Common Data Area *****/
#ifdef UseSHARE
  char   cfd[10];
  syscom_fd = shmget(IPC_PRIVATE, SC_SIZ, 0644 | IPC_CREAT);
  if (syscom_fd == -1)
    return (-1);
  
  /***** Setting System Common File Discripter *****/
  sprintf(cfd, "%d", syscom_fd);
#ifdef  HAVE_PUTENV
  strcpy(system_common, "SYS_COMMON=");
  strcat(system_common, cfd);
  putenv(system_common);
#else
  strcpy(system_common, cfd);
  setenv("SYS_COMMON", system_common, 1);
#endif /* HAVE_PUTENV */
  
#else  /* Use TEMPORARY FILE */
  char            fname[FILE_LENGTH];
  sprintf(fname, "%s%s", get_tmpdir(), "SYSCOM");
  syscom_fd = open(fname, O_RDWR | O_CREAT | O_TRUNC, 0644);
  if (syscom_fd == -1)
    return (-1);
#ifdef HAVE_PUTENV
  strcpy(system_common, "SYS_COMMON=");
  strcat(system_common, fname);
  putenv(system_common);
#else
  strcpy(system_common, fname);
  setenv("SYS_COMMON", system_common, 1);
#endif /* HAVE_PUTENV */
#endif /* Use SHARE */
  
  return 0;
}


int
init_syscom()
     /**********************************************************
       Initialize      for System Common Data
       ***********************************************************/
{
  Bzero(&syscom, SC_SIZ);
  syscom.buff_leng     = 2048;
  syscom.sam_freq      = 1000.0;
  syscom.buffer_number = 64;

  return 0;
}


int
free_syscom()
{
  int flag = 0;

  if (syscom_fd != 0) {
    if ( close(syscom_fd) == 0 ) {
#ifdef UseSHARE
      (void) shmctl(syscom_fd, IPC_RMID, 0);
#else
      char fname[FILE_LENGTH];

      sprintf(fname, "%s%s", get_tmpdir(), "SYSCOM");
      flag = unlink(fname);
#endif
    }
  }
  return flag;
}


int
set_mypid(pid)
     int pid;
{
  int flag = 0;

#ifdef HAVE_PUTENV
  sprintf(my_pid, "SATELLITE=%d", pid);
  flag = putenv(my_pid);
#else
  sprintf(my_pid, "%d", pid);
  flag = setenv("SATELLITE", my_pid, 1);
#endif

  return flag;
}


int
set_tmpdir(dir)
     char *dir;
{
  strcpy(syscom.temp_dir, dir);
  return 0;
}


char *
get_tmpdir()
{
  return syscom.temp_dir;
}


int
set_sampling(freq)
     double  freq;
{
  syscom.sam_freq = freq;
  return 0;
}


double
get_sampling()
{
  return syscom.sam_freq;
}

int
NextBuffer()
{
  return syscom.buffer_number++;
}


int
read_syscom()
{
#ifdef UseSHARE
  SystemCommon   *area;
#endif
  char       *buf;
  int         status = 0;
  RETSIGTYPE (*istat) (), (*qstat) ();

  istat = (RETSIGTYPE (*) ()) signal(SIGINT,  SIG_IGN);
  qstat = (RETSIGTYPE (*) ()) signal(SIGQUIT, SIG_IGN);
  
#ifdef UseSHARE
  if (syscom_fd == 0) {		/* for External Command */
    buf = (char*)getenv("SYS_COMMON");
    if (buf) {
      strcpy(system_common, buf);
      syscom_fd = atoi(system_common);
    } else
      status = -1;
  }
  area = (SystemCommon *) shmat(syscom_fd, 0, 0);
  Bcopy(area, &syscom, SC_SIZ);
  (void) shmdt(area);
#else  /* Use TEMPORARY FILE */
  if (syscom_fd == 0) {		/* for External Command */
    buf = (char*)getenv("SYS_COMMON");
    if (buf) {
      strcpy(system_common, buf);
      if ((syscom_fd = open(system_common, O_RDWR)) == -1)
	status = -1;
    } else
      status = -1;
  }
  if (status == 0) {
    lseek(syscom_fd, 0L, 0);
    if (read(syscom_fd, (char *) &syscom, SC_SIZ) != SC_SIZ)
      status = -1;
  }
#endif

  if ( status == -1 ) {
    fprintf( stderr, "FAILED TO READ SYSTEM COMMON FILE\n");
    exit(0);
  }
  
  signal(SIGINT,  (RETSIGTYPE (*)())istat);
  signal(SIGQUIT, (RETSIGTYPE (*)())qstat);
  return status;
}


int
write_syscom()
{
#ifdef UseSHARE
  SystemCommon   *area;
#endif

  int             status = 0;
  char           *buf;
  RETSIGTYPE     (*istat) (), (*qstat) ();

  istat = (RETSIGTYPE (*) ()) signal(SIGINT,  SIG_IGN);
  qstat = (RETSIGTYPE (*) ()) signal(SIGQUIT, SIG_IGN);
  
#ifdef UseSHARE
  if (syscom_fd == 0) {		/* for External Command */
    buf = (char*)getenv("SYS_COMMON");
    if (buf) {
      strcpy(system_common, getenv("SYS_COMMON"));
      syscom_fd = atoi(system_common);
    } else
      status = -1;
  }
  area = (SystemCommon *) shmat(syscom_fd, 0, 0);
  Bcopy(&syscom, area, SC_SIZ);
  (void) shmdt(area);
#else
  if (syscom_fd == 0) {		/* for External Command */
    buf = (char*)getenv("SYS_COMMON");
    if (buf) {
      strcpy(system_common, getenv("SYS_COMMON"));
      if ((syscom_fd = open(system_common, O_RDWR)) == -1)
	status = -1;
    } else
      status = -1;
  }
  if (status == 0) {
    lseek(syscom_fd, 0L, 0);
    if (write(syscom_fd, (char *) &syscom, SC_SIZ) != SC_SIZ)
      status = -1;
  }
#endif

  if ( status == -1 ) {
    fprintf( stderr, "FAILED TO WRITE SYSTEM COMMON FILE\n");
    exit(0);
  }

  signal(SIGINT,  (RETSIGTYPE (*)()) istat);
  signal(SIGQUIT, (RETSIGTYPE (*)()) qstat);
  return status;
}
