/*****************************************************************************
  
   	  Function Library to Control NCS Simulation Condition File 
  	       	   for information of DELAY variables

    $Id: scfdly.c,v 1.1.1.1 2004/03/26 14:57:15 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <string.h>
#include <fcntl.h>

#include "ncsdef.h"
#include "common.h"
#include "scflib.h"
#include "scfn.h"
#include "scfdly.h"

/* File pointer for simulaiton condition file of output information */
FILE	*dly;

static int   DlyOpenSCF		_ANSI_ARGS_(( void ));
static int   DlyCreateSCF	_ANSI_ARGS_(( void ));
static int   DlyCloseSCF	_ANSI_ARGS_(( void ));
static int   DlyWriteSCF	_ANSI_ARGS_(( DlyFile *src ));
static int   DlyReadSCF		_ANSI_ARGS_(( DlyFile *src ));
static void  DlyTopSCF		_ANSI_ARGS_(( void ));
static void  DlyEndSCF		_ANSI_ARGS_(( void ));
static void  DlyNextSCF		_ANSI_ARGS_(( void ));
static void  DlyNthNextSCF	_ANSI_ARGS_(( int ));
static void  DlyPreviousSCF	_ANSI_ARGS_(( void ));
static void  DlyNthPreviousSCF	_ANSI_ARGS_(( int ));
static int   DlyListupAllSCF	_ANSI_ARGS_(( FILE *disp ));

/**************************************************************************
  	Basic functions to control simulation condition file
 **************************************************************************/


/*
 *  Open delay condition file to read and write
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int 
DlyOpenSCF()
{
	char		*path;
	char 		mname[SCF_SCFN_PATH_SIZE];
	char 		fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_DLY_EXTN );
	} else {
	  return( FALSE );
	}

	if( (dly = OpenSCF( fname )) == NULL ){
	  return( FALSE );
	} else {
	  return( TRUE );
	}
}

/*
 *  Create delay condition file
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int
DlyCreateSCF()
{
	char		*path;
	char 		mname[SCF_SCFN_PATH_SIZE];
	char 		fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_DLY_EXTN );
	} else {
	  return( FALSE );
	}

	if( (dly = CreateSCF( fname )) == NULL ){
	  return( FALSE );
	} else {
	  return( TRUE );
	}
}

/*
 *  Close delay condition file
 */
static int
DlyCloseSCF( void )
{
  	return( CloseSCF( dly ) );
}

/*
 *  Write one entry to delay condition file
 * 	OutFile *src: reading at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE  : success.
 *                    FALSE : failed.
 */
static int
DlyWriteSCF( src )
DlyFile *src;
{
  	if( WriteSCF( dly, (void *)src, sizeof(DlyFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Read one entry from delay condition file
 * 	OutFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int
DlyReadSCF( src )
DlyFile *src;
{
  	if( ReadSCF( dly, (void *)src, sizeof(DlyFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Set file pointer top of file
 */
static void
DlyTopSCF( void )
{
  	fseek( dly, 0L, SEEK_SET );
}

/*
 *  Set file pointer end of file
 */
static void
DlyEndSCF( void )
{
  	DlyFile	tmp;
	DlyTopSCF();
	while( DlyReadSCF( &tmp ) != FALSE && tmp.type != SCF_EOF );
	fseek( dly, (long)sizeof(DlyFile)*(-1), SEEK_CUR );
}

/*
 *  advance file pointer 
 */
static void
DlyNextSCF( void )
{
  	fseek( dly, (long)sizeof(DlyFile), SEEK_CUR );
}

/*
 *  advance nth file pointer 
 */
static void
DlyNthNextSCF( n )
int 	n;
{
  	int	i;

	for( i = 0; i < n; i++ ){
	  DlyNextSCF();
	}
}

/*
 *  retreat file pointer 
 */
static void
DlyPreviousSCF( void )
{
  	fseek( dly, (long)sizeof(DlyFile)*(-1), SEEK_CUR );
}

/*
 *  retreat nth file pointer 
 */
static void
DlyNthPreviousSCF( n )
int	n;
{
  	int	i;

	for( i = 0; i < n; i++ ){
	  DlyPreviousSCF();
	}
}


/**************************************************************************
       Functions to write simulation condition file
 **************************************************************************/

/*
 *  Make output information file
 *      ( ->SCF_EOF )
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
DlyMakeSCF()
{
  	DlyFile 	src;

	if( DlyCreateSCF() == FALSE ){
	  return( FALSE );
	}
	
	src.type = SCF_EOF;
	if( DlyWriteSCF( &src ) == FALSE ){
	  DlyCloseSCF();
	  return( FALSE );
	}
	
	DlyCloseSCF();
	return( TRUE );
}

/*
 *  Add one entry to end of delay condition file
 * 	OutFile *add: add informaiton
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
DlyAddInfoSCF( add )
DlyFile *add;
{
	DlyFile src;

	if( DlyOpenSCF() == FALSE ){
	  return( FALSE );
	}
  	
	DlyEndSCF();
  	if( DlyWriteSCF( add ) == SCF_ERROR ){
	  DlyCloseSCF();
	  return( FALSE );
	}

	src.type = SCF_EOF;
  	if( DlyWriteSCF( &src ) == SCF_ERROR ){
	  DlyCloseSCF();
	  return( FALSE );
	}

	DlyCloseSCF();
	return( TRUE );
}

/*
 *  Write one entry to n position of delay condition file 
 * 	OutFile *add: add informaiton
 * 	int n       : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
DlyPutInfoSCF( add, n )
DlyFile *add;
int	n;
{
	if( DlyOpenSCF() == FALSE ){
	  return( FALSE );
	}
  	
	DlyTopSCF();
	DlyNthNextSCF( n );
  	if( DlyWriteSCF( add ) == SCF_ERROR ){
	  DlyCloseSCF();
	  return( FALSE );
	}

	DlyCloseSCF();
	return( TRUE );
}

/*
 *  Set one entry to delay condition structure.
 * 	OutFile *src    : storing at the location 
 *	int buff_id     : buffer ID to set information.
 *	char *buff_name : buffer name to set information.
 *      int buff_dim    : index of buffer to set information.
 *	char *mdl_name  : module name to set information.
 *      int comp_no     : component number to set information.
 *      int type        : type of output variable. 
 *	                  ( output, input, observable )
 *	char *obs_name  : name of variable in model file.
 *      Return          : TRUE  : success.
 *                        FALSE : failed.
 */
int
DlySetInfoSCF( src, mdl_name, inp_name, time, init_out, flag )
DlyFile *src;
char 	*mdl_name;
char 	*inp_name;
double 	time;
double	init_out;
char	*flag;
{
  	src->type = SCF_DLY_TYPE_NORMAL;
	if( mdl_name != NULL ){
	  if( strlen( mdl_name ) >= ( SCF_NAME_LEN-1 ) ){
	    strncpy( src->delay.mdl_name, mdl_name, SCF_NAME_LEN-1 );
	    src->delay.mdl_name[SCF_NAME_LEN-1] = '\0';
	    fprintf( stderr, "Warnning: model name is too long.\n" );
	  }else{
	    strcpy( src->delay.mdl_name, mdl_name );
	  }
	} else {
	  src->delay.mdl_name[0] = '\0';
	}
	  
	if( inp_name != NULL ){
	  if( strlen( inp_name ) >= (SCF_NAME_LEN-1) ){
	    strncpy( src->delay.inp_name, inp_name, SCF_NAME_LEN-1 );
	    src->delay.inp_name[SCF_NAME_LEN-1] = '\0';
	    fprintf( stderr, "Warnning: module name is too long.\n" );
	  }else{
	    strcpy( src->delay.inp_name, inp_name );
	  }
	} else {
	  src->delay.inp_name[0] = '\0';
	}

	if( time >= 0 ){
	  src->delay.time = time;
	} else {
	  src->delay.time = 0.0;
	}
	src->delay.init_flag = *flag;
	if( *flag == SCF_DLY_FLAG_FIX ){
	  src->delay.init_out = init_out;
	} else {
	  src->delay.init_out = 0.0;
	}

	return( TRUE );
}

/*
 *  Write one entry to delay condition file
 * 	DlyFile *src: storing at the location 
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
DlyWriteInfoSCF( src )
DlyFile *src;
{
	int             pos;

	pos = DlySearchInfoSCF( src );
	if( pos < 0 ){
	  return( DlyAddInfoSCF( src ) );
	} else {
	  return( DlyPutInfoSCF( src, pos ) );
	}
}



/**************************************************************************
       Functions to read simulation condition file
 **************************************************************************/

/*
 *  Read one entry from n potition of delay condition file
 * 	DlyFile *src: storing at the location 
 *      int  n      : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
DlyGetInfoSCF( src, n )
DlyFile *src;
int	n;
{
	if( DlyOpenSCF() == FALSE ){
	  return( FALSE );
	}

	DlyTopSCF();
	DlyNthNextSCF( n );
  	if( DlyReadSCF( src ) == FALSE ){
	  DlyCloseSCF();
	  return( FALSE );
	}else{
	  DlyCloseSCF();
	  return( TRUE );
	}
}

/*
 *  Search same entry to delay condition file
 * 	DlyFile *cmp: compare entry
 *      Return      :  n > 0  : potion of the same entry
 *                     n < 0  : no match. n is end of entry.
 *                     0      : ERROR
 */
int
DlySearchInfoSCF( cmp )
DlyFile *cmp;
{
	DlyFile src;
	int     n = 0;

	if( DlyOpenSCF() == FALSE ){
	  return( 0 );
	}
	DlyTopSCF();

	while( DlyReadSCF( &src ) != SCF_ERROR ){
	  if( src.type == SCF_EOF ){
	    DlyCloseSCF();	
	    return( -(n+1) );
	    break;
	  }

	  if( ( cmp->delay.mdl_name == src.delay.mdl_name ) &&
	     ( cmp->delay.inp_name == src.delay.inp_name ) ){
	    break;
	  }
	  
	  n++;
	}

	DlyCloseSCF();	
	return( n );
}


/**************************************************************************
       Functions to delete entry of simulation condition file
 **************************************************************************/

/*
 *  Delete one entry from n potition of delay condition file
 *      int  n      : position   
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
DlyDeleteInfoSCF( n )
int	n;
{
  	DlyFile	tmp;

	if( DlyOpenSCF() == FALSE ){
	  return( FALSE );
	}

	DlyTopSCF();
	DlyNthNextSCF( n );
	do{
	  DlyNextSCF();
	  if( DlyReadSCF( &tmp ) == FALSE ){
	    DlyCloseSCF();
	    return( FALSE );
	  }
	  DlyNthPreviousSCF( 2 );
	  if( DlyWriteSCF( &tmp ) == FALSE ){
	    DlyCloseSCF();
	    return( FALSE );
	  }
	}while( tmp.type != SCF_EOF );
	
	DlyCloseSCF();
	return( TRUE );
}


/*
 *  Delete all entry of delay condition file
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
DlyDeleteAllInfoSCF()
{
  	DlyFile src;

	if( DlyOpenSCF() == FALSE ){
	  return( FALSE );
	}

	DlyTopSCF();
	while( ( DlyReadSCF( &src ) != FALSE ) &&
	       ( src.type != SCF_EOF ) ){
	  DlyPreviousSCF();
	  src.type = SCF_EOF;
	  if( DlyWriteSCF( &src ) == FALSE ){
	    DlyCloseSCF();
	    return( FALSE );
	  }
	}
	
	DlyCloseSCF();
	return( TRUE );
}

/**************************************************************************
       Functions to display content of simulation condition file
 **************************************************************************/


/*
 *  Print list of outupt variables 
 * 	FILE *disp  : output file
 */
static int
DlyListupAllSCF( disp )
FILE	*disp;     
{
	int     num = 0;
	int	row = 0;
	DlyFile	src;

	if( DlyOpenSCF() == FALSE ){
	  return( FALSE );
	}

	DlyTopSCF();
	while( DlyReadSCF( &src ) != FALSE && src.type != SCF_EOF ){
	  if( num == 0 ){
	    fprintf( disp, 
		    " Data No.,         Input Name,        Delay Time,    Initial Output\n");
	  }
	  num++;
	  fprintf( disp, " %3d          %9s(%9s)      %-8g", num,
		  src.delay.mdl_name, src.delay.inp_name, src.delay.time );

	  switch( src.delay.init_flag ){
	    case SCF_DLY_FLAG_AUTO:
	      fprintf( disp, "      %8s\n", "(Auto)");
	      row++;
	      break;
	    case SCF_DLY_FLAG_FIX:
	      fprintf( disp, "      %-8g\n",src.delay.init_out);
	      row++;
	      break;
  	    default:
	      break;
	  }
	}			/* while */
	if( num == 0 ){
	  fprintf( disp, "\tNo information.\n" );
	  row++;
	}

	DlyCloseSCF();
	return( row );
}


/*
 *  Print list of all entry of delay information file
 * 	FILE *disp  : output file
 *      Return      : success : number of lines of lists
 *                    0       : failed
 */
int
DlyListupSCF( disp )
FILE	*disp;
{
  	int	row = 1;

	fprintf( disp, "\nDELAY INFORMATION\n");
  
	row += DlyListupAllSCF( disp );

	return( row );
}

