/*****************************************************************************
  
  	Function Library to Control NCS Simulation Condition File 
  	       for information of MATHEMATICAL variables

    $Id: scfmat.c,v 1.1.1.1 2004/03/26 14:57:15 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <string.h>
#include <fcntl.h>

#include "ncsdef.h"
#include "common.h"
#include "scflib.h"
#include "scfn.h"
#include "scfmat.h"

/* File pointer for simulaiton condition file of mathematical information */
FILE	*mat;

static int   MatOpenSCF		_ANSI_ARGS_(( void ));
static int   MatCreateSCF	_ANSI_ARGS_(( void ));
static int   MatCloseSCF	_ANSI_ARGS_(( void ));
static int   MatWriteSCF	_ANSI_ARGS_(( MatFile *src ));
static int   MatReadSCF		_ANSI_ARGS_(( MatFile *src ));
static void  MatTopSCF		_ANSI_ARGS_(( void ));
static void  MatEndSCF		_ANSI_ARGS_(( void ));
static void  MatNextSCF		_ANSI_ARGS_(( void ));
static void  MatNthNextSCF	_ANSI_ARGS_(( int n ));
static void  MatPreviousSCF	_ANSI_ARGS_(( void ));
static void  MatNthPreviousSCF	_ANSI_ARGS_(( int n ));

/**************************************************************************
  	Basic functions to control simulation condition file
 **************************************************************************/


/*
 *  Open mathematical information file to read and write
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int 
MatOpenSCF()
{
	char		*path;
	char 		mname[SCF_SCFN_PATH_SIZE];
	char 		fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_MAT_EXTN );
	} else {
	  return( FALSE );
	}

	if( (mat = OpenSCF( fname )) == NULL ){
	  return( FALSE );
	} else {
	  return( TRUE );
	}
}

/*
 *  Create mathematical information file
 * 	char *fname : file name of file
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int
MatCreateSCF()
{
	char		*path;
	char 		mname[SCF_SCFN_PATH_SIZE];
	char 		fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_MAT_EXTN );
	} else {
	  return( FALSE );
	}

	if( (mat = CreateSCF( fname )) == NULL ){
	  return( FALSE );
	} else {
	  return( TRUE );
	}
}

/*
 *  Close mathematical information file
 */
static int
MatCloseSCF( void )
{
  	return( CloseSCF( mat ) );
}

/*
 *  Write one entry to mathematical information file
 * 	MatFile *src: reading at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE  : success.
 *                    FALSE : failed.
 */
static int
MatWriteSCF( src )
MatFile *src;
{
  	if( WriteSCF( mat, (void *)src, sizeof(MatFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Read one entry from mathematical information file
 * 	MatFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int
MatReadSCF( src )
MatFile *src;
{
  	if( ReadSCF( mat, (void *)src, sizeof(MatFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Set file pointer top of mathematical informatition file
 */
static void
MatTopSCF( void )
{
  	fseek( mat, 0L, SEEK_SET );
}

/*
 *  Set file pointer end of mathematical information file
 */
static void
MatEndSCF( void )
{
  	fseek( mat, (long)sizeof(MatFile)*(-1), SEEK_END );
}

/*
 *  advance file pointer of mathematical information file
 */
static void
MatNextSCF( void )
{
  	fseek( mat, (long)sizeof(MatFile), SEEK_CUR );
}

/*
 *  advance nth file pointer of mathematical information file
 */
static void
MatNthNextSCF( n )
int 	n;
{
  	int	i;

	for( i = 0; i < n; i++ ){
	  MatNextSCF();
	}
}

/*
 *  retreat file pointer of mathematical information file
 */
static void
MatPreviousSCF( void )
{
  	fseek( mat, (long)sizeof(MatFile)*(-1), SEEK_CUR );
}

/*
 *  retreat nth file pointer of mathematical information file
 */
static void
MatNthPreviousSCF( n )
int	n;
{
  	int	i;

	for( i = 0; i < n; i++ ){
	  MatPreviousSCF();
	}
}


/**************************************************************************
       Functions to write simulation condition file
 **************************************************************************/

/*
 *  Make mathematical information file
 *      ( SCF_EOF )
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
MatMakeSCF()
{
  	MatFile 	src;

	if( MatCreateSCF() == FALSE ){
	  return( FALSE );
	}

	src.type = SCF_EOF;
	if( MatWriteSCF( &src ) == FALSE ){
	  MatCloseSCF();
	  return( FALSE );
	}
	
	MatCloseSCF();
	return( TRUE );
}

/*
 *  Add one entry to end of mathematical information file
 * 	MatFile *add: add informaiton
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
MatAddInfoSCF( add )
MatFile *add;
{
	MatFile src;

	if( MatOpenSCF() == FALSE ){
	  return( FALSE );
	}
  	
	MatEndSCF();
  	if( MatWriteSCF( add ) == SCF_ERROR ){
	  MatCloseSCF();
	  return( FALSE );
	}

	src.type = SCF_EOF;
  	if( MatWriteSCF( &src ) == SCF_ERROR ){
	  MatCloseSCF();
	  return( FALSE );
	}

	MatCloseSCF();
	return( TRUE );
}

/*
 *  Write one entry to n position of mathematical information file 
 * 	MatFile *add: add informaiton
 * 	int n       : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
MatPutInfoSCF( add, n )
MatFile *add;
int	n;
{
	if( MatOpenSCF() == FALSE ){
	  return( FALSE );
	}
  	
	MatTopSCF();
	MatNthNextSCF( n );
  	if( MatWriteSCF( add ) == SCF_ERROR ){
	  MatCloseSCF();
	  return( FALSE );
	}

	MatCloseSCF();
	return( TRUE );
}

/*
 *  Set one entry to mathematical information structure.
 * 	MatFile *src        : storing at the location 
 *	double last_time    : Last time
 *	double calc_time    : Calculation time
 *      double store_time   : Store time
 *	double write_time   : Write time
 *      char integ_method   : integration method
 *      int integ_maxcell   : max cell
 *	double integ_relerr : relative error
 *      Return              : TRUE  : success.
 *                            FALSE : failed.
 */
int
MatSetInfoSCF( src, last_time, calc_time, store_time, write_time,
	      integ_method, integ_maxcell, integ_relerr )
MatFile *src;
double last_time, calc_time, store_time, write_time;
char *integ_method;
int integ_maxcell;
double integ_relerr;
{
  	src->type = SCF_MAT_TYPE_NORMAL;

  	src->timer.sim_timer[0] = last_time;
  	src->timer.sim_timer[1] = calc_time;
  	src->timer.sim_timer[2] = store_time;
  	src->timer.sim_timer[3] = write_time;

  	src->integral.method  = *integ_method;
  	src->integral.maxcell = integ_maxcell;
  	src->integral.relerr  = integ_relerr;
	return( TRUE );
}

/*
 *  Search same entry to mathematical information file
 * 	ObsFile *cmp: compare entry
 *      Return      :  n > 0  : potion of the same entry
 *                     n < 0  : no match. n is end of entry.
 *                     0      : ERROR
 */
int
MatSearchInfoSCF( cmp )
MatFile *cmp;
{
	MatFile src;
	int     n = 0;
	

	if( MatOpenSCF() == FALSE ){
	  return( 0 );
	}
	MatTopSCF();

	while( MatReadSCF( &src ) != SCF_ERROR ){
	  if( src.type == SCF_EOF ){
	    MatCloseSCF();	
	    return( -n );
	    break;
	  }

	  if( cmp->type == src.type ){
	    break;
	  }
	  
	  n++;
	}

	MatCloseSCF();	
	return( n );
}

/*
 *  Write one entry to mathematical information file
 * 	char *fname : file name
 * 	OutFile *src: storing at the location 
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
MatWriteInfoSCF( src )
MatFile *src;
{
	return( MatPutInfoSCF( src, 0 ) );
}



/**************************************************************************
       Functions to read simulation condition file
 **************************************************************************/

/*
 *  Read one entry from n potition of mathematical information file
 * 	char *fname : file name
 * 	MatFile *str: storing at the location 
 *      int  n      : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
MatGetInfoSCF( str, n )
MatFile *str;
int	n;
{
	if( MatOpenSCF() == FALSE ){
	  return( FALSE );
	}

	MatTopSCF();
	MatNthNextSCF( n );
  	if( MatReadSCF( str ) == FALSE ){
	  MatCloseSCF();
	  return( FALSE );
	}else{
	  MatCloseSCF();
	  return( TRUE );
	}
}


/**************************************************************************
       Functions to display content of simulation condition file
 **************************************************************************/

/*
 *  Print list of entry of mathematical information file
 *  Information of time of simulation
 * 	FILE *disp  : output file
 * 	char *fname : file name
 *      Return      : success : number of lines of lists
 *                    0       : failed
 */
int
MatTimeListupSCF( disp )
FILE	*disp;
{
  	MatFile	src;
	int 	row = 0;

	fprintf( disp, "\nTIMER\n" );

	if( MatOpenSCF() == FALSE ){
	  return( 0 );
	}

	MatTopSCF();

	if( MatReadSCF( &src ) != FALSE ){
	  if( src.type == SCF_MAT_TYPE_NORMAL ){
	    fprintf( disp,"  Last  Time = %8g\n", src.timer.sim_timer[0] );
	    fprintf( disp,"  Calc. Step = %8g\n", src.timer.sim_timer[1] );
	    fprintf( disp,"  Store Step = %8g\n", src.timer.sim_timer[2] );
	    fprintf( disp,"  BufferStep = %8g\n", src.timer.sim_timer[3] );
	    row +=4;
	  } else {
	    fprintf( disp,"Error: Can't find mathematical information in the condition file\n" );
	    row++;
	  }

	  fprintf( disp, "\n");
	  row++;

	  MatCloseSCF();

	} else {
	  fprintf( disp, "Error: Can't find the condition file of mathematical information\n" );
	  row++;
	}
	
	return( row );
}

/*
 *  Print list of entry of mathematical information file
 *  Information of integration
 * 	FILE *disp  : output file
 * 	char *fname : file name
 *      Return      : success : number of lines of lists
 *                    0       : failed
 */
int
MatIntegListupSCF( disp )
FILE	*disp;
{
  	MatFile	src;
	int 	row = 0;

	fprintf( disp, "\nINTEGRAL\n" );

	if( MatGetInfoSCF( &src, 0 ) != FALSE ){
	  fprintf( disp, "  Integration Method : " );
	  switch( src.integral.method ){
	    case SCF_MAT_INTEG_FERBERG:
	      fprintf( disp, "5th Fehlberg Method (Adaptive Integration)\n" );
	      row++;
	      break;
	    case SCF_MAT_INTEG_EULER:
	      fprintf( disp, "Euler Method\n" );
	      row++;
	      break;
	    case SCF_MAT_INTEG_RKG:
	      fprintf( disp, "4th Runge-Kutta-Gill Method\n" );
	      row++;
	      break;
	    }	/* switch */
	  fprintf( disp, "  Number of Max Cell : %d\n", src.integral.maxcell );
	  row++;
	  if( src.integral.method == SCF_MAT_INTEG_FERBERG ) {
	    fprintf( disp, "  Relative Error     : %g\n", 
		     src.integral.relerr );
	    row+=2;
	  }
	} else {
	  fprintf( disp,"Error: Can't find mathematical information in the condition file\n" );
	  row++;
	}

	return( row );
}
