/*****************************************************************************

  	Function Library to Control NCS Simulation Condition File 
   		for information of OBSERVABLE variables

    $Id: scfobs.c,v 1.1.1.1 2004/03/26 14:57:15 orrisroot Exp $

*****************************************************************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif /* HAVE_MALLOC_H */
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif /* HAVE_SYS_TYPES_H */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <string.h>

#include "common.h"
#include "scflib.h"
#include "scfn.h"
#include "scfobs.h"

/* File pointer for simulaiton condition file of observable information */
FILE	*obs;

static int   ObsOpenSCF		_ANSI_ARGS_(( void ));
static int   ObsCreateSCF	_ANSI_ARGS_(( void ));
static int   ObsCloseSCF	_ANSI_ARGS_(( void ));
static int   ObsReadSCF		_ANSI_ARGS_(( ObsFile *src ));
static int   ObsWriteSCF	_ANSI_ARGS_(( ObsFile *src ));
static void  ObsEndSCF		_ANSI_ARGS_(( void ));


/*
 *  Open observable information file
 *      Return      : TRUE  : success
 *                    FALSE : failed
 */
static int
ObsOpenSCF()
{
	char		*path;
	char 		mname[SCF_SCFN_PATH_SIZE];
	char 		fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_OBS_EXTN );
	} else {
	  return( FALSE );
	}

	if( (obs=OpenSCF(fname)) == NULL ){
	  return( FALSE );
	} else {
	  return( TRUE );
	}
}

/*
 *  Open observable information file
 * 	char *fname : file name of file
 *      Return      : TRUE  : sucess
 *                    FALSE : failed
 */
static int
ObsCreateSCF()
{
	char		*path;
	char 		mname[SCF_SCFN_PATH_SIZE];
	char 		fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_OBS_EXTN );
	} else {
	  return( FALSE );
	}

	if( (obs = CreateSCF( fname )) == NULL ){
	  return( FALSE );
	} else {
	  return( TRUE );
	}
}

/*
 *  Close observable information file
 */
static int
ObsCloseSCF()
{
  	return( CloseSCF( obs ) );
}

/*
 *  Write one entry to observable information file
 * 	ObsFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int
ObsWriteSCF( src )
ObsFile *src;
{
  	if( WriteSCF( obs, (void *)src, sizeof(ObsFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Read one entry from observable information file
 * 	ObsFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int
ObsReadSCF( src )
ObsFile *src;
{
  	if( ReadSCF( obs, (void *)src, sizeof(ObsFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Set file pointer end of file
 * 	FILE    *fp : file pointer of file
 */
static void
ObsEndSCF()
{
  	fseek( obs, (long)sizeof(ObsFile)*(-1), SEEK_END );
}


/*
 *  Make observable information file
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
ObsMakeSCF()
{
  	ObsFile 	src;

	if( ObsCreateSCF() == FALSE ){
	  return( FALSE );
	}
	
	src.type = SCF_EOF;
	if( ObsWriteSCF( &src ) == FALSE ){
	  ObsCloseSCF();
	  return( FALSE );
	}
	
	ObsCloseSCF();
	return( TRUE );
}


/*
 *  Add one entry to observable information file
 * 	char *fname : file name
 * 	ObsFile *add: add informaiton
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
ObsAddInfoSCF( add )
ObsFile *add;
{
	ObsFile src;

	if( ObsOpenSCF() == FALSE ){
	  return( FALSE );
	}
  	
	ObsEndSCF();
  	if( ObsWriteSCF( add ) == SCF_ERROR ){
	  ObsCloseSCF();
	  return( FALSE );
	}

	src.type = SCF_EOF;
  	if( ObsWriteSCF( &src ) == SCF_ERROR ){
	  ObsCloseSCF();
	  return( FALSE );
	}

	ObsCloseSCF();
	return( TRUE );
}

/*
 *  Read one entry from observable information file
 * 	char *fname : file name
 * 	ObsFile *src: storing at the location 
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
ObsGetInfoSCF( src )
ObsFile *src;
{
	if( ObsOpenSCF() == FALSE ){
	  return( FALSE );
	}

  	if( ObsReadSCF( src ) == FALSE ){
	  ObsCloseSCF();
	  return( FALSE );
	}else{
	  ObsCloseSCF();
	  return( TRUE );
	}
}

/*
 *  Search same entry to observable information file
 * 	char *fname : name of file
 * 	ObsFile *cmp: compare entry
 *      Return      :  n : potion of the same entry
 *                    -1 : no match
 */
int
ObsSearchInfoSCF( cmp )
ObsFile *cmp;
{
	ObsFile src;
	int     n = 0;

	if( ObsOpenSCF() == FALSE ){
	  return( FALSE );
	}
	
	while( ObsReadSCF( &src ) != SCF_ERROR ){
	  if( src.type == SCF_EOF ){
	    ObsCloseSCF();
	    return( -(n+1) );
	    break;
	  }
	  
	  if( (strcmp( cmp->info.module, src.info.module ) == 0) 
	      && (strcmp( cmp->info.var, src.info.var ) == 0) 
	      && (cmp->info.max_comp < src.info.max_comp) ){
	    break;
	  }
	  
	  n++;
	}

	ObsCloseSCF();
	return( n );
}


/*
 *  Print list of all entry to observable information file
 * 	FILE *disp  : output file
 * 	char *fname : file name
 */
int
ObsListupSCF( disp )
FILE	*disp;
{
	ObsFile src;
	int	n = 0, line = 0;
	char	*module = NULL;


	if( ObsOpenSCF() == FALSE ){
	  fprintf( stderr, "Wanning: Failed to open the SCF\n" );
	  return( 0 );
	}

	fprintf( disp, "\nOBSERVABLE VARIABLE" );
	line++;
	
	while( ObsReadSCF( &src ) != SCF_ERROR ){
	  if( src.type == SCF_EOF ){
	    break;
	  }

	  if( ( n == 0 ) || (strcmp( src.info.module, module ) != 0 ) ){
	    if( n != 0 ){
	      free( module );
	    }
	    module = strdup( src.info.module );
	    fprintf( disp, "\n  Module: \"%s[%d]\"\n    ", 
		     module, src.info.max_comp );
	    line += 2;
	  } else {
	    fprintf( disp, ", " );
	  }
	  
	  fprintf( disp, "%s", src.info.var );
	  n++;
	}

	if( n == 0 ){
	    fprintf( disp, "\n  none.\n\n" );
	    line+= 2;
	}else{
	  fprintf( disp, "\n\n" );
	  line++;
	}
	ObsCloseSCF();

	return( line );
}


/*
 *  Set one entry to observable information structure.
 * 	OutFile *src    : storing at the location 
 *	char *mdl_name  : module name to set information.
 *	char *var_name  : name of variable in model file.
 *      Return          : TRUE  : success.
 *                        FALSE : failed.
 */
int
ObsSetInfoSCF( src, mdl_name, comp, var_name )
ObsFile *src;
char    *mdl_name;
int	comp;
char	*var_name;
{
  	if( mdl_name != NULL ){
	  if( strlen( mdl_name ) >= ( SCF_NAME_LEN-1 ) ){
	    strncpy( src->info.module, mdl_name, SCF_NAME_LEN-1 );
	    src->info.module[SCF_NAME_LEN-1] = '\0';
	    fprintf( stderr, "Warnning: module name is too long.\n" );
	  }else{
	    strcpy( src->info.module, mdl_name );
	  }
	} else {
	  src->info.module[0] = '\0';
	}

	if( comp >= 0 ){
	  src->info.max_comp = comp;
	} else {
	  src->info.max_comp = 0;
	}

	if( var_name != NULL ){
	  if( strlen( var_name ) >= ( SCF_NAME_LEN-1 ) ){
	    strncpy( src->info.var, var_name, SCF_NAME_LEN-1 );
	    src->info.var[SCF_NAME_LEN-1] = '\0';
	    fprintf( stderr, "Warnning: module name is too long.\n" );
	  }else{
	    strcpy( src->info.var, var_name );
	  }
	} else {
	  src->info.var[0] = '\0';
	}
	return( TRUE );
}
