/*****************************************************************************
  
   	  Function Library to Control NCS Simulation Condition File 
  	       for information of OUTPUT & OBSERVABLE variables

    $Id: scfout.c,v 1.1.1.1 2004/03/26 14:57:15 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <string.h>
#include <fcntl.h>

#include "ncsdef.h"
#include "common.h"
#include "scflib.h"
#include "scfn.h"
#include "scfout.h"

/* File pointer for simulaiton condition file of output information */
FILE	*out;

static int   OutOpenSCF		 _ANSI_ARGS_(( void ));
static int   OutCreateSCF	 _ANSI_ARGS_(( void ));
static int   OutCloseSCF	 _ANSI_ARGS_(( void ));
static int   OutWriteSCF	 _ANSI_ARGS_(( OutFile *src ));
static int   OutReadSCF		 _ANSI_ARGS_(( OutFile *src ));
static void  OutTopSCF		 _ANSI_ARGS_(( void ));
static void  OutEndSCF		 _ANSI_ARGS_(( void ));
static void  OutNextSCF		 _ANSI_ARGS_(( void ));
static void  OutNthNextSCF	 _ANSI_ARGS_(( int n ));
static void  OutPreviousSCF	 _ANSI_ARGS_(( void ));
static void  OutNthPreviousSCF	 _ANSI_ARGS_(( int n ));
static int   OutListupVarSCF	 _ANSI_ARGS_(( FILE *disp ));
static int   OutListupMonitorSCF _ANSI_ARGS_(( FILE *disp, OutFile *src ));
static int   OutListupEnvSCF	 _ANSI_ARGS_(( FILE *disp ));

/**************************************************************************
  	Basic functions to control simulation condition file
 **************************************************************************/


/*
 *  Open output information file to read and write
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int 
OutOpenSCF()
{
	char		*path;
	char 		mname[SCF_SCFN_PATH_SIZE];
	char 		fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_OUT_EXTN );
	} else {
	  return( FALSE );
	}

	if( (out = OpenSCF( fname )) == NULL ){
	  return( FALSE );
	} else {
	  return( TRUE );
	}
}

/*
 *  Create output information file
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int
OutCreateSCF()
{
	char		*path;
	char 		mname[SCF_SCFN_PATH_SIZE];
	char 		fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_OUT_EXTN );
	} else {
	  return( FALSE );
	}

	if( (out = CreateSCF( fname )) == NULL ){
	  return( FALSE );
	} else {
	  return( TRUE );
	}
}

/*
 *  Close output information file
 */
static int
OutCloseSCF( void )
{
  	return( CloseSCF( out ) );
}

/*
 *  Write one entry to output information file
 * 	OutFile *src: reading at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE  : success.
 *                    FALSE : failed.
 */
static int
OutWriteSCF( src )
OutFile *src;
{
  	if( WriteSCF( out, (void *)src, sizeof(OutFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Read one entry from output information file
 * 	OutFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int
OutReadSCF( src )
OutFile *src;
{
  	if( ReadSCF( out, (void *)src, sizeof(OutFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Set file pointer top of file
 */
static void
OutTopSCF( void )
{
  	fseek( out, 0L, SEEK_SET );
	OutNextSCF();
}

/*
 *  Set file pointer end of file
 */
static void
OutEndSCF( void )
{
  	OutFile	tmp;
	OutTopSCF();
	while( OutReadSCF( &tmp ) != FALSE && tmp.type != SCF_EOF );
	fseek( out, (long)sizeof(OutFile)*(-1), SEEK_CUR );
  /* 	fseek( out, (long)sizeof(OutFile)*(-1), SEEK_END );*/
}

/*
 *  advance file pointer 
 */
static void
OutNextSCF( void )
{
  	fseek( out, (long)sizeof(OutFile), SEEK_CUR );
}

/*
 *  advance nth file pointer 
 */
static void
OutNthNextSCF( n )
int 	n;
{
  	int	i;

	for( i = 0; i < n; i++ ){
	  OutNextSCF();
	}
}

/*
 *  retreat file pointer 
 */
static void
OutPreviousSCF( void )
{
  	fseek( out, (long)sizeof(OutFile)*(-1), SEEK_CUR );
}

/*
 *  retreat nth file pointer 
 */
static void
OutNthPreviousSCF( n )
int	n;
{
  	int	i;

	for( i = 0; i < n; i++ ){
	  OutPreviousSCF();
	}
}


/**************************************************************************
       Functions to write simulation condition file
 **************************************************************************/

/*
 *  Make output information file
 *      ( SCF_OUT_NO->SCF_OUT_BUFF_MAX->SCF_EOF )
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
OutMakeSCF()
{
  	OutFile 	src;

	if( OutCreateSCF() == FALSE ){
	  return( FALSE );
	}

	src.type = SCF_OUT_NO;
	if( OutWriteSCF( &src ) == FALSE ){
	  OutCloseSCF();
	  return( FALSE );
	}
	
	src.type = SCF_OUT_BUFF_SIZE;
	src.out.rec4.max_number = SCF_OUT_BUFF_MAX;
	/*OutTopSCF( fp );*/
	if( OutWriteSCF( &src ) == FALSE ){
	  OutCloseSCF();
	  return( FALSE );
	}
	
	src.type = SCF_EOF;
	if( OutWriteSCF( &src ) == FALSE ){
	  OutCloseSCF();
	  return( FALSE );
	}
	
	OutCloseSCF();
	return( TRUE );
}

/*
 *  Add one entry to end of output information file
 * 	OutFile *add: add informaiton
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
OutAddInfoSCF( add )
OutFile *add;
{
	OutFile src;

	if( OutOpenSCF() == FALSE ){
	  return( FALSE );
	}
  	
	OutEndSCF();
  	if( OutWriteSCF( add ) == SCF_ERROR ){
	  OutCloseSCF();
	  return( FALSE );
	}

	src.type = SCF_EOF;
  	if( OutWriteSCF( &src ) == SCF_ERROR ){
	  OutCloseSCF();
	  return( FALSE );
	}

	OutCloseSCF();
	return( TRUE );
}

/*
 *  Write one entry to n position of output information file 
 * 	OutFile *add: add informaiton
 * 	int n       : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
OutPutInfoSCF( add, n )
OutFile *add;
int	n;
{
	if( OutOpenSCF() == FALSE ){
	  return( FALSE );
	}
  	
	OutTopSCF();
	OutNthNextSCF( n );
  	if( OutWriteSCF( add ) == SCF_ERROR ){
	  OutCloseSCF();
	  return( FALSE );
	}

	OutCloseSCF();
	return( TRUE );
}

/*
 *  Set one entry to output information structure.
 * 	OutFile *src    : storing at the location 
 *	int buff_id     : buffer ID to set information.
 *	char *buff_name : buffer name to set information.
 *      int buff_dim    : index of buffer to set information.
 *	char *mdl_name  : module name to set information.
 *      int comp_no     : component number to set information.
 *      int type        : type of output variable. 
 *	                  ( output, input, observable )
 *	char *obs_name  : name of variable in model file.
 *      Return          : TRUE  : success.
 *                        FALSE : failed.
 */
int
OutSetInfoSCF( src, buf_id, buf_name, buf_dim, mdl_name, 
	       comp_no, type, obs_name )
OutFile *src;
int     buf_id, buf_dim;
char    *buf_name;
char    *mdl_name;
int	comp_no, type;
char	*obs_name;
{
	src->out.rec3.buff_num = buf_id;
	if( buf_name != NULL ){
	  if( strlen( buf_name ) >= ( SCF_NAME_LEN-1 ) ){
	    strncpy( src->out.rec3.buf_name, buf_name, SCF_NAME_LEN-1 );
	    src->out.rec3.buf_name[SCF_NAME_LEN-1] = '\0';
	    fprintf( stderr, "Warnning: buffer name is too long.\n" );
	  }else{
	    strcpy( src->out.rec3.buf_name, buf_name );
	  }
	} else {
	  src->out.rec3.buf_name[0] = '\0';
	}
	  
	src->out.rec3.buff_dim = buf_dim;
	if( mdl_name != NULL ){
	  if( strlen( mdl_name ) >= (SCF_NAME_LEN-1) ){
	    strncpy( src->out.rec3.mdl_name, mdl_name, SCF_NAME_LEN-1 );
	    src->out.rec3.mdl_name[SCF_NAME_LEN-1] = '\0';
	    fprintf( stderr, "Warnning: module name is too long.\n" );
	  }else{
	    strcpy( src->out.rec3.mdl_name, mdl_name );
	  }
	} else {
	  src->out.rec3.mdl_name[0] = '\0';
	}
	 
	src->out.rec3.comp_number = comp_no;
	src->type = (char)('0'+type);
	if( src->type == SCF_OUT_BUFF_OBS ){
	  if( obs_name != NULL ){
	    if( strlen( obs_name ) >= (SCF_NAME_LEN)-1 ){
	      strncpy( src->out.rec3.obs_name, obs_name, SCF_NAME_LEN );
	      src->out.rec3.obs_name[SCF_NAME_LEN-1] = '\0';
	      fprintf( stderr, "Warnning: variable name is too long.\n" );
	    }else{
	      strcpy( src->out.rec3.obs_name, obs_name );
	    }
	  } else {
	    src->out.rec3.obs_name[0] = '\0';
	  }
	}
	return( TRUE );
}

/*
 *  Write one entry to output information file
 * 	OutFile *src: storing at the location 
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
OutWriteInfoSCF( src )
OutFile *src;
{
	int             pos;

	pos = OutSearchInfoSCF( src );
	if( pos < 0 ){
	  return( OutAddInfoSCF( src ) );
	} else {
	  return( OutPutInfoSCF( src, pos ) );
	}
}



/**************************************************************************
       Functions to read simulation condition file
 **************************************************************************/

/*
 *  Read one entry from n potition of output information file
 * 	OutFile *src: storing at the location 
 *      int  n      : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
OutGetInfoSCF( src, n )
OutFile *src;
int	n;
{
	if( OutOpenSCF() == FALSE ){
	  return( FALSE );
	}

	OutTopSCF();
	OutNthNextSCF( n );
  	if( OutReadSCF( src ) == FALSE ){
	  OutCloseSCF();
	  return( FALSE );
	}else{
	  OutCloseSCF();
	  return( TRUE );
	}
}

/*
 *  Search same entry to observable information file
 * 	ObsFile *cmp: compare entry
 *      Return      :  n > 0  : potion of the same entry
 *                     n < 0  : no match. n is end of entry.
 *                     0      : ERROR
 */
int
OutSearchInfoSCF( cmp )
OutFile *cmp;
{
	OutFile src;
	int     n = 0;
	

	if( OutOpenSCF() == FALSE ){
	  return( 0 );
	}
	OutTopSCF();

	while( OutReadSCF( &src ) != SCF_ERROR ){
	  if( src.type == SCF_EOF ){
	    OutCloseSCF();	
	    return( -(n+1) );
	    break;
	  }

	  if( strcmp(cmp->out.rec3.buf_name,src.out.rec3.buf_name) == 0 ){
	    if( cmp->out.rec3.buff_dim == SCF_ALL_MATCH ){
	      break;
	    } else {
	      if( cmp->out.rec3.buff_dim == src.out.rec3.buff_dim ){
		break;
	      }
	    }
	  }
	  
	  n++;
	}

	OutCloseSCF();	
	return( n );
}

/**************************************************************************
       Functions to delete entry of simulation condition file
 **************************************************************************/

/*
 *  Delete one entry from n potition of output information file
 *      int  n      : position   
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
OutDeleteInfoSCF( n )
int	n;
{
  	OutFile	tmp;

	if( OutOpenSCF() == FALSE ){
	  return( FALSE );
	}

	OutTopSCF();
	OutNthNextSCF( n );
	do{
	  OutNextSCF();
	  if( OutReadSCF( &tmp ) == FALSE ){
	    OutCloseSCF();
	    return( FALSE );
	  }
	  OutNthPreviousSCF( 2 );
	  if( OutWriteSCF( &tmp ) == FALSE ){
	    OutCloseSCF();
	    return( FALSE );
	  }
	}while( tmp.type != SCF_EOF );
	
	OutCloseSCF();
	return( TRUE );
}


/*
 *  Delete all entry of output information file
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
OutDeleteAllInfoSCF()
{
  	OutFile src;
	int	tmp;

	if( OutOpenSCF() == FALSE ){
	  return( FALSE );
	}

	OutTopSCF();
	OutNextSCF();	/* skip information of BUFF MAX */
	src.type = SCF_EOF;
	if( OutWriteSCF( &src ) == FALSE ){
	  return( FALSE );
	}
	while( (tmp=OutReadSCF( &src )) != FALSE ){
	  if( src.type == SCF_EOF ){
	    break;
	  }
	  OutPreviousSCF();
	  src.type = SCF_ERASED;
	  if( OutWriteSCF( &src ) == FALSE ){
	    OutCloseSCF();
	    return( FALSE );
	  }
	}
	
	OutCloseSCF();
	return( tmp );
}

int 
OutRemoveInfoSCF( src, flag )
OutFile	*src;
int flag;
{
	int	dnum;
	int     dflag = 0;

	/* one or more Buffer */
	if( flag != SCF_ALL_MATCH ){
	  while( (dnum = OutSearchInfoSCF( src )) > 0 ){
	    dflag++;
	    if( OutDeleteInfoSCF( dnum ) == FALSE ){
	      return( FALSE );
	    }
	  }
	  if( dflag == 0 ){
	    return( FALSE );
	  } else {
	    return( TRUE );
	  }
	}	
	/* all Buffer */
	return( OutDeleteAllInfoSCF() );
}


/**************************************************************************
       Functions to display content of simulation condition file
 **************************************************************************/


/*
 *  Print list of outupt variables 
 * 	FILE *disp  : output file
 */
static int
OutListupVarSCF( disp )
FILE	*disp;     
{
	int     num = 0;
	int	row = 0;
	int 	flag;
	OutFile	src;

	OutTopSCF();
	OutNextSCF();	/* skip information of BUFF MAX */
	
	while( OutReadSCF( &src ) != FALSE && src.type != SCF_EOF ){
	  flag = 0;

	  if( num == 0 ){
	    fprintf( disp, "  Variable       Model Variable\n");
	    row++;
	  }
	  num++;

	  if( src.type != SCF_ERASED && src.type != SCF_OUT_NO ){
	    fprintf( disp, "%10s[%3d]   ",
		     src.out.rec3.buf_name, src.out.rec3.buff_dim );
	    
	    switch( src.type ){
	      case SCF_OUT_BUFF_OUT:
		fprintf( disp, "OUTPUT   of ");
		break;

	      case SCF_OUT_BUFF_EXIN:
		fprintf( disp, "EX.INPUT of ");
		break;

	      case SCF_OUT_BUFF_OBS:
		fprintf( disp, "%-8s of ", src.out.rec3.obs_name);
		break;
		
	      default:
		flag = 1;
		break;
	    }	

	    if( flag == 0 ){
	      fprintf( disp, "%s[%d]\n", src.out.rec3.mdl_name, 
		       src.out.rec3.comp_number);
	      row++;
	    }
	  }

	}			/* while */

	if( num == 0 ){
	  fprintf( disp, "  No information\n" );
	  row++;
	}

	return( row );
}



/*
 *  Print of variables of monitor mode 
 * 	FILE *disp  : output file
 */
static int
OutListupMonitorSCF( disp, src )
FILE 	*disp;
OutFile *src;
{
	int	count;
	char    moni_mode;
	int 	row = 1;

	fprintf( disp, "\n MONITER MODE : " );
	switch( src->out.rec1.monitor_mode ){
	  case SCF_OUT_MONITOR_NO_CRT:
	    fprintf( disp, "NO CRT OUTPUT\n" );
	    break;
  	  case SCF_OUT_MONITOR_CRT:
	    fprintf( disp, "VALUES ONLY\n" );
	    break;
	  default:
	    fprintf( disp, "NO SET\n" );
	    break;
	}			/* switch */
	row++;
	moni_mode = src->out.rec1.monitor_mode;
	if( moni_mode == SCF_OUT_MONITOR_CRT ){
	  for( count = 0; count < SCF_OUT_MONITOR_MAX; count++ ){
	    if(src->out.rec1.vmoni_buffnum[count] != 0){
	      fprintf( disp, " VALUE%d : %d(%d)\n", (count + 1), 
		       src->out.rec1.vmoni_buffnum[count],
		       src->out.rec1.vmoni_buffdim[count]);
	    }else{
	      fprintf( disp, " VALUE%d :\n", (count + 1));
	    }
	    row++;
	  }		
	}		
	return( row );
}



/*
 *  Print list of environment variables 
 * 	FILE *disp  : output file
 */
static int
OutListupEnvSCF( disp )
FILE	*disp;
{
	OutFile	src;
	int	row = 0;

	OutTopSCF();

	while( ( OutReadSCF( &src ) != FALSE )
	       && ( src.type >= OUT_MONITOR_TYPE )
	       && ( src.type != SCF_EOF ) ) {

	  switch( src.type ){
	    case SCF_OUT_MONITOR:
	      row += OutListupMonitorSCF( disp, &src );
	      break;
	    case SCF_OUT_FILE:
	      if( src.out.rec2.strf_flag == '1' ){
		fprintf( disp, " DATA FILE : %s\n", src.out.rec2.strf_name);
		row++;
	      }
	      break;
	    case SCF_OUT_BUFF_SIZE:
	      fprintf( disp, "Max of Number of Output = %d\n", 
		       src.out.rec4.max_number);
	      row++;
	      break;
	    default:
	      break;  
	  }	

	}
	fprintf( disp, "\n" );
	row++;
	return( row );
}


/*
 *  Print list of all entry of output information file
 * 	FILE *disp  : output file
 *      Return      : success : number of lines of lists
 *                    0       : failed
 */
int
OutListupSCF( disp )
FILE	*disp;
{
  	int	row = 1;

	fprintf( disp, "\nOUTPUT\n");
  
	if( OutOpenSCF() == FALSE ){
	  return( 0 );
	}
	row += OutListupVarSCF( disp );
	row += OutListupEnvSCF( disp );
	OutCloseSCF();

	return( row );
}
