/*****************************************************************************
  
  	Function Library to Control NCS Simulation Condition File 
  	          for information of parameter variables

    $Id: scfprm.c,v 1.1.1.1 2004/03/26 14:57:15 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <string.h>
#include <fcntl.h>

#include "ncsdef.h"
#include "common.h"
#include "scflib.h"
#include "scfn.h"
#include "scfprm.h"

/* File pointer for simulaiton condition file of output information */
FILE	*prm;

static int   PrmOpenSCF		_ANSI_ARGS_(( void ));
static int   PrmCreateSCF	_ANSI_ARGS_(( void ));
static int   PrmCloseSCF	_ANSI_ARGS_(( void ));
static int   PrmWriteSCF	_ANSI_ARGS_(( PrmFile *src ));
static int   PrmReadSCF		_ANSI_ARGS_(( PrmFile *src ));
static void  PrmTopSCF		_ANSI_ARGS_(( void ));
static void  PrmEndSCF		_ANSI_ARGS_(( void ));
static void  PrmNextSCF		_ANSI_ARGS_(( void ));
static void  PrmNthNextSCF	_ANSI_ARGS_(( int n ));
static void  PrmPreviousSCF	_ANSI_ARGS_(( void ));
static void  PrmNthPreviousSCF	_ANSI_ARGS_(( int n ));
static int  PrmNthGetInfoSCF	_ANSI_ARGS_(( PrmFile *src, int n ));


/**************************************************************************
  	Basic functions to control simulation condition file
 **************************************************************************/


/*
 *  Open parameter information file to read and write
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int 
PrmOpenSCF()
{
	char		*path;
	char 		mname[SCF_SCFN_PATH_SIZE];
	char 		fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_PRM_EXTN );
	} else {
	  return( FALSE );
	}

	if( (prm = OpenSCF( fname )) == NULL ){
	  return( FALSE );
	} else {
	  return( TRUE );
	}
}

/*
 *  Create parameter information file
 * 	char *fname : file name of file
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int
PrmCreateSCF()
{
	char		*path;
	char 		mname[SCF_SCFN_PATH_SIZE];
	char 		fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_PRM_EXTN );
	} else {
	  return( FALSE );
	}

	if( (prm = CreateSCF( fname )) == NULL ){
	  return( FALSE );
	} else {
	  return( TRUE );
	}
}

/*
 *  Close parameter information file
 */
static int
PrmCloseSCF( void )
{
  	return( CloseSCF( prm ) );
}

/*
 *  Write one entry to parameter information file
 * 	PrmFile *src: reading at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE  : success.
 *                    FALSE : failed.
 */
static int
PrmWriteSCF( src )
PrmFile *src;
{
  	if( WriteSCF( prm, (void *)src, sizeof(PrmFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Read one entry from parameter information file
 * 	PrmFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int
PrmReadSCF( src )
PrmFile *src;
{
  	if( ReadSCF( prm, (void *)src, sizeof(PrmFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Set file pointer top of parameter information file
 */
static void
PrmTopSCF( void )
{
  	fseek( prm, 0L, SEEK_SET );
}

/*
 *  Set file pointer end of parameter information file file
 */
static void
PrmEndSCF( void )
{
  	fseek( prm, (long)sizeof(PrmFile)*(-1), SEEK_END );
}

/*
 *  advance file pointer of parameter information file file
 */
static void
PrmNextSCF( void )
{
  	fseek( prm, (long)sizeof(PrmFile), SEEK_CUR );
}

/*
 *  advance nth file pointer of parameter information file file
 */
static void
PrmNthNextSCF( n )
int 	n;
{
  	int	i;

	for( i = 0; i < n; i++ ){
	  PrmNextSCF();
	}
}

/*
 *  retreat file pointer of parameter information file file
 */
static void
PrmPreviousSCF( void )
{
  	fseek( prm, (long)sizeof(PrmFile)*(-1), SEEK_CUR );
}

/*
 *  retreat nth file pointer of parameter information file file
 */
static void
PrmNthPreviousSCF( n )
int	n;
{
  	int	i;

	for( i = 0; i < n; i++ ){
	  PrmPreviousSCF();
	}
}


/**************************************************************************
       Functions to write simulation condition file
 **************************************************************************/

/*
 *  Make parameter information file
 *      ( ->SCF_EOF )
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
PrmMakeSCF()
{
  	PrmFile 	src;

	if( PrmCreateSCF() == FALSE ){
	  return( FALSE );
	}

	src.type = SCF_EOF;
	if( PrmWriteSCF( &src ) == FALSE ){
	  PrmCloseSCF();
	  return( FALSE );
	}
	
	PrmCloseSCF();
	return( TRUE );
}

/*
 *  Add one entry to end of parameter information file
 * 	PrmFile *add: add informaiton
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
PrmAddInfoSCF( add )
PrmFile *add;
{
	PrmFile src;

	if( PrmOpenSCF() == FALSE ){
	  return( FALSE );
	}
  	
	PrmEndSCF();
  	if( PrmWriteSCF( add ) == SCF_ERROR ){
	  PrmCloseSCF();
	  return( FALSE );
	}

	src.type = SCF_EOF;
  	if( PrmWriteSCF( &src ) == SCF_ERROR ){
	  PrmCloseSCF();
	  return( FALSE );
	}

	PrmCloseSCF();
	return( TRUE );
}

/*
 *  Write one entry to n position of parameter information file 
 * 	PrmFile *add: add informaiton
 * 	int n       : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
PrmPutInfoSCF( add, n )
PrmFile *add;
int	n;
{
        PrmFile org;

	if( PrmOpenSCF() == FALSE ){
	  return( FALSE );
	}
  	
	PrmTopSCF();
	PrmNthNextSCF( n );
	if( PrmReadSCF( &org ) == SCF_ERROR ){
	  PrmCloseSCF();
	  return( FALSE );
	} 
	org.prm_val.prm_real = add->prm_val.prm_real;
	PrmPreviousSCF();
	if( PrmWriteSCF( &org ) == SCF_ERROR ){
	  PrmCloseSCF();
	  return( FALSE );
	}

	PrmCloseSCF();
	return( TRUE );
}

/*
 *  Set one entry to parameter information structure.
 * 	PrmFile *src    : storing at the location 
 *	char type       : type of parameter
 *	char *mdl_name  : module name to set information.
 *	char *prm_name  : parameter name with ID to set information.
 *      double data     : value of parameter
 *      Return          : TRUE  : success.
 *                        FALSE : failed.
 */
int
PrmSetInfoSCF( src, type, mdl_name, prm_name, data )
PrmFile *src;
char	*type;
char 	*mdl_name, *prm_name;
double  data;
{
  	src->type = *type;

  	if( mdl_name != NULL ){
	  if( strlen( mdl_name ) >= ( SCF_NAME_LEN-1 ) ){
	    strncpy( src->mdl_name, mdl_name, SCF_NAME_LEN-1 );
	    src->mdl_name[SCF_NAME_LEN-1] = '\0';
	    fprintf( stderr, "Warnning: module name is too long.\n" );
	  }else{
	    strcpy( src->mdl_name, mdl_name );
	  }
	} else {
	  src->mdl_name[0] = '\0';
	}

  	if( prm_name != NULL ){
	  if( strlen( prm_name ) >= ( SCF_NAME_LEN-1 ) ){
	    strncpy( src->prm_name, prm_name, SCF_NAME_LEN-1 );
	    src->prm_name[SCF_NAME_LEN-1] = '\0';
	    fprintf( stderr, "Warnning: module name is too long.\n" );
	  }else{
	    strcpy( src->prm_name, prm_name );
	  }
	} else {
	  src->prm_name[0] = '\0';
	}

	switch( *type ){
	  case SCF_PRM_TYPE_NORMAL:
	    src->prm_val.prm_real = data;
	    break;
	  case SCF_PRM_TYPE_INT:
	    src->prm_val.prm_real = (long)data;
	    break; 
	}

	return( TRUE );
}

/*
 *  Search same entry to parameter information file
 * 	PrmFile *cmp: compare entry
 *      Return      :  n > 0  : potion of the same entry
 *                     n < 0  : no match. n is end of entry.
 *                     0      : ERROR
 */
int
PrmSearchInfoSCF( cmp )
PrmFile *cmp;
{
	PrmFile src;
	int     n = 0;
	

	if( PrmOpenSCF() == FALSE ){
	  return( 0 );
	}
	PrmTopSCF();

	while( PrmReadSCF( &src ) != SCF_ERROR ){
	  if( src.type == SCF_EOF ){
	    PrmCloseSCF();	
	    return( -n );
	    break;
	  }

	  if( !strcmp(src.mdl_name,cmp->mdl_name) 
	     && !strcmp(&src.prm_name[3], cmp->prm_name) ){
	    break;
	  }
	  
	  n++;
	}

	PrmCloseSCF();	
	return( n );
}

/*
 *  Write one entry to parameter information file
 * 	PrmFile *src: storing at the location 
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
PrmWriteInfoSCF( src )
PrmFile *src;
{
	int             pos;

	pos = PrmSearchInfoSCF( src );
	if( pos < 0 ){
	  return( FALSE );
	} else {
	  return( PrmPutInfoSCF( src, pos ) );
	}
}



/**************************************************************************
       Functions to read simulation condition file
 **************************************************************************/

/*
 *  Read one entry from n potition of output information file
 * 	PrmFile *src: storing at the location 
 *      int  n      : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
static int
PrmNthGetInfoSCF( src, n )
PrmFile *src;
int	n;
{
	if( PrmOpenSCF() == FALSE ){
	  return( 0 );
	}
	PrmTopSCF();
	PrmNthNextSCF( n );
  	if( PrmReadSCF( src ) == FALSE ){
	  PrmCloseSCF();
	  return( FALSE );
	}else{
	  PrmCloseSCF();
	  return( TRUE );
	}
}

/*
 *  Read one entry from n potition of output information file
 * 	PrmFile *src: search name & return value
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
PrmGetValueSCF( src )
PrmFile *src;
{
  	int pos;

	if( (pos = PrmSearchInfoSCF( src )) < 0 ){
	  return( FALSE );
	}
	return(PrmNthGetInfoSCF( src, pos ));
}


/**************************************************************************
       Functions to display content of simulation condition file
 **************************************************************************/

/*
 *  Print list of all entry of output information file
 * 	FILE *disp  : output file
 * 	char *fname : file name
 *      Return      : success : number of lines of lists
 *                    0       : failed
 */
int
PrmListupSCF( disp, module )
FILE	*disp;
char	*module;
{
  	int	cmp_mdl, cmp_all;
	int     found = 0;
	char    current_mdl_name[SCF_NAME_LEN];
	PrmFile	sc;
	int	row = 0;

	fprintf( disp, "\nParameter List\n" );
	row++;

	if( PrmOpenSCF() != FALSE ){
	  current_mdl_name[0] = '\0';
	  while( PrmReadSCF(&sc) && sc.type != SCF_EOF ){
	    cmp_mdl = strcmp( module, sc.mdl_name );
	    cmp_all = ( ( strcmp( module, SCF_PRM_ALL_MDL ) == 0 )
	             || ( strcmp( module, SCF_PRM_ALL_MDL_OLD ) == 0 ) );
	    if( ( cmp_mdl == 0 ) || cmp_all ){
	      if( strcmp( current_mdl_name, sc.mdl_name ) ){
		fprintf( disp, "\nModule name : %s\n", sc.mdl_name );
		row++;
	      }
	      if( sc.type != SCF_PRM_TYPE_INT ){
		fprintf( disp, "  %-9s = [%8g]\n", 
			&sc.prm_name[3], sc.prm_val.prm_real);
		row++;
	      } else {
		fprintf( disp, "  %-9s = [%8ld]\n", &sc.prm_name[3], 
			sc.prm_val.prm_intg);
		row++;
	      }
	      found = 1;
	      strcpy( current_mdl_name, sc.mdl_name );
	    }
	  }			/* while */
	  if( found == 0 ){
	    fprintf( disp, "\nThere is no parameter.\n");
	    row+=2;
	  }
	  PrmCloseSCF();
	} else {
	  fprintf( disp, 
		  "Error: Can't open the file of parameter condition!!\n");
	}
	    
	return( row );
}
