/*****************************************************************************

   	Function Library to Control NCS Simulation Condition File 
  	for information of EXTERNAL INPUT FUNCTION parameters(XFP)

    $Id: scfxfp.c,v 1.1.1.1 2004/03/26 14:57:15 orrisroot Exp $

*****************************************************************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif /* HAVE_MALLOC_H */
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif /* HAVE_SYS_TYPES_H */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <string.h>

#include "common.h"
#include "scflib.h"
#include "scfn.h"
#include "scfxin.h"
#include "scfxfp.h"

/* File pointer for simulaiton condition file of XFP */
FILE	*xfp;

static int   XfpOpenSCF		_ANSI_ARGS_(( void ));
static int   XfpCreateSCF	_ANSI_ARGS_(( void ));
static int   XfpCloseSCF	_ANSI_ARGS_(( void ));
static int   XfpWriteSCF	_ANSI_ARGS_(( XfpFile *src ));
static int   XfpReadSCF		_ANSI_ARGS_(( XfpFile *src ));
static void  XfpTopSCF		_ANSI_ARGS_(( void ));
static void  XfpEndSCF		_ANSI_ARGS_(( void ));
static void  XfpNextSCF		_ANSI_ARGS_(( void ));
static void  XfpNthNextSCF	_ANSI_ARGS_(( int n ));
static void  XfpPreviousSCF	_ANSI_ARGS_(( void ));
static void  XfpNthPreviousSCF	_ANSI_ARGS_(( int n ));
static int   XfpGetNpXfdata	_ANSI_ARGS_(( char *type ));
static void  FixSizeStrcpy	_ANSI_ARGS_(( char *to, char *from, 
					      int size ));


/*
 *  Open function of XFP file
 *      Return      : TRUE  : success
 *                    FALSE : failed
 */
static int
XfpOpenSCF()
{
	char		*path;
	char 		mname[SCF_SCFN_PATH_SIZE];
	char 		fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_XFP_EXTN );
	} else {
	  return( FALSE );
	}

	if( (xfp=OpenSCF(fname)) == NULL ){
	  return( FALSE );
	} else {
	  return( TRUE );
	}
}

/*
 *  Create function of XFP file
 *      Return      : TRUE  : sucess
 *                    FALSE : failed
 */
static int
XfpCreateSCF()
{
	char	*path;
	char 	mname[SCF_SCFN_PATH_SIZE];
	char 	fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_XFP_EXTN );
	} else {
	  return( FALSE );
	}

	if( (xfp = CreateSCF( fname )) != NULL ){
	  return( TRUE );
	} else {
	  return( FALSE );
	}
}

/*
 *  Close functions of XFP file
 */
static int
XfpCloseSCF()
{
  	return( CloseSCF( xfp ) );
}

/*
 *  Write one entry to functions of exinput information file
 * 	XfpFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int
XfpWriteSCF( src )
XfpFile *src;
{
  	if( WriteSCF( xfp, (void *)src, sizeof(XfpFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Read one entry from functions of XFP file
 * 	ObsFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int
XfpReadSCF( src )
XfpFile *src;
{
  	if( ReadSCF( xfp, (void *)src, sizeof(XfpFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Set file pointer top of file
 */
static void
XfpTopSCF( void )
{
  	fseek( xfp, 0L, SEEK_SET );
}

/*
 *  Set file pointer end of file
 */
static void
XfpEndSCF()
{
  	XfpFile tmp;

	XfpTopSCF();
  	do{
	  XfpReadSCF( &tmp );
	}while( tmp.type != SCF_EOF );
	XfpPreviousSCF();
}

/*
 *  advance file pointer
 */
static void
XfpNextSCF( void )
{
  	fseek( xfp, (long)sizeof(XfpFile), SEEK_CUR );
}

/*
 *  advance nth file pointer
 */
static void
XfpNthNextSCF( n )
int 	n;
{
  	int	i;

	for( i = 0; i < n; i++ ){
	  XfpNextSCF();
	}
}

/*
 *  retreat file pointer
 */
static void
XfpPreviousSCF( void )
{
  	fseek( xfp, (long)sizeof(XfpFile)*(-1), SEEK_CUR );
}

/*
 *  retreat nth file pointer
 */
static void
XfpNthPreviousSCF( n )
int	n;
{
  	int	i;

	for( i = 0; i < n; i++ ){
	  XfpPreviousSCF();
	}
}


/**************************************************************************
       Functions to write simulation condition file
 **************************************************************************/

/*
 *  Make XFP file
 *      ( ->SCF_EOF )
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
XfpMakeSCF()
{
  	XfpFile 	src;

	if( XfpCreateSCF() == FALSE ){
	  return( FALSE );
	}
	
	src.type = SCF_EOF;
	if( XfpWriteSCF( &src ) == FALSE ){
	  XfpCloseSCF();
	  return( FALSE );
	}
	
	XfpCloseSCF();
	return( TRUE );
}


/*
 *  Add one entry to XFP file
 * 	XfpFile *add: add informaiton
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
XfpAddInfoSCF( add )
XfpFile *add;
{
	XfpFile src;

	if( XfpOpenSCF() == FALSE ){
	  return( FALSE );
	}
  	
	XfpEndSCF();
  	if( XfpWriteSCF( add ) == SCF_ERROR ){
	  XfpCloseSCF();
	  return( FALSE );
	}

	src.type = SCF_EOF;
  	if( XfpWriteSCF( &src ) == SCF_ERROR ){
	  XfpCloseSCF();
	  return( FALSE );
	}

	XfpCloseSCF();
	return( TRUE );
}


/*
 *  Write one entry to n position of XFP file 
 * 	XfpFile *add: add informaiton
 * 	int n       : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
XfpPutInfoSCF( add, n )
XfpFile *add;
int	n;
{
	if( XfpOpenSCF() == FALSE ){
	  return( FALSE );
	}
  	
	XfpTopSCF();
	XfpNthNextSCF( n );
  	if( XfpWriteSCF( add ) == SCF_ERROR ){
	  XfpCloseSCF();
	  return( FALSE );
	}

	XfpCloseSCF();
	return( TRUE );
}

/*************************************************************************
   functions to setting data
 *************************************************************************/
/*
 *  Set one entry to XFP structure.
 * 	XfpFile *src    : storing at the location 
 *	char type       : type of external function
 *	int nval        : number of parameters
 *	double *val     : value of parameters
 *	char *name      : buffer name or file name 
 *	char *time      : buffer name for time information.
 *      Return          : TRUE  : success.
 *                        FALSE : failed.
 */
int
XfpSetInfoSCF( src, type, val, fname, bname, time )
XfpFile *src;
char    *type;
double  val[];
char    *fname;
char    *bname;
char	*time;

{
  	char	xff_num = 0;
	int	i;
	int	nval = 0;

	nval = XfpGetNpXfdata( type );
	switch( *type ){
	  case 'R':		/* Ramp   */
	    xff_num = SCF_XFP_FCN_RAMP;
	    break;
	    
	  case 'P':		/* Pulse  */
	    xff_num = SCF_XFP_FCN_PULSE;
	    break;

	  case 'F':		/* File   */
	    xff_num = SCF_XFP_FCN_FILE;
	    break;

	  case 'B':		/* Buffer */
	    xff_num = SCF_XFP_FCN_BUFFER;
	    break;

	  default:
	    xff_num = atoi(type);
	    break;
	}
	src->func_number = xff_num;

	switch( xff_num ){
	  case 1:
	  case 2:		/* Pulse & Ramp */
	    src->type = SCF_XFP_TYPE_PULSE;
	    for( i = 0; i < nval; i++ ){
	      src->xfp.rec1.func_para[i] = val[i];
	    }
	    break;

	  case 3:		/* File */
	    src->type = SCF_XFP_TYPE_FILE;
	    if( fname != NULL ){
	      if( strlen( fname ) >= ( SCF_NAME_LEN-1 ) ){
		strncpy( src->xfp.rec2.datf_name, fname, SCF_NAME_LEN-1 );
		src->xfp.rec2.datf_name[SCF_NAME_LEN-1] = '\0';
		fprintf( stderr, "Warnning: file name is too long.\n" );
	      }else{
		strcpy( src->xfp.rec2.datf_name, fname );
	      }
	    } else {
	      src->xfp.rec2.datf_name[0] = '\0';
	    }
	    src->xfp.rec2.buffnum = (int)val[0];
	    break;

	  case 4:		/* Buffer */
	    src->type = SCF_XFP_TYPE_BUFFER;
	    strcpy( src->xfp.rec2.datf_name, "BUFFER" );
	    src->xfp.rec2.buffnum = (int)val[0];
	    src->xfp.rec2.buffdim = (int)val[1];
	    if( bname != NULL ){
	      if( strlen( bname ) >= ( SCF_NAME_LEN-1 ) ){
		strncpy( src->xfp.rec2.buffname, bname, SCF_NAME_LEN-1 );
		src->xfp.rec2.buffname[SCF_NAME_LEN-1] = '\0';
		fprintf( stderr, "Warnning: buffer name is too long.\n" );
	      }else{
		strcpy( src->xfp.rec2.buffname, bname );
	      }
	    } else {
	      src->xfp.rec2.buffname[0] = '\0';
	    }
	    src->xfp.rec2.timebuff = (int)val[2];
	    if( time != NULL ){
	      if( strlen( time ) >= ( SCF_NAME_LEN-1 ) ){
		strncpy( src->xfp.rec2.timename, time, SCF_NAME_LEN-1 );
		src->xfp.rec2.timename[SCF_NAME_LEN-1] = '\0';
		fprintf( stderr, 
			"Warnning: buffer name of timer info. is too long.\n" );
	      }else{
		strcpy( src->xfp.rec2.timename, time );
	      }
	    } else {
	      src->xfp.rec2.timename[0] = '\0';
	    }
	    break;

	  default:
	    src->type = SCF_XFP_TYPE_DEFAULT;
	    for( i = 0; i < nval; i++ ){
	      src->xfp.rec1.func_para[i] = val[i];
	    }
	    break;
	}			/* switch */

	return( TRUE );
}

/*
 *  Set message function parameters
 * 	Xfpdata *data   :  structure to store messages
 *      Return          : number of functions - 1
 *                        
 */
int
XfpSetDefaultsXfdata( data )
Xfpdata data[];
{
	int             nxf = 0;

	strcpy(data[nxf].xf_name, "RAMP");
	data[nxf].xf_npara = 3;
	strcpy( data[nxf].xf_para[0], "start_tm" );
	strcpy( data[nxf].xf_para[1], "init_out" );
	strcpy( data[nxf].xf_para[2], "gradient" );
	nxf++;

	strcpy(data[nxf].xf_name, "PULSE");
	data[nxf].xf_npara = 5;
	strcpy( data[nxf].xf_para[0], "start_tm" );
	strcpy( data[nxf].xf_para[1], "init_out" );
	strcpy( data[nxf].xf_para[2], "height" );
	strcpy( data[nxf].xf_para[3], "width" );
	strcpy( data[nxf].xf_para[4], "period" );
	nxf++;

	strcpy(data[nxf].xf_name, "FILE");
	data[nxf].xf_npara = 3;
	strcpy( data[nxf].xf_para[0], "filename" );
	strcpy( data[nxf].xf_para[1], "buff No." );
	strcpy( data[nxf].xf_para[2], "index" );
	nxf++;

	strcpy(data[nxf].xf_name, "BUFFER");
	data[nxf].xf_npara = 3;
	strcpy( data[nxf].xf_para[0], "variable" );
	strcpy( data[nxf].xf_para[1], "buff No." );
	strcpy( data[nxf].xf_para[2], "time" );
	nxf++;

	return( nxf - 1 );
}

/*
 *  Set message function parameters
 * 	Xfpdata *data   : stucture to store messages
 *      Return          : number of functions - 1
 *                        
 */
static int
XfpGetNpXfdata( type )
char *type;
{
  	int num;

	switch( *type ){
	  case 'R':		/* Ramp   */
	    num = 3;
	    break;
	    
	  case 'P':		/* Pulse  */
	    num = 5;
	    break;

	  case 'F':		/* File   */
	    num = 3;
	    break;

	  case 'B':		/* Buffer */
	    num = 3;
	    break;

	  default:
	    num = 5;
	    break;
	}

	return( num );
}

/*************************************************************************
   basic functions 
 *************************************************************************/
/*
 *  Write one entry to XfP file
 * 	XfpFile *src: storing at the location 
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
/*
int
XfpWriteInfoSCF( comp, src )
XinFile *comp;
XfpFile *src;
{
	int             pos;

	pos = XinSearchInfoSCF( comp );
	if( pos < 0 ){
	  return( XfpAddInfoSCF( src ) );
	} else {
	  return( XfpPutInfoSCF( src, pos ) );
	}
}
*/

int XfpSetStimSCF( sc_xin, sc_xfp )
XinFile *sc_xin;
XfpFile *sc_xfp;
{
  	int	pos;
	pos = XinSearchInfoSCF( sc_xin );
	if( pos < 0 ){
	  sc_xin->xin.rec3.xfd_number = -pos;
	  return( XinAddInfoSCF( sc_xin ) && XfpAddInfoSCF( sc_xfp ) );
	} else {
	  sc_xin->xin.rec3.xfd_number = pos+1;
	  return( XinPutInfoSCF(sc_xin,pos) && XfpPutInfoSCF(sc_xfp,pos) );
	}
}

/**************************************************************************
       Functions to read simulation condition file
 **************************************************************************/

/*
 *  Read one entry from n potition from XFP file
 * 	XfpFile *src: storing at the location 
 *      int  n      : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
XfpGetInfoSCF( src, n )
XfpFile *src;
int	n;
{
	if( XfpOpenSCF() == FALSE ){
	  return( 0 );
	}

	XfpTopSCF();
	XfpNthNextSCF( n );
  	if( XfpReadSCF( src ) == FALSE ){
	  XfpCloseSCF();
	  return( FALSE );
	}else{
	  XfpCloseSCF();
	  return( TRUE );
	}
}

/**************************************************************************
       Functions to delete entry of simulation condition file
 **************************************************************************/

/*
 *  Delete one entry from n potition of output information file
 *      int  n      : position   
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
XfpDeleteInfoSCF( n )
int	n;
{
  	XfpFile	tmp;

	if( n < 0 ){
	  return( TRUE );
	}
	if( XfpOpenSCF() == FALSE ){
	  return( FALSE );
	}

	XfpTopSCF();
	XfpNthNextSCF( n );
	do{
	  XfpNextSCF();
	  if( XfpReadSCF( &tmp ) == FALSE ){
	    XfpCloseSCF();
	    return( FALSE );
	  }
	  XfpNthPreviousSCF( 2 );
	  if( XfpWriteSCF( &tmp ) == FALSE ){
	    XfpCloseSCF();
	    return( FALSE );
	  }
	}while( tmp.type != SCF_EOF );
	
	XfpCloseSCF();
	return( TRUE );
}


/*
 *  Delete all entry of output information file
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
XfpDeleteAllInfoSCF()
{
  	XfpFile src;
	int	tmp;

	if( XfpOpenSCF() == FALSE ){
	  return( FALSE );
	}

	XfpTopSCF();	
	src.type = SCF_EOF;
	if( XfpWriteSCF( &src ) == FALSE ){
	  return( FALSE );
	}
	while( (tmp=XfpReadSCF( &src )) != FALSE ){
	  if( src.type == SCF_EOF ){
	    break;
	  }
	  src.type = SCF_ERASED;
	  XfpPreviousSCF();
	  if( XfpWriteSCF( &src ) == FALSE ){
	    XfpCloseSCF();
	    return( FALSE );
	  } 
	}
	XfpCloseSCF();
	return( tmp );
}

int 
XfpRemoveInfoSCF( src, aflag )
XinFile	*src;
int	aflag;
{
	int	dnum=0;

	/* one or more Buffer */
	if( aflag != SCF_ALL_MATCH ){
	  dnum = src->xin.rec3.comp_number-1;
	  return( XinDeleteInfoSCF(dnum) && XfpDeleteInfoSCF(dnum) );
	} else {
	/* all Buffer */
	  return( XinDeleteAllInfoSCF() && XfpDeleteAllInfoSCF() );
	}
}

/**************************************************************************
       Functions to display content of simulation condition file
 **************************************************************************/

/*
 * copy strings. if strings which is copied is smaller than size, ...
 * 	char *to   : copy to
 * 	char *from : copy from
 *	size       : size to copy 
 */
static void
FixSizeStrcpy( to, from, size )
char	to[], from[];
int     size;
{
	int             cnt;

	strcpy(to, from);
	for( cnt = strlen(to); cnt < size; cnt++ ){
	  to[cnt] = SPACE;
	}
	to[cnt] = EOS;
}

/*
 *  Print list of all entry from XFP file
 * 	FILE *disp  : output file
 */
int
XfpListupSCF( disp )
FILE	*disp;
{
	XfpFile src;
	int	row = 1;
	int     xfd_num = 0, tmp_fnum, tmp_pnum, tmp_npara=0;
	Xfpdata xf_data[10];
	char    tmp_xfpara[SCF_NAME_LEN];

	if( XfpSetDefaultsXfdata( xf_data ) < 0 ){
	  fprintf( stderr, "Error: Failed to set the SCF data\n" );
	  return( 0 );
	};

	if( XfpOpenSCF() == FALSE ){
	  fprintf( stderr, "Wanning: Failed to open the SCF\n" );
	  return( 0 );
	}
	
	while( XfpReadSCF( &src ) != SCF_ERROR ){
	  if( src.type == SCF_EOF ){
	    break;
	  }

	  fprintf( disp, "No.%3d ", ++xfd_num);
	  tmp_fnum = src.func_number - 1;
	  fprintf( disp, " Function  ");
	  tmp_npara = xf_data[tmp_fnum].xf_npara;
	  for (tmp_pnum = 0; tmp_pnum < tmp_npara; tmp_pnum++) {
	    if( (src.func_number != SCF_XFP_FCN_BUFFER) || (tmp_pnum != 1) ){
	      FixSizeStrcpy( tmp_xfpara, 
			   xf_data[tmp_fnum].xf_para[tmp_pnum], 10 );
	      fprintf( disp, "%10s ", tmp_xfpara );
	    }
	      
	  }
	  fprintf( disp, "\n       ");
	  row++;
	  fprintf( disp, " %8s ", xf_data[tmp_fnum].xf_name);
	  switch( src.type ){
	    case SCF_XFP_TYPE_PULSE:	/* Pulse & Ramp */
	      for( tmp_pnum = 0; tmp_pnum < tmp_npara; tmp_pnum++ ){
		fprintf( disp, "  %-8g ", src.xfp.rec1.func_para[tmp_pnum]);
	      }
	      fprintf( disp, "\n");
	      row++;
	      break;
	    
  	    case SCF_XFP_TYPE_BUFFER:	/* Buffer & File */
	      if( src.func_number == 3 ){
		fprintf( disp, "   %8s ", src.xfp.rec2.datf_name );
		fprintf( disp, "   %8d ", src.xfp.rec2.buffnum );
		fprintf( disp, "   %8d \n", src.xfp.rec2.buffdim );
	      }else{
		fprintf( disp, " %7s[%2d]", src.xfp.rec2.buffname, 
			src.xfp.rec2.buffdim );
		if( strlen(src.xfp.rec2.timename) == 0 ){
		  fprintf( disp, "(%8s)\n", "none" );
		}else{
		  fprintf( disp, " %8s \n", src.xfp.rec2.timename );
		}
	      }
	      row++;
	      break;

	    case SCF_ERASED:	/* Erased data */
	      fprintf( disp, "   ERASED DATA\n" );
	      row++;
	      break;
	      
	    case '3':
	      break;

  	    default:
	      break;
	  }		/* switch */
	}			/* while */
	fprintf( disp, "\n" );
	row++;
	return( row );
}


/*
 *  Print list of all entry to exinput information file
 * 	FILE *disp  : output file
 */
int
XfpListupAllSCF( disp )
FILE	*disp;
{
	int row = 2;

	fprintf( disp, "\nEXTERNAL INPUTS\n");

	row += XinListupSCF( disp );
	row += XfpListupSCF( disp );

	return( row );
}

