/*****************************************************************************

   	Function Library to Control NCS Simulation Condition File 
   		for information of EXINPUT variables(XIN)

    $Id: scfxin.c,v 1.1.1.1 2004/03/26 14:57:15 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif /* HAVE_MALLOC_H */
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif /* HAVE_SYS_TYPES_H */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <string.h>

#include "common.h"
#include "scflib.h"
#include "scfn.h"
#include "scfxin.h"

/* File pointer for simulaiton condition file of exinput information */
FILE	*xin;

static int   XinOpenSCF		_ANSI_ARGS_(( void ));
static int   XinCreateSCF	_ANSI_ARGS_(( void ));
static int   XinCloseSCF	_ANSI_ARGS_(( void ));
static int   XinWriteSCF	_ANSI_ARGS_(( XinFile *src ));
static int   XinReadSCF		_ANSI_ARGS_(( XinFile *src ));
static void  XinTopSCF		_ANSI_ARGS_(( void ));
static void  XinEndSCF		_ANSI_ARGS_(( void ));
static void  XinNextSCF		_ANSI_ARGS_(( void ));
static void  XinNthNextSCF	_ANSI_ARGS_(( int n ));
static void  XinPreviousSCF	_ANSI_ARGS_(( void ));
static void  XinNthPreviousSCF	_ANSI_ARGS_(( int n ));
static void  XinOutputEntrySCF	_ANSI_ARGS_(( XinFile *src, FILE *disp, 
					      int n ));


/*
 *  Open exinput information file
 *      Return      : TRUE  : success
 *                    FALSE : failed
 */
static int
XinOpenSCF()
{
	char		*path;
	char 		mname[SCF_SCFN_PATH_SIZE];
	char 		fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_XIN_EXTN );
	} else {
	  return( FALSE );
	}

	if( (xin=OpenSCF(fname)) == NULL ){
	  return( FALSE );
	} else {
	  return( TRUE );
	}
}

/*
 *  Create exinput information file
 *      Return      : TRUE  : sucess
 *                    FALSE : failed
 */
static int
XinCreateSCF()
{
	char	*path;
	char 	mname[SCF_SCFN_PATH_SIZE];
	char 	fname[SCF_SCFN_PATH_SIZE];
	
	if( (path=GetWorkDirSCF()) != NULL ){
	  sprintf( fname, "%s%s.%s", 
		   path, ModelNameSCFN( mname ), SCF_XIN_EXTN );
	} else {
	  return( FALSE );
	}

	if( (xin = CreateSCF( fname )) != NULL ){
	  return( TRUE );
	} else {
	  return( FALSE );
	}
}

/*
 *  Close exinput information file
 */
static int
XinCloseSCF()
{
  	return( CloseSCF( xin ) );
}

/*
 *  Write one entry to exinput information file
 * 	XinFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int
XinWriteSCF( src )
XinFile *src;
{
  	if( WriteSCF( xin, (void *)src, sizeof(XinFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Read one entry from exinput information file
 * 	ObsFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int
XinReadSCF( src )
XinFile *src;
{
  	if( ReadSCF( xin, (void *)src, sizeof(XinFile) ) == SCF_ERROR ){
	  return( FALSE );
	}else{
	  return( TRUE );
	}
}

/*
 *  Set file pointer top of parameter information file
 */
static void
XinTopSCF( void )
{
  	fseek( xin, 0L, SEEK_SET );
}

/*
 *  Set file pointer end of file
 */
static void
XinEndSCF()
{
  	XinFile tmp;

	XinTopSCF();
  	do{
	  XinReadSCF( &tmp );
	}while( tmp.type != SCF_EOF );
	XinPreviousSCF();
	/*  	fseek( xin, (long)sizeof(XinFile)*(-1), SEEK_END );*/
}

/*
 *  advance file pointer
 */
static void
XinNextSCF( void )
{
  	fseek( xin, (long)sizeof(XinFile), SEEK_CUR );
}

/*
 *  advance nth file pointer
 */
static void
XinNthNextSCF( n )
int 	n;
{
  	int	i;

	for( i = 0; i < n; i++ ){
	  XinNextSCF();
	}
}

/*
 *  retreat file pointer
 */
static void
XinPreviousSCF( void )
{
  	fseek( xin, (long)sizeof(XinFile)*(-1), SEEK_CUR );
}

/*
 *  retreat nth file pointer
 */
static void
XinNthPreviousSCF( n )
int	n;
{
  	int	i;

	for( i = 0; i < n; i++ ){
	  XinPreviousSCF();
	}
}


/**************************************************************************
       Functions to write simulation condition file
 **************************************************************************/

/*
 *  Make exinput information file
 *      ( ->SCF_EOF )
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
XinMakeSCF()
{
  	XinFile 	src;

	if( XinCreateSCF() == FALSE ){
	  return( FALSE );
	}
	
	src.type = SCF_EOF;
	if( XinWriteSCF( &src ) == FALSE ){
	  XinCloseSCF();
	  return( FALSE );
	}
	
	XinCloseSCF();
	return( TRUE );
}


/*
 *  Add one entry to exinput information file
 * 	XinFile *add: add informaiton
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
XinAddInfoSCF( add )
XinFile *add;
{
	XinFile src;

	if( XinOpenSCF() == FALSE ){
	  return( FALSE );
	}
  	
	XinEndSCF();
  	if( XinWriteSCF( add ) == SCF_ERROR ){
	  XinCloseSCF();
	  return( FALSE );
	}

	src.type = SCF_EOF;
  	if( XinWriteSCF( &src ) == SCF_ERROR ){
	  XinCloseSCF();
	  return( FALSE );
	}

	XinCloseSCF();
	return( TRUE );
}

/*
 *  Write one entry to n position of exinput information file 
 * 	XinFile *add: add informaiton
 * 	int n       : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
XinPutInfoSCF( add, n )
XinFile *add;
int	n;
{
	if( XinOpenSCF() == FALSE ){
	  return( FALSE );
	}
  	
	XinTopSCF();
	XinNthNextSCF( n );
  	if( XinWriteSCF( add ) == SCF_ERROR ){
	  XinCloseSCF();
	  return( FALSE );
	}

	XinCloseSCF();
	return( TRUE );
}

/*
 *  Set one entry to exinput information structure.
 * 	XinFile *src    : storing at the location 
 *	char *mdl_name  : module name 
 *	int  comp       : component number
 *	int  xfp_num    : index of external input function
 *      Return          : TRUE  : success.
 *                        FALSE : failed.
 */
int
XinSetInfoSCF( src, mdl_name, comp )
XinFile *src;
char    *mdl_name;
int	comp;
{
  	int	xfp_num;

  	src->type = SCF_XIN_TYPE_NORMAL;

  	if( mdl_name != NULL ){
	  if( strlen( mdl_name ) >= ( SCF_NAME_LEN-1 ) ){
	    strncpy( src->xin.rec3.mdl_name, mdl_name, SCF_NAME_LEN-1 );
	    src->xin.rec3.mdl_name[SCF_NAME_LEN-1] = '\0';
	    fprintf( stderr, "Warnning: module name is too long.\n" );
	  }else{
	    strcpy( src->xin.rec3.mdl_name, mdl_name );
	  }
	} else {
	  src->xin.rec3.mdl_name[0] = '\0';
	}

	if( comp >= 0 ){
	  src->xin.rec3.comp_number = comp;
	} else {
	  src->xin.rec3.comp_number = 0;
	}

	xfp_num = XinSearchInfoSCF( src );

	if( xfp_num >= 0 ){
	  src->xin.rec3.xfd_number = xfp_num;
	} else {
	  src->xin.rec3.xfd_number = 0;
	}

	return( TRUE );
}

/*
 *  Search same entry to exinput information file
 * 	XinFile *cmp: compare entry
 *      Return      : n     : potion of the same entry
 *                    n < 0 : no match
 */
int
XinSearchInfoSCF( cmp )
XinFile *cmp;
{
	XinFile src;
	int     n = 0;

	if( XinOpenSCF() == FALSE ){
	  return( FALSE );
	}
	
	while( XinReadSCF( &src ) != SCF_ERROR ){
	  if( src.type == SCF_EOF ){
	    XinCloseSCF();
	    return( -(n+1) );
	    break;
	  }
	  
	  if( (strcmp( cmp->xin.rec3.mdl_name, src.xin.rec3.mdl_name ) == 0) 
	      && (cmp->xin.rec3.comp_number == src.xin.rec3.comp_number) ){
	    break;
	  }
	  
	  n++;
	}

	XinCloseSCF();
	return( n );
}


/*
 *  Write one entry to exinput information file
 * 	XinFile *src: storing at the location 
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
/*
int
XinWriteInfoSCF( src )
XinFile *src;
{
	int             pos;

	pos = XinSearchInfoSCF( src );
	if( pos < 0 ){
	  return( XinAddInfoSCF( src ) );
	} else {
	  return( XinPutInfoSCF( src, pos ) );
	}
}
*/

/**************************************************************************
       Functions to read simulation condition file
 **************************************************************************/

/*
 *  Read one entry from n potition of exinput information file
 * 	XinFile *src: storing at the location 
 *      int  n      : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
XinGetInfoSCF( src, n )
XinFile *src;
int	n;
{
	if( XinOpenSCF() == FALSE ){
	  return( 0 );
	}

	XinTopSCF();
	XinNthNextSCF( n );
  	if( XinReadSCF( src ) == FALSE ){
	  XinCloseSCF();
	  return( FALSE );
	}else{
	  XinCloseSCF();
	  return( TRUE );
	}
}

/**************************************************************************
       Functions to delete entry of simulation condition file
 **************************************************************************/

/*
 *  Delete one entry from n potition of output information file
 *      int  n      : position   
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
XinDeleteInfoSCF( n )
int	n;
{
  	XinFile	tmp;

	if( n < 0 ){
	  return( TRUE );
	}
	if( XinOpenSCF() == FALSE ){
	  return( FALSE );
	}

	XinTopSCF();
	XinNthNextSCF( n );
	do{
	  XinNextSCF();
	  if( XinReadSCF( &tmp ) == FALSE ){
	    XinCloseSCF();
	    return( FALSE );
	  }
	  XinNthPreviousSCF( 2 );
	  tmp.xin.rec3.xfd_number--;
	  if( XinWriteSCF( &tmp ) == FALSE ){
	    XinCloseSCF();
	    return( FALSE );
	  }
	}while( tmp.type != SCF_EOF );
	
	XinCloseSCF();
	return( TRUE );
}


/*
 *  Delete all entry of output information file
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int
XinDeleteAllInfoSCF()
{
  	XinFile src;
	int 	tmp;

	if( XinOpenSCF() == FALSE ){
	  return( FALSE );
	}

	XinTopSCF();
	src.type = SCF_EOF;
	if( XinWriteSCF( &src ) == FALSE ){
	  return( FALSE );
	}
	while( (tmp=XinReadSCF( &src )) != FALSE ){
	  if( src.type == SCF_EOF ){
	    break;
	  }
	  src.type = SCF_ERASED;
	  XinPreviousSCF();
	  if( (tmp=XinWriteSCF( &src )) == FALSE ){
	    XinCloseSCF();
	    return( FALSE );
	  }
	}
	XinCloseSCF();
	return( tmp );
}

/**************************************************************************
       Functions to display content of simulation condition file
 **************************************************************************/

/*
 *  Print list of all entry to exinput information file
 * 	FILE *disp  : output file
 */
int
XinListupSCF( disp )
FILE	*disp;
{
	XinFile src;
	int	n = 0, row = 1;


	if( XinOpenSCF() == FALSE ){
	  fprintf( stderr, "Wanning: Failed to open the SCF\n" );
	  return( 0 );
	}

	while( XinReadSCF( &src ) != SCF_ERROR ){
	  if( src.type == SCF_EOF ){
	    if( n == 0 ){
	      fprintf( disp, "No data.\n" );
	      row++;
	    } else {
	      fprintf( disp, "\n" );
	      row++;
	    }
	    break;
	  }

	  if( src.type != SCF_ERASED ){
	    if( n == 0 ){
	      fprintf( disp, "  No.      Module     Func.No.\n");
	    }
	    XinOutputEntrySCF( &src, disp, n  );
	  }else{
	    fprintf( disp, " ERASED DATA\n" );
	  }
	  row++;
	  n++;
	}			/* while */

	XinCloseSCF();
	return( row );
}


/*
 *  Display one entry to exinput information file
 * 	FILE *disp  : output file
 */
static void
XinOutputEntrySCF( src, disp, n  )
XinFile *src;
FILE	*disp;
int	n;
{
  	fprintf( disp, " %3d %9s(%3d)     %2d\n", n+1,
		 src->xin.rec3.mdl_name, src->xin.rec3.comp_number,
		 src->xin.rec3.xfd_number );
}
