/*****************************************************************************

   		Functions to input/output to SATELLITE buffers

    $Id: ncssbuff.c,v 1.1.1.1 2004/03/26 14:57:15 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#ifdef HAVE_MALLOC_H
#ifndef _STDLIB_H_
#include <malloc.h>
#endif /* _STDLIB_H_ */
#endif /* HAVE_MALLOC_H */
#include <string.h>
#include <math.h>

#include "SL_macro.h"
#include "SL_cmd.h"

#include "ncsdef.h"
#include "common.h"
#include "comlib.h"
#include "ncssbuff.h"

int interp_flag;

void 	debug			_ANSI_ARGS_(( int j, int ncount, 
					      double *xdata, double *ydata ));
void 	decidep			_ANSI_ARGS_(( int iypt, float *ybuf, 
					      double *xdata, double *ydata ));
int 	rebuff			_ANSI_ARGS_(( Exbuff *exbuff_info, 
					      int buff_index ));
int 	read_time_buff		_ANSI_ARGS_(( Exbuff *exbuff_info, 
					       int buff_index ));
double 	get_buffer_data_at_time	_ANSI_ARGS_(( double t, Exbuff *exbuff_info ));
Exbuff *get_exbuf_info_st	_ANSI_ARGS_(( int buffnum, char *datafile, 
					      int timeid, char *timefile ));
void 	remove_exbuf_info_st	_ANSI_ARGS_(( Exbuff *buf ));



/*********************************************************************
*       Debugging routine                                            *
*                                                                    *
*                              Coded by Sigeru Hitomi                *
*                                                       09/21/1990   *
*********************************************************************/

void 
debug(j, ncount, xdata, ydata)
int             j, ncount;
double          xdata[4], ydata[4];
{

	printf("%10s %10s %10s %10s %10s\n",
	       "J", "XDATA[0]", "XDATA[1]", "XDATA[2]", "XDATA[3]");
	printf("%10d %10.3g %10.3g %10.3g %10.3g\n",
	       j, xdata[0], xdata[1], xdata[2], xdata[3]);

	printf("%10s %10s %10s %10s %10s\n",
	       "NCOUNT", "YDATA[0]", "YDATA[1]", "YDATA[2]", "YDATA[3]");
	printf("%10d %10.3g %10.3g %10.3g %10.3g\n",
	       ncount, ydata[0], ydata[1], ydata[2], ydata[3]);

}				/* End of debug	 */

/*********************************************************************
*                                                                    *
*         4th Order Lagrange Interpolation                           *
*                                                                    *
*         FUNCTION :  NCSL_INTERP                                    *
*                                                                    *
*  on Unix                Coded by Shigeru Hitomi     09/23/1990     *
*--------------------------------------------------------------------*
*                                                                    *
*    double                                                          *
*       ncsl_interp( dummy0, dummy1, dummy2, dummy3, dummy4 )        *
*                                                                    *
*       double  dummy1 - 4    :  dummy parameter                     *
*       double                                                       *
*        (Interpolated value) :  return value                        *
*                                                                    *
*    GLOVAL VALIABLES                                                * 
*       double  NCS_TIME         :  Current Time                     *
*       double  ncsg_endtime     :  Calculation End Time             *
*       double  ncsg_calstep     :  Calculation Step                 *
*       int     ncsg_cdpoint     :  Current Data point ( dpoint < t )*
*       int     ncsg_inp_buffnum :  buffer or block number           *
*       char   *ncsg_datfile     :  Data filename                    *
*                                                                    *
*********************************************************************/

double 
LagInterp( t0, xdata, ydata )
double 	t0, xdata[], ydata[];
{
	register int    j, k;
	double          sumx, sx;

	sumx = 0.0;
	for( j = 0; j < 4; j++ ){
	  sx = 1.0;
	  for( k = 0; k < 4; k++ ){
	    if (j != k){
	      sx *= (t0 - xdata[k]) / (xdata[j] - xdata[k]);
	    }
	  }
	  sumx += sx * ydata[j];
	}

	/*
	 * printf( "t0 = %f, sumx = %f\n", t0, sumx ); 
	 * debug( j, ncount, xdata, ydata );
	 */
	return( sumx );
}				/* End of LagInterp */


/*********************************************************************
*                                                                    *
*         Decide position for Interpolation                          *
*                                                                    *
*                              Coded by Sigeru Hitomi                *
*                                                       09/21/1990   *
*                                                                    *
**********************************************************************/

void 
decidep( iypt, ybuf, xdata, ydata )
int	iypt;
float   ybuf[];
double  xdata[], ydata[];
{
	register int    i, j;
	int             ncount;

	if( ncsg_cdpoint < 1 ){
	  ncount = 1;
	}else{
	  if ( ncsg_cdpoint >= (iypt - 1)){
	    ncount = iypt - 2;
	  } else {
	    ncount = ncsg_cdpoint;
	  }
	}

	j = ncount - 1;
	for( i = 0; i < 4; i++ ){
	  xdata[i] = (double) j * ncsg_calstep;
	  ydata[i] = (double) ybuf[j];
	  j++;
	}
	/*
	 * debug( j, ncount, xdata, ydata ); 
	 * puts( "In decidep .. [Press CR]" ); getchar();
	 */
}				/* End of decidep */


/*********************************************************************
*                                                                    *
*         Read from Data Buffer  /  File                             *
*                              Coded by Sigeru Hitomi  [09/21/1990]  *
*                              Modified                [28,Aug,'94]  *
*                                                                    *
**********************************************************************/

int 
rebuff( exbuff_info, buff_index )
Exbuff *exbuff_info;
int buff_index;
{
	int             iypt, nt, i;
	double          *buf;
	int             dim;
	int             index[MAX_INDEX];
	Header 		head;
	int 		buffnum;
	char            *datfile;
	int		sub_index[MAX_INDEX];

	buffnum = exbuff_info->ncsg_inp_buffnum;
	datfile = exbuff_info->ncsg_datfile;
	
	if( strlen(datfile) != 0 ){
	  if( ( buf = (double *)ReadFile( datfile, &head ) ) == NULL ){
	    exit(226);
	  }else{
	    if( ( dim = head.dim ) > 2 ){
	      exit( 227 );
	    }
	    for( i = 0; i < dim; i++ ){
	      index[i] = head.index[i];
	    }
	  }
	}else{
	  if( ( buf = ReadBuffer( buffnum, &dim, index ) ) == NULL ){
	    exit(211);
	  }
	}
	iypt = index[0];

	if( ( exbuff_info->ybuf = (float *)malloc( sizeof(float)*iypt ) )
	   == NULL ){
	  exit(213);
	}

	{
	    int i;
	    sub_index[1] = buff_index;
	    for(i=0; i<iypt; i++){
	      sub_index[0] = i;
	      exbuff_info->ybuf[i] = (float)buf[Index(sub_index, dim, index )];
	    }
	}

	if( iypt <= 0 ) {
	  if( strlen(datfile) == 0 ){
	    printf(" >> Data Buffer[%3d] not ready <<%d\n", buffnum, iypt);
	    exit(214);
	  }else{
	    printf(" >> Data File[%s] not ready <<%d\n", datfile, iypt);
	    exit(214);
	  }
	}
	nt = (int) (ncsg_endtime / ncsg_calstep + 1.5);
	if( iypt < nt ){
	  printf("\n >> Data Buffer less than Calc. points <<\n");
	  printf(" >>Buffer dpoint = %d\n", iypt );
	  printf(" >>Calc.   point = %d\n", nt );
	  exit(215);
	}

	free( buf );

	return (iypt);

}				/* End of rebuff	 */


/*********************************************************************
*                                                                    *
*         Read from Time Infomation Buffer  ( /  File )              *
*                                 Coded                [17,May,'96]  *
*                                                                    *
**********************************************************************/

int 
read_time_buff( exbuff_info, buff_index )
Exbuff *exbuff_info;
int buff_index;
{
  	int 	buffid;
	char 	*filename;
	int     dim;
	int     index[MAX_INDEX], sub_index[MAX_INDEX];
	Header 	head;
	Buffer  *buf = NULL;
	int 	i;

	buffid = exbuff_info->timeid;
	filename = exbuff_info->timefile;
	
	if( strlen(filename) != 0 ){/* time information buffer set or not? */
	  if( buffid == 0 ){     /* filename = file name */
	    if( ( buf = (Buffer *)ReadFile( filename, &head ) ) == NULL ){
	      exit(228);
	    }else{
	      if( ( dim = head.dim ) > 2 ){
		exit( 229 );
	      }
	      for( i = 0; i < dim; i++ ){
		index[i] = head.index[i];
	      }
	    }
	  }else{		/* finename = buffer name (ignore) */
	    if( ( buf = ReadBuffer( buffid, &dim, index ) ) == NULL ){
	      exit(211);
	    }
	    if (dim <= 0) {
	      exit(214);
	    }
	  }
	  
	  exbuff_info->itpt = index[0];
	  if( (exbuff_info->tbuf=(float *)malloc( sizeof(float)*index[0] ) )
	      == NULL ){
	    exit(213);
	  }
	  sub_index[1] = buff_index;
	  for( i = 0; i < index[0]; i++ ){
	    sub_index[0] = i;
	    exbuff_info->tbuf[i] = (float)buf[Index(sub_index, dim, index )];
	  }
	}else{ /* time information buffer isn't used */
	  if( (exbuff_info->tbuf
	       =(float *)malloc( sizeof(float)*exbuff_info->iypt )) == NULL ){
	    exit(213);
	  }
	  exbuff_info->itpt = exbuff_info->iypt;
	  for( i = 0; i < exbuff_info->iypt; i++ ){
	    exbuff_info->tbuf[i] = (float)(ncsg_endtime/(double)exbuff_info->iypt * (double)i);
	  }
	}

	if( buf != NULL ){
	  free( buf );
	}

	return (exbuff_info->itpt);
}				/* End of rebuff	 */


/*********************************************************************
*                                                                    *
*         Interporation Main Routine                                 *
*                              Coded by Sigeru Hitomi  [09/21/1990]  *
*                              Last Modified           [17,May,'96]  *
*                                                                    *
**********************************************************************/

double 
ncsl_interp(pos, buf_dim, dummy0, dummy1, dummy2)
double 	pos;
double  buf_dim;
double  dummy0, dummy1, dummy2;
{

	double          xdata[4], ydata[4], y0;
	Exbuff         *exbuff_info;
	double          buff_timestep;

	/* read buffer information */
	ncsg_buff_info_p = plist_go_nth( ncsg_buff_info, (int)pos );
	exbuff_info = (Exbuff *)plist_read_data( ncsg_buff_info, ncsg_buff_info_p );

	if (exbuff_info->ic == 0) {	/* read buffer & file */
	  exbuff_info->ic = 1;

	  if (exbuff_info->ncsg_inp_buffnum <= 0) {
	    exit(216);
	  }
	  exbuff_info->iypt = rebuff( exbuff_info, (int)buf_dim );

	  if( exbuff_info->timeid != 0 ){ /* read time infomation */
	    if( exbuff_info->iypt != read_time_buff( exbuff_info, 0 ) ){
	      fprintf( stderr, 
		       "Wanning: Mismatch Number of Data Buffer and Time Buffer\n" );
	    }
	  }
	  /* Now every data buffer have a time information buffer 
	     respectively.					   */

	  /* use or not Lagulange Interpolation? */
	  if( exbuff_info->timeid != 0 ){
	    buff_timestep = ncsg_endtime/(double)exbuff_info->iypt;
	    interp_flag = FALSE;
	    switch( ncsg_intg_method ){
	     case 'F':
	       interp_flag = TRUE;
	       break;
	     case 'E':
	       if( buff_timestep > ncsg_calstep ){
		 interp_flag = TRUE;
	       }
	       break;
	     case 'R':
	      if( buff_timestep > ncsg_calstep/2.0 ){
		interp_flag = TRUE;
	      }
	      break;
	     default:
	      interp_flag = TRUE;
	      break;
	    }
	  }else{
	    interp_flag = TRUE;
	  }

	  if( interp_flag != TRUE ){
	    printf( "\n*** Infomation : Stim %d don't Use Interpolation ***\n", (int)pos );
	  }
	}

	if( interp_flag == TRUE ){
	  decidep(exbuff_info->iypt, exbuff_info->ybuf, xdata, ydata);
	  y0 = LagInterp(NCS_TIME, xdata, ydata);
	}else{
	  y0 = get_buffer_data_at_time( NCS_TIME, exbuff_info );
	}

	return (y0);
}				/* End of ncsl_interp	 */

/*********************************************************************
         Get buffer data at time			             
                                 	               [17,May,'96]  
**********************************************************************/
double
get_buffer_data_at_time( t, exbuff_info )
double t;
Exbuff *exbuff_info;
{
	int 	cnt;
	int 	flag;
	double 	diff, y0=0.0;
	double 	thresh;

	flag = FALSE;
	if( t != 0.0 ){
	  thresh = floor(log10(t)) - 6.0; 
	}else{
	  thresh = -6.0;
	}
	for( cnt = 0; cnt < exbuff_info->itpt; cnt++ ){
	  diff = fabs((double)(*(exbuff_info->tbuf+cnt))-t);
	  if( diff != 0.0 ){
	    diff = log10(diff);
	    if( (diff != 0.0) && (diff < thresh) ){
	      diff = 0.0;
	    }else{
	      diff = 1.0;
	    }
	  }
	  if( diff == 0.0 ){
	    y0 = (double)*(exbuff_info->ybuf+cnt);
	    flag = TRUE;
	    break;
	  }
	}

	if( flag != TRUE ){
	  exit(230);
	}else{
	  return(y0);
	}
}


/*************************************************************************
         Functions To Control Exinput Information Structure
	 				         Coded       [ 9,Aug,'94]
						 Modified    [17,May,'96]
**************************************************************************/

/*
 *  Function to Get Exinput Buffer Infomation Structure(  )
 *	input: int buffnum --- a number of exinput buffer
 *             char *datafile --- name of exinput buffer or exinput file
 *             timeid --- buffer id number of time information buffer
 *             timefile --- name of time infomation buffer or file
 *     output: address of getting buffer infomation structure
 */
Exbuff *
get_exbuf_info_st( buffnum, datafile, timeid, timefile )
int 	buffnum;
char 	*datafile;
int     timeid; 
char    *timefile;
{
  	Exbuff *buf;
	char s[10];

	if( ( buf = (Exbuff *)malloc( sizeof( Exbuff ) ) ) == NULL ){
	  exit( 213 );
	}

	buf->ncsg_inp_buffnum = buffnum;
	if( datafile != NULL ){
	  if( strcmp(datafile,"BUFFER") != 0 ){
	    buf->ncsg_datfile = strdup( datafile );
	  }else{
	    s[0] = EOS;
	    buf->ncsg_datfile = strdup( s );
	  }
	}else{
	  s[0] = EOS;
	  buf->ncsg_datfile = strdup( s );
	}
	buf->timeid = timeid;
	buf->timefile = strdup( timefile );
	buf->ic = 0;
	
	return( buf );
}

/*
 *  Function to Add Exinput Buffer Infomation to Structure
 *	input: PList buffinfo --- exinput buffer infomation(EIBI) list
 * 	       PLPointer buffinfo_p --- pointer of EIBI list 
 *             buffnum --- a number of exinput buffer
 *             datafile --- name of exinput buffer or exinput file
 *             timeid --- buffer id number of time information buffer
 *             timefile --- name of time infomation buffer or file
 *     output: pointer of getting buffer infomation structure
 */
PLPointer 
add_exbuf_info( buffinfo, buffinfo_p, buffnum, datafile,
			  timeid, timefile )
PList 		buffinfo;
PLPointer 	buffinfo_p;
int 		buffnum;
char 		*datafile;
int 		timeid;
char 		*timefile;
{
  	Exbuff *buf;

	buf = get_exbuf_info_st( buffnum, datafile, timeid, timefile );

	buffinfo_p = plist_insert( buffinfo, buffinfo_p, (PLMember)buf );
	buffinfo_p = plist_advance( buffinfo, buffinfo_p );
	return( buffinfo_p );
}  

/*
 *  Function to Remove Exinput Buffer Infomation Structure(  )
 *	input: PLMember buf --- exinput buffer information structure
 *     output: none
 */
void 
remove_exbuf_info_st( buf )
Exbuff 	*buf;
{
  	free( buf->ncsg_datfile );
	free( buf->timefile );
	free( buf->ybuf );

	free( buf );
}

/*
 *  Function to Remove Exinput Buffer Infomation to Structure
 *	input: PList buffinfo --- exinput buffer infomation(EIBI) list
 * 	       PLPointer point --- pointer of EIBI list 
 *     output: pointer of getting buffer infomation structure
 */
PLPointer 
remove_exbuf_info( buffinfo, point )
PList 		buffinfo;
PLPointer 	point;
{
  	Exbuff *buf;

	buf = plist_read_data( buffinfo, point );
	remove_exbuf_info_st( buf );
	point = plist_delete( buffinfo, point );
	return( point );
}  

/*
 *  Function to Remove all Exinput Buffer Infomation to Structure
 *	input: PList buffinfo --- exinput buffer infomation(EIBI) list
 * 	       PLPointer point --- pointer of EIBI list 
 *     output: void
 */
void 
remove_all_exbuf_info( buffinfo )
PList 	buffinfo;
{
  	Exbuff *buf;
  	PLPointer point1;

  	point1 = plist_go_end( buffinfo );
	while( point1 != buffinfo ){
	  buf = plist_read_data( buffinfo, point1 );
	  remove_exbuf_info_st( buf );
	  point1 = plist_delete( buffinfo, point1 );
	}
}  

