/*******************************
**  NPE main control routine  **
*******************************/

#include <stdio.h>
#include <string.h>

#include "SL_macro.h"
#include "SL_cmd.h"

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#if !defined(OSF1)
#include <signal.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#include "NPEcmd.h"
#include "global_vars.h"


#if defined(ANSI)
extern void  renpe    (void);
extern void  wrnpe    (void);
extern void  initHead (void);
extern void  renpe_   (char *filename);
extern void  wrnpe_   (char *filename, int storeflag);
#else
extern void  renpe();
extern void  wrnpe();
extern void  initHead();
extern void  renpe_();
extern void  wrnpe_();
#endif

#define DEFALT_PARALLEL_SIZE (3)
#define MAX_PARALLEL_SIZE    (5)

#if !defined(NPECOMMONAREA)
#define NPECOMMONAREA     "npecommon.area"
#endif

/* global variables in this file */
int   elementCheckTable[ELEMENTNUMBER+1];
int   NumProcess, NumParallel;
char  fname[MAXFNAME];
int   ResultBuffID, HistoryBuffID;
int   Verbose_mode = FALSE;


#define  WINDOW    (0)
#define  ESTIMATOR  (1)

typedef struct id {
  int    pid;  /** Process ID           **/
  int    type; /** Estimation or Window **/
  int    num;  /** Sequencial Number    **/
  int    dummy;
  struct id  *next;
} ID;
ID  *PIDHEAD;


/* function prototypes in this file */
void  registerPID(int num, int pid, int type );
void  waitUntilChildrenEnd(void);
int   waitUntilChildEnd(void);
void  printStatus(int pid, int status );
ID   *removePID(int pid );
void  copyElements(Set *setp, FILE *fp );
void  writeMethod(Method  *work, FILE *fp );
void  writeLsearch(Lsearch *work, FILE *fp );
void  writeModel(Model *work, FILE *fp );
void  writePenalty(Penalty *work, FILE *fp );
void  writeInit(Init *work, FILE *fp );
void  writeScale(Scale *work, FILE *fp );
void  writeTerm(Term *work, FILE *fp );
void  writeNumber(Number *work, FILE *fp );
void  writePoint(Point *work, FILE *fp );
void  writeData(Experiment *work, FILE *fp );
void  writeWeight(Weightt *work, FILE *fp );
void  writeResult(Result *work, FILE *fp );
void  writeHistory(History *work, FILE *fp );
void  writeXinteg(Xinteg *work, FILE *fp );
void  writeDisplay(Display *work, FILE *fp );
void  writeNorm(Norm *work, FILE *fp );
FILE  *makeTempoFile(int n, char *fname );
void  initCheckTable(void);
void  makeExeAndExec(char *fn );
int   npe_link(char *line );
#if !defined(OSF1)
void  onquit();
void  onint();
void  setSignal(void);
#endif
void StoreBuffID(int ResultBuffID , int HistoryBuffID );

int main(int argc,char *argv[] )
{
  int    i, pid, pfd[2];
  Set    *SETp;
  FILE    *ftmp;
  int     windowFlag=TRUE;

  read_syscom(); /* read common area (SATELLITE) */
  renpe();       /* read common area (NPE) */

  ResultBuffID = GetBufferID(0);  /* get buffer ID to save the result  */
  HistoryBuffID = GetBufferID(1); /* get buffer ID to save the history */

  /* output link conditon */
  if((GetArgType(2) != '\0')){
    char *v_mode;

    v_mode = GetString(2);
    if((v_mode[0]=='v')||(v_mode[0]=='V'))
      Verbose_mode = TRUE;
  }

  /* ID check */
  if( ResultBuffID == 0 || HistoryBuffID == 0 )
    exit( 43 );

  /* write buffer ID to save the result and the history */
  StoreBuffID( ResultBuffID , HistoryBuffID );

  NumParallel = DEFALT_PARALLEL_SIZE;
  windowFlag = FALSE;
  
  PIDHEAD = NULL;

#if !defined(OSF1)
  setSignal();
#endif

  if( BlockHead == NULL )  exit( 3 );

  SETp = SetHead;
  NumProcess = 0;

  for( i=1; SETp != NULL; ) {
     for( ; (NumProcess<NumParallel) && (SETp!=NULL); i++,SETp=SETp->next ) {
      /* makeTempoFile : file name -> fname */
      ftmp = makeTempoFile( i, fname );
      initCheckTable();
      copyElements( SETp, ftmp );
      fclose( ftmp );

      switch( pid=fork() ) {
        case -1:
          exit( 8 );
        case 0:
          if( windowFlag ) {
            /* Replacing "stdout" -> pipe[output] */
            if( close(1) == -1 )    exit( 9 );
            if( dup(pfd[1]) == -1 ) exit( 10 );
            if( close(pfd[0]) == -1 || close(pfd[1]) == -1 )
              exit( 9 );
        }
          makeExeAndExec( fname );
      }
      registerPID( i, pid, ESTIMATOR );
    }
    waitUntilChildEnd();
  }
  waitUntilChildrenEnd();

  /****** test *****/  
  puts("Estimation is done.\n");

  return(0);
}



void  registerPID(int num, int pid, int type )
/* Making List of PID for Sending Signal */
{
  ID  *work;

  if((work = (ID *)malloc(sizeof(ID)))==NULL){
    printf("Out of memory\n");
  }
  work->num  = num;
  work->pid  = pid;
  work->type = type;
  work->next = PIDHEAD; /* move the current first element to second */
  PIDHEAD  = work;      /* change the heading */
  if( type == ESTIMATOR ) NumProcess++;
}


void  waitUntilChildrenEnd(void)
{
  for( ; waitUntilChildEnd() != -1; ) ;
}



int  waitUntilChildEnd(void)
{
  int  len, pid, status;
  char  num[10], dummy[MAXFNAME];
  ID  *work;

  strcpy( dummy, fname );

  pid = wait( &status );
  if( pid == -1 ) {
    return( -1 );
  }
  printStatus( pid, status );

  work = removePID( pid );
  if( work->type == ESTIMATOR ) {
    NumProcess--;
    len = strlen( fname );
    fname[len-1] = '\0';
    sprintf( num, "%d", work->num );
    if( Verbose_mode != TRUE ){
      strcat( fname, num );
      unlink( fname );  /** erase temporary file  **/
      strcat( fname, ".exe" );
      unlink( fname );  /** erase executable file **/
    }
  }
  free( work );

  strcpy( fname, dummy );
  return( 0 );
}


ID  *removePID(int pid )
{
  ID  *pre, *now;

  if( PIDHEAD == NULL ) {
    return( NULL );
  } else if( PIDHEAD->pid == pid ) {
    now  = PIDHEAD;
    PIDHEAD  = PIDHEAD->next;
    return( now );
  }
  pre = PIDHEAD;
  now = PIDHEAD->next;
  for( ; now != NULL; pre=now, now=now->next ) {
    if( now->pid == pid ) {
      pre->next = now->next;
      return( now );
    }
  }
  return( NULL );
}


void  copyElements(Set *setp, FILE *fp )
{
  writeMethod ( setp->methodp , fp ); elementCheckTable[METHOD]  = TRUE;
  writeLsearch( setp->lsearchp, fp ); elementCheckTable[LSEARCH] = TRUE;
  writeModel  ( setp->modelp  , fp ); elementCheckTable[MODEL]   = TRUE;
  writePenalty( setp->penaltyp, fp ); elementCheckTable[PENALTY] = TRUE;
  writeInit   ( setp->initp   , fp ); elementCheckTable[INIT]    = TRUE;
  writeScale  ( setp->scalep  , fp ); elementCheckTable[SCALE]   = TRUE;
  writeTerm   ( setp->termp   , fp ); elementCheckTable[TERM]    = TRUE;
  writeNumber ( setp->numberp , fp ); elementCheckTable[NUMBER]  = TRUE;
  writePoint  ( setp->pointp  , fp ); elementCheckTable[POINT]   = TRUE;
  writeData   ( setp->datap   , fp ); elementCheckTable[DATA]    = TRUE;
  writeWeight ( setp->weightp , fp ); elementCheckTable[WEIGHT]  = TRUE;
  writeResult ( setp->resultp , fp ); elementCheckTable[RESULT]  = TRUE;
  writeHistory( setp->historyp, fp ); elementCheckTable[HISTORY] = TRUE;
  writeXinteg ( setp->xintegp , fp ); elementCheckTable[XINTEG]  = TRUE;
  writeDisplay( setp->dispp   , fp ); elementCheckTable[DISPLAY] = TRUE;
  writeNorm   ( setp->normp   , fp ); elementCheckTable[NORM]    = TRUE;
}

void  writeMethod(Method  *work, FILE *fp )
{
  if( work == NULL ) {
    exit( 20 );
  }
  fprintf( fp, "%%method\n%s\n:\n", work->fname );
}


void  writeLsearch(Lsearch *work, FILE *fp )
{
  if( work != NULL ) {
    fprintf( fp, "%%lsearch\n%s,%g\n:\n",
      work->fname, work->value );
  }
}


void  writeModel(Model *work, FILE *fp )
{
  if( work == NULL ) {
    exit( 22 );
  }

  fprintf( fp, "%%model\n%d,", work->type );

  switch( work->type ) {
   case USR   :
     fprintf( fp, "%s\n:\n",    work->inform.Usr.fname );
     elementCheckTable[ XINTEG ] = TRUE;
     break;
   case NCS :
     fprintf( fp, "%s\n:\n",    work->inform.Ncs.fname );
     break;
  }
}


void  writePenalty(Penalty *work, FILE *fp )
{
  if( work != NULL )
    fprintf( fp, "%%penalty\n%s\n:\n", work->fname );
}


void  writeInit(Init *work, FILE *fp )
{
  struct estparam  *w;

  if( work == NULL ) {
    exit( 23 );
  }

  fprintf( fp, "%%init\n%d\n", work->paramnum );
  for( w = work->EstParam; w != NULL; w = w->next ) {
    fprintf( fp, "%15.8g,%c,%s,%g\n", w->value,
      ((w->flag==FIX)?'F':'V'), w->name,  w->span );
  }
  fprintf( fp, ":\n" );
}


void  writeScale(Scale *work, FILE *fp )
{
  if( work != NULL )
    fprintf( fp, "%%scale\n%d\n:\n", work->scalemethod );
  else
    fprintf( fp, "%%scale\n%d\n:\n", 0);
}


void  writeTerm(Term *work, FILE *fp )
{
  struct termcrite *w;

  if( work == NULL ) {
    exit( 25 );
  }
  fprintf( fp, "%%term\n%s\n", work->logic );
  for( w = work->TermCrite; w != NULL; w = w->next ) {
    fprintf( fp, "%d,%g\n",
      w->critenum, w->critevalue );
  }
  fprintf( fp, ":\n" );
}


void  writeNumber(Number *work, FILE *fp )
{
  if( !elementCheckTable[POINT] ) {
    if( work == NULL ) {
      exit( 26 );
    }
    fprintf( fp, "%%number\n%d\n:\n", work->numwave );
  }
}


void  writePoint(Point *work, FILE *fp )
{
  if( !elementCheckTable[POINT] ) {
    if( work == NULL ) {
      exit( 27 );
    }
    fprintf( fp, "%%point\n%d\n:\n", work->datapoint );
  }
}


void  writeData(Experiment *work, FILE *fp )
{
  if( !elementCheckTable[DATA] ) {
    if( work == NULL ) {
      exit( 28 );
    }
    fprintf( fp, "%%data\n%s,%d\n:\n", work->fname, work->recnum );
  }
}


void  writeWeight(Weightt *work, FILE *fp )
{
  if( !elementCheckTable[WEIGHT] ) {
    if( work == NULL ) {
      exit( 29 );
    }
    fprintf( fp, "%%weight\n%s,%d\n:\n", work->fname, work->recnum );
  }
}


void  writeResult(Result *work, FILE *fp )
{
  if( !elementCheckTable[RESULT] ) {
    if( work == NULL )  exit( 30 );
    fprintf( fp, "%%result\n%s,%d\n:\n",
      work->fname, work->storeinterval );
  }
}


void  writeHistory(History *work, FILE *fp )
{
  if( work == NULL ) {
    exit( 31 );
  }
  fprintf( fp, "%%history\n%s,%d\n:\n",
    work->fname, work->storeinterval );
}


void  writeXinteg(Xinteg *work, FILE *fp )
{
  if( !elementCheckTable[XINTEG] ) {
    if( work == NULL ) exit( 32 );
    fprintf( fp, "%%xinteg\n%c\n:\n", work->integmethod );
  }
}


void  writeDisplay(Display *work, FILE *fp )
{/* Default : With Weight */
  if( work != NULL )
    fprintf( fp, "%%display\n%d\n:\n", work->type );
  else
    fprintf( fp, "%%display\n%d\n:\n", WITH_WEIGHT );
}


void  writeNorm(Norm *work, FILE *fp )
{/* Default : 2-norm */
  if( work != NULL )
    fprintf( fp, "%%norm\n%d\n:\n", work->type );
  else
    fprintf( fp, "%%norm\n%d\n:\n", NORM_TWO );
}


#define Temp_File_Head   "USR"

FILE  *makeTempoFile(int n, char *fname )
{
  int  pid;    /* PID */
  char  dummy[MAXLINE];
  FILE  *ftmp;

  strcpy ( fname,  Temp_File_Head  );   /* fname = Temp_File_Head */
  pid = getpid();

  sprintf( dummy, "%d", pid );
  strcat ( fname,  dummy  );   /* fname = Temp_File_Headxxxx */
  sprintf( dummy, "%d", n );
  strcat ( fname,  dummy  );   /* fname = Temp_File_Headxxxxn */
  if( (ftmp=fopen( fname, "w" )) == NULL ) {
    exit( 40 );
  }
  return( ftmp );
}

void  initCheckTable(void)
{
  int  i;

  for( i=0; i<ELEMENTNUMBER+1; i++ )
    elementCheckTable[i] = FALSE;
}

/*********************************************
**  main()-fork()-makeExeAndExec()          **
**    "stdout" was replaced to pipe[output] **
*********************************************/
static char  path [] = NPE_LIBRARYPATH;
static char CC[] = NPE_CC;
static char Flag1[] = NPE_COMPILEOPT;
static char Flag2[] = "";
static char usr[] = " -lNPEusr ";
static char ncs[] = " -lNPEncs -lNCSnpe -lncsc -lnscf";
static char diff[]    = "diff.o ";
static char diff2[]   = "secdiff.o ";
static char penanull[]= "null2.o";
static char disp[] = "display.o ";

char  exe[MAXFNAME];

void  makeExeAndExec(char *fn )
{
  char  line[ 800 ];
  char    Result_ID[10];
  char    History_ID[10];

  
  renpe_( fn );
  strcpy( exe, fn );
  strcat( exe, ".exe"  );
  /***************************
  ** Making Executable File **
  ****************************/
  printStatus( getpid(), npe_link(line) );

  sprintf(Result_ID , " %d" , ResultBuffID );
  sprintf(History_ID , " %d" , HistoryBuffID );
  if( Verbose_mode == TRUE )
    printf("%s  %s  %s  %s\n",exe, fn, Result_ID, History_ID);

#if defined(SunOS) || defined(HP_UX)
  execl( exe, exe, fn, Result_ID, History_ID, (char *)0 );
#else
  execl( exe, exe, fn, Result_ID, History_ID, NULL );
#endif

  exit( 205 );
}


int  npe_link(char *line )
{
  int  ret;

  strcpy( line, CC    );
  strcat( line, Flag1  );
  strcat( line, Flag2 );
  strcat( line, path  );  strcat( line, MethodHead->fname );
  strcat( line, ".o"  );
  strcat( line, path  );  strcat( line, disp   );
  strcat( line, path  );  strcat( line, diff   );
  strcat( line, path  );  strcat( line, diff2  );
  if( LsearchHead != NULL ) {
    strcat( line, path  );  strcat( line, LsearchHead->fname );
    strcat( line, ".o " );
  }
  if( PenaltyHead != NULL ) {
    strcat( line, PenaltyHead->fname );
  } else {
    strcat( line, path  );
    strcat( line, penanull );
  }
  strcat( line, " " );

  switch( ModelHead->type ) {
    case USR :
      strcat( line,ModelHead->inform.Usr.fname );
      strcat( line, usr );
      break;
    case NCS : 
      strcat( line, ModelHead->inform.Ncs.fname );
      strcat( line, ncs );
      break;
  }
  strcat( line, " -lm -lsatellite -o " );
  strcat( line, exe );

  if( Verbose_mode == TRUE )  puts( line );

  printf( " Making Executable file " ); fflush( stdout );
  if ( (ret=system(line)) == 0 ) {
    puts( "is done." );
    fflush( stdout );
    putchar( '\n' );
  }
  return( ret );
}

#if !defined(OSF1)
void  onquit()
{
  int  i, n, len;
  char  num[10];
  ID  *work;

  for( work = PIDHEAD; work != NULL; work = work->next )
    kill( work->pid, SIGUSR1 );

  len = strlen( fname );
  n   = fname[len-1]-'0';
  for( i = n; i>=0; i-- ) {
    fname[len-1] = '\0';
    sprintf( num, "%d", i );
    strcat( fname, num );
    unlink( fname );
    strcat( fname, ".exe" );
    unlink( fname );
  }
  exit( 207 );
}
#endif

#if !defined(OSF1)
void  onint()
{
  ID  *work;

  for( work = PIDHEAD; work != NULL; work = work->next )
    kill( work->pid, SIGUSR1 );

  exit( 206 );
}
#endif

#if !defined(OSF1)
void  setSignal(void)
{
  signal( SIGUSR1, onquit );
  signal( SIGINT,  onint  );
  signal( SIGQUIT, onint  );
  signal( SIGABRT, onint  );
}
#endif

/**************************************************************************/
#define  MAXSIG  19
#define lowbyte(w)  ((w)&0x00ff)
#define highbyte(w) lowbyte((w)>>8)

void  printStatus(int pid, int status )
{
  int  code;
  static char *sigmsg[] = {
    "", "Hangup", "Interruput", "Quit", "Illegal Instruction",
    "Trace trap", "IOT Instruction", "EMT Instruction",
    "Floating Point Exception", "Kill", "Bus Error",
    "Segmentation Violation", "Bad Arg to System Call",
    "Write on Pipe", "Alarm Clock", "Terminate Signal",
    "User Signal 1", "User Signal 2", "Death of Child", 
    "Power Fail"
  };
  
  if( lowbyte(status) != 0 ) {
    if( (code = status &0x007f) <= MAXSIG )
      printf( "%s\n", sigmsg[code] );
  }
}

/**************************************************************************/

void StoreBuffID(int ResultBuffID , int HistoryBuffID )
{
  Result *wRe;        /* work */
  History *wHis;      /* work */


  for( wRe = ResultHead ; wRe != NULL ; wRe = wRe->next )
      wRe->buffnum = ResultBuffID;

  for( wHis = HistoryHead ; wHis != NULL ; wHis = wHis->next )
      wHis->buffnum = HistoryBuffID;
}
