/**************************************************
**  brent.c                                      **
**  parameter estimation method                  **
**************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "LIB.h"
#include "NPEcmd.h"
#include "ex_global_vars.h"

/* use in bracketing() */
#define  GOLD    1.618034
#define  GLIMIT  100.0
#define  TINY    1.0e-20

#define  CGOLD    0.3819660
#define  TOL     1.0e-12
#define  ZEPS    1.0e-10
#define  ITMAX  100

#define  SHFT(a,b,c,d)   (a)=(b); (b)=(c); (c)=(d);
#define  SIGN(a,b)  ((b) >= 0.0 ? fabs(a) : -fabs(a))

/* global variable */
extern double  *Direction;

/* function prototype */
void  interpolation(void);
void  bracketing(double *ax, double *bx, double *cx, double *Xk);


/* function body */
void  interpolation(void)
{
  int  i, iter;
  double  a, b, d = 0.0, etemp, fu, fv, fw, fx, p, q, r;
  double  tol1, tol2, u, v, w, x, xm;
  double  e = 0.0;
  double  ax, bx, cx;
  double  *Xk;

   if((Xk = (double *)malloc(sizeof(double)*NumVarParam)) == NULL){
      printf("Error : Out of Memory\n");
      exit(1);
   }

   /* set the both side position at the initial condition */
  ax = 0.0;
  bx = LsearchHead->value;
  for(i=0;i<NumVarParam;i++)  Xk[i] = VarParam[i]->value;

  /* cx is decided in bracketing function */
  bracketing(&ax,&bx,&cx,Xk); 


  /* naked brent method */

  a = ( ax < cx ? ax : cx );   /* to be a < b  */
  b = ( ax > cx ? ax : cx );

  x = w = v = bx;              /* initialization */

  for (i=0;i<NumVarParam;i++)
    VarParam[i]->value = Xk[i] + x*Direction[i];
  fw = fv = fx  = errorFunc( AllParam );


  for(iter=0; iter < ITMAX; iter++){
    printf("."); fflush( stdout );
    xm = 0.5*( a+b );
    tol2 = 2.0*(tol1 = TOL*fabs(x) + ZEPS);

    /* judge the convergence */
    if( fabs(x-xm) <= (tol2 - 0.5*(b - a)) ){
      for (i=0;i<NumVarParam;i++)
	VarParam[i]->value = Xk[i] + x*Direction[i];
      ErrorValue = fx;
      free(Xk);
      printf("\n");
      return;
    }

    /* interpolate by using the before updated value */
    if( fabs(e) > tol1 ){
      r = (x - w)*(fx - fv);
      q = (x - v)*(fx - fw);
      p = (x - v)*q - (x - w)*r;
      q = 2.0*(q - r);
      if( a > 0.0 ) p = -p;
      q = fabs(q);
      etemp = e;
      e = d;
      if( fabs(p) >= fabs(0.5*q*etemp) || (p <= q*(a - x)) || (p >= q*(b - x)))
        d = CGOLD*(e=(x >= xm ? a-x : b-x));
      else {
        d = p/q;
        u = x + d;
        if((u-a < tol2) || ( b-u < tol2))
          d = SIGN(tol1,xm-x);
      }
    } else {
      d = CGOLD*( e = (x >= xm ? a-x : b-x));
    }
    u = (fabs(d) >= tol1 ? x+d : x+SIGN(tol1,d));
    /* evaluate the function value */
    for (i=0;i<NumVarParam;i++)
      VarParam[i]->value = Xk[i] + u*Direction[i];
    fu = errorFunc( AllParam );

    if( fu <= fx){
      if( u >= x ) a = x; else b = x;
      SHFT(v,w,x,u)
      SHFT(fv,fw,fx,fu)
    } else {
      if(u < x) a = u; else b = u;
      if((fu <= fw) || (w == x)) {
        v=w;
        w=u;
        fv=fw;
        fw=fu;
      } else if((fu <= fv) || (v == x) || (v == w)){
	v=u;
	fv=fu;
      }
    }
  }

  printf("\nWarrning: Over Max iteration[%d] ... why ?\n", ITMAX);

  for (i=0;i<NumVarParam;i++)
      VarParam[i]->value = Xk[i] + x*Direction[i];
  ErrorValue = fx;
  free(Xk);
} 


void bracketing(double *ax, double *bx, double *cx, double *Xk)
{
  int  i;
  double  fa, fb, fc;
  double  ulim, u,r,q,fu,dum,tmp;


  if( Xk == NULL ){
    printf(" Error : tmporal vector is NULL in linear search\n");
    exit(1);
  }

  for(i=0;i<NumVarParam;i++)
    VarParam[i]->value = Xk[i] + (*ax)*Direction[i];
  fa = errorFunc( AllParam );  /* fa = func(ax) */

  for(i=0;i<NumVarParam;i++)
    VarParam[i]->value = Xk[i] + (*bx)*Direction[i];
  fb = errorFunc( AllParam ); /* fb = func(bx) */

  if( fb > fa ){
      dum = *ax;  *ax = *bx;
      *bx = dum;
      dum = fb; fb = fa;
      fa = dum;  
  }

  *cx = (*bx) + GOLD*(*bx - *ax); /* initial estimated value of cx */

  for(i=0;i<NumVarParam;i++)
    VarParam[i]->value = Xk[i] + (*cx)*Direction[i];
  fc = errorFunc( AllParam );    /* fc = func(cx) */

  while( fb > fc ){ /* iterate until enclose */

    /* cal. u from a,b,c */
    r = (*bx - *ax)*(fb - fc);
    q = (*bx - *cx)*(fb - fa);
    if(fabs(q-r) > TINY) tmp = fabs(q-r);
    else tmp = TINY;
    if( q-r > 0.0 ) dum = fabs(tmp);
    else dum = -fabs(tmp);
    dum *= 2.0;
    u = (*bx) - ((*bx - *cx)*q - (*bx - *ax)*r)/dum;
    /* avoid to divide by zero */
    ulim = (*bx) + GLIMIT*(*cx - *bx);
    if( (*bx - u)*(u - *cx) > 0.0 ){ /* b < u < c */
      for(i=0;i<NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = errorFunc( AllParam ); /* fu = func(u) */
      if( fu < fc ){          /* found a minimum value between b and c  */
        *ax = (*bx);
        *bx = u;
        fa = fb;
        fb = fu;
        return;
      } else if( fu > fb ) {  /* found a minimum value between a and u  */
        *cx = u;
        fc = fu;
        return;
      }
      u = (*cx) + GOLD*(*cx - *bx);
      for(i=0;i<NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = errorFunc( AllParam );
    } else if((*cx - u)*(u - ulim) > 0.0){
      for(i=0;i<NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = errorFunc( AllParam );

      if( fu < fc ){
        *bx = *cx; *cx = u;
        u = (*cx + GOLD*(*cx - *bx));

        for(i=0;i<NumVarParam;i++)
          VarParam[i]->value = Xk[i] + u*Direction[i];
        tmp = errorFunc( AllParam );
        fb = fc; fc = fu; fu = tmp;
      }
    } else if((u - ulim)*(ulim - *cx) >= 0.0) {

      u = ulim;
      for(i=0;i<NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = errorFunc( AllParam );

    } else {
      u = (*cx) + GOLD*(*cx - *bx);
      for(i=0;i<NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = errorFunc( AllParam );
    }
    *ax = *bx; *bx = *cx; *cx = u;
    fa = fb; fb = fc; fc = fu;
  }
}
