/**************************************************************
**  conjfr.c                                                 **
**  conjugate gradient method using fletcher-reeves formula  **
**************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "NPE.h"
#include "NPEcmd.h"
#include "global_vars.h"

#define MAXITERATION	(10000)

/* usually RESTART sets to n, n+1, 2n as n:parameter num. */
#define	RESTART		(NumVarParam)

/* global vars in this file */
double  Beta, *Gamma;         /* for Beta */
double  *nablaF, *Direction;  /* nabla f(x), & Direction vector */

/* function prototype */
#if defined(ANSI)
void  conjGradientInit(void);
void  calcDirection(void);
void  linearSearch(void);
void  renewBeta(void);
void  StoreBuffID();
void  renpe_(char *filename);
void  initHead(void);

#else
void  conjGradientInit();
void  calcDirection();
void  linearSearch(void);
void  renewBeta(void);
void  StoreBuffID();
void  renpe_();
void  initHead();

#endif /* ANSI */

/*********************************************************************
**  Main routine of quasi-Newton Method                             **
**********************************************************************/
int main(int argc, char **argv )
{
  void  cleanUp(char **argv);
  int   ResultBuffID , HistoryBuffID;


  if(argc < 4 ){
    printf("Error too few argument\n");
    exit(1);
  }

  renpe_( argv[1] );

  ResultBuffID = atoi(argv[2]);
  HistoryBuffID = atoi(argv[3]);
  StoreBuffID(ResultBuffID , HistoryBuffID);

/* start up message */
  printf("[ NPE ] CONJFR\n");
  fflush(stdout);


  conjGradientInit();

  for( Step=0; Step < MAXITERATION; Step++ ) {
    display( AllParam, DispValue );

    storeParamHist( AllParam, FALSE );

    /* whether stop condition has been satisfied or not */
    if( term() ) break; 

    scaleTrans( VarParam );

    calcDirection();
    linearSearch();
    renewBeta();
  }
  storeParamHist( AllParam, TRUE );
  cleanUp( argv );
  initHead();
  return( 0 );
}

/*********************************************************************
**  Subroutine of Initializition                                    **
**********************************************************************/
void  conjGradientInit(void)
{
  int  i;

  readData();
  readParam();
  readTermCriterion();
  readSystemParam( &samp );
  readStoreFile();
  readScaleMethod();

  Beta = 0.0;
  nablaF    = (double *) malloc2Dim( 0, NumVarParam, 'D' );
  Gamma     = (double *) malloc2Dim( 0, NumVarParam, 'D' );
  Direction = (double *) malloc2Dim( 0, NumVarParam, 'D' );

  ErrorValue = errorFunc( AllParam );
  DispValue = tmpDisp;
  PenaltyValue = tmpPena;
  differentiate( VarParam, nablaF, NULL );
  for( i=0; i<NumVarParam; i++ )
    Direction[i] =  0.0;
}

/*********************************************************************
**  Subroutine of Calculation of Direction                          **
**********************************************************************/
void  calcDirection(void)
{
  int  i;
  double  D=0.0;

  for( i=0; i<NumVarParam; i++ ) {
    Direction[i] = - nablaF[i] + Beta * Direction[i];
    D += square( Direction[i] );
  }

  if( D != 0.0 ) {
    D = 1.0 / sqrt( D );
    for( i=0; i<NumVarParam; i++ )
      Direction[i] *= D;     /* normalization */
  }
}

/*********************************************************************
**  Subroutine of LinearSearch                                      **
**********************************************************************/
void  linearSearch(void)
{
  int  i;

/*  kepp a gradient at X_k-1. Use in the cal. of beta */
  for( i=0; i<NumVarParam; i++ )
    Gamma[i] = nablaF[i];    /* Gamma = dF(Xk-1) */

  /* optimization by means of linearity */
  interpolation();
  DispValue = tmpDisp;
  PenaltyValue = tmpPena;
}


/*********************************************************************
**  Subroutine of Calculation of Beta : Fletcher-Reeves             **
**********************************************************************/
void  renewBeta(void)
{
  int  i;
  double  norm2 = 0.0; /* keep before norm^2 */

  Beta = 0.0;

  differentiate( VarParam, nablaF, NULL ); 

  if( (Step+1) % RESTART == 0 ) {
    /* restart as \beta_0 = 0.0 */
    Beta = 0.0;
  } else {
/*-------------------------------------------------------------------
             || nablaF(X[k]) ||^2
  Beta = --------------------------
            || nablaF(X[k-1] ) ||^2
--------------------------------------------------------------------*/
    for( i=0; i<NumVarParam; i++ )
      norm2 += square( Gamma[i] );

    if( norm2 == 0.0 ){ /* infrequently gradient = 0 */
      fprintf(stderr,"Derivates are zero.\n");
      fprintf(stderr,"I guess the estimation is done. Now storing results\n");
      storeParamHist( AllParam, TRUE );
      initHead();
      exit(0);
    }

    for( i=0; i<NumVarParam; i++ )
      Beta += square( nablaF[i] );

    Beta /= norm2;
  }
}


void StoreBuffID(int ResultBuffID , int HistoryBuffID )
{
  Result *wRe;        /* work */
  History *wHis;      /* work */

  for( wRe = ResultHead ; wRe != NULL ; wRe = wRe->next )
      wRe->buffnum = ResultBuffID;

  for( wHis = HistoryHead ; wHis != NULL ; wHis = wHis->next )
      wHis->buffnum = HistoryBuffID;

}
