/**************************************************
**  dfp.c                                        **
**  parameter estimation method                  **
**************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "NPE.h"
#include "NPEcmd.h"
#include "global_vars.h"

#define MAXITERATION  (10000)
/* #define TINY 1.0e-10 */

/* global vars in this file */
double  **InvHess, *Dif_x, *Dif_nabla, *H_Dif_nabla;  /* for Inverse Hessian */
double  *nablaF, *Direction;     /* nabla f(x), & Direction vector */


/* function prototypes */
#if defined(ANSI)
void  quasiNewtonInit(void);
void  initHessian(double  **Hesse);
void  calcDirection(void);
BOOL  checkDescentDir(void);
BOOL  overRestart( int count );
void  linearSearch(void);
void  renewHessian(void);
void  StoreBuffID(int ResultBuffID, int HistoryBuffID );
void  renpe_(char *filename);
void  initHead(void);
void free_quasi_Newton(void);

#else /* traditional */
void  quasiNewtonInit();
void  initHessian();
void  calcDirection(void);
BOOL  checkDescentDir();
BOOL  overRestart();
void  linearSearch();
void  renewHessian();
void  StoreBuffID();
void  renpe_();
void  initHead();
void free_quasi_Newton();

#endif /* ANSI */

/*********************************************************************
**  Main routine of quasi-Newton Method                             **
**********************************************************************/
int main(int argc, char *argv[] )
{
  void  cleanUp(char **argv);
  int  restartCount;
  double  Old_ErrorVal;
  int     ResultBuffID , HistoryBuffID;

  if(argc < 4 ){
    fprintf(stderr,"Error too few argument\n");
    exit(1);
  }

  renpe_( argv[1] );


/* start up message */
  printf("[ NPE ] DFP \n");
  fflush(stdout);


  quasiNewtonInit();
  Old_ErrorVal = ErrorValue;
  calcDirection();

  ResultBuffID = atoi(argv[2]);
  HistoryBuffID = atoi(argv[3]);
  StoreBuffID(ResultBuffID , HistoryBuffID);

  for( Step=0; Step < MAXITERATION; Step++ ) {

    display( AllParam, DispValue );

    storeParamHist( AllParam, FALSE );

    /* whether stop condition has been satisfied or not */
    if( term() ) break;

    scaleTrans( VarParam );

    if( (!checkDescentDir())||(Old_ErrorVal<ErrorValue) ){
      for( restartCount=0; ; restartCount++ ) {
        fprintf(stderr,"I seem Hess matrix is not positive define...\n");
        fprintf(stderr,"Initilize Hess matrix\n");
        initHessian( InvHess );
        calcDirection();
        if( checkDescentDir() == TRUE )
          break;
        if( overRestart(restartCount) == TRUE ) {
          storeParamHist( AllParam, TRUE );
          cleanUp( argv );
          initHead();
          free_quasi_Newton();
          return( 5 );
        }
      }
    }
    Old_ErrorVal = ErrorValue;
    linearSearch();  /* linea search */
    renewHessian();  /* modify the inv. Hess matrix */
    /* decide the direction by using modified Hess matrix */
    calcDirection();
  }
  storeParamHist( AllParam, TRUE );
  cleanUp( argv );
  initHead();
  free_quasi_Newton();
  return( 0 );
}

/*********************************************************************
**  Subroutine of Initializition                                    **
**********************************************************************/
void  quasiNewtonInit(void)
{

  readData();
  readParam();

  readTermCriterion();
  readSystemParam( &samp );
  readStoreFile();
  readScaleMethod();

  InvHess = (double **)malloc2Dim( NumVarParam, NumVarParam, 'D' );
  nablaF  = (double *) malloc2Dim( 0, NumVarParam, 'D' );
  Dif_x   = (double *) malloc2Dim( 0, NumVarParam, 'D' );
  Dif_nabla  = (double *) malloc2Dim( 0, NumVarParam, 'D' );
  H_Dif_nabla  = (double *) malloc2Dim( 0, NumVarParam, 'D' );
  Direction = (double *) malloc2Dim( 0, NumVarParam, 'D' );

  ErrorValue = errorFunc( AllParam );
  DispValue = tmpDisp;    /* tmpDisp and tmpPena is cal. inerrorFunc() */
  PenaltyValue = tmpPena;
  differentiate( VarParam, nablaF, NULL );
  initHessian( InvHess );
}


void free_quasi_Newton(void)
{
  int i;

  for(i=0; i < NumVarParam; i++)
    free(InvHess[i]);

  free( InvHess );
  free( nablaF );
  free( Dif_x );
  free( Dif_nabla );
  free( H_Dif_nabla );
  free( Direction );
}

/*********************************************************************
**  Subroutine of Initializition of Hessian                         **
**********************************************************************/
void  initHessian(double  **Hesse )
{
  int  i, j;

  for( i=0; i<NumVarParam; i++ ) {
    for( j=0; j<NumVarParam; j++ ) {
      Hesse[i][j] = 0.0;
    }
  }
  for( i=0; i<NumVarParam; i++ )
      Hesse[i][i] = 1.0;
}

/*********************************************************************
**  Subroutine of Calculation of Direction                          **
**********************************************************************/
void  calcDirection(void)
/* Direction : global var. */
{
  int  i, j;
  double  D;

  D = 0.0;
  for( i=0; i<NumVarParam; i++ ) {
    Direction[i] = 0.0;
    for( j=0; j<NumVarParam; j++ )
      Direction[i] -= InvHess[i][j] * nablaF[j];

    D += Direction[i]*Direction[i];
  }

  if( D != 0.0 ){

    D = 1.0/sqrt(D);
    for( i=0; i<NumVarParam; i++ )
      Direction[i] *= D;
  }

}

/*********************************************************************
**  Subroutine of Checking Descent Direction                        **
**********************************************************************/
BOOL  checkDescentDir(void)
{
  int  i;
  double  D_ = 0.0;

  for( i=0; i<NumVarParam; i++ )
    D_ += nablaF[i] * Direction[i];

  if( D_ < 0.0 )
    return( TRUE  );
  else  return( FALSE );
}

/*********************************************************************
**  Subroutine of Checking Number of Restart                        **
**********************************************************************/
BOOL  overRestart( int count )
{
  if( count >= 3 )   return( TRUE  );
  return( FALSE );
}


/*********************************************************************
**  Subroutine of LinearSearch                                      **
**********************************************************************/
void  linearSearch(void)
{
  int  i;
  double norm,norm2;

  for( i=0; i<NumVarParam; i++ ) {
    Dif_x[i] = -VarParam[i]->value;  /* Dif_x = Xk+1-Xk       */
    Dif_nabla[i] = -nablaF[i];       /* Dif_nabla = dF(Xk+1)-dF(Xk) */
  }

  interpolation();
  DispValue = tmpDisp;
  PenaltyValue = tmpPena;
  differentiate( VarParam, nablaF, NULL );

  norm = 0.0;
  norm2 = 0.0;
  for( i=0; i<NumVarParam; i++ ) {
    Dif_x[i] += VarParam[i]->value;  /* Dif_x = Xk+1-Xk     */
    Dif_nabla[i] += nablaF[i];       /* Dif_nabla = dF(Xk+1)-dF(Xk) */
    norm += Dif_nabla[i]*Dif_nabla[i];
    norm2 += Dif_x[i]*Dif_x[i];
  }
  if(norm*norm2 == 0.0){
    fprintf(stderr,"Too few changes... I can't update Hess Matrix\n");
    fprintf(stderr,"I guess the estimation is done. Now storing results\n"); 
    storeParamHist( AllParam, TRUE );
    initHead();
    free_quasi_Newton();
    exit(0);
  }
}

/*********************************************************************
*  Subroutine to renew Inverse Hessian : DFP                         *
**********************************************************************/
void  renewHessian(void)
{
  int  i, j;
  double  sy ,yHy;

/*-------------------------------------------------------------------
  Hy = Hessian*y  yHy = <y,Hy>
--------------------------------------------------------------------*/

  yHy = 0.0;
  for( i=0; i<NumVarParam; i++ ){
    H_Dif_nabla[i] = 0.0;
    for( j=0; j<NumVarParam; j++ )
      H_Dif_nabla[i] += InvHess[i][j]*Dif_nabla[j];

    yHy += Dif_nabla[i]*H_Dif_nabla[i];
  }

/*-------------------------------------------------------------------
  sy = Dif_x^T * Dif_nabla
--------------------------------------------------------------------*/
  sy = 0.0;
  for( i=0; i<NumVarParam; i++ )
    sy += Dif_x[i]*Dif_nabla[i];

/*-------------------------------------------------------------------
  InvHess = InvHess - <Hy, yH>/yHy + <Dif_x, Dif_x>/sy
--------------------------------------------------------------------*/
  for( i=0; i<NumVarParam; i++ ){
    for( j=0; j<i; j++ ) {
      InvHess[i][j] += - H_Dif_nabla[i]*H_Dif_nabla[j]/yHy
                       + Dif_x[i]*Dif_x[j]/sy;
      InvHess[j][i] = InvHess[i][j];
    }
  }
}


void  StoreBuffID(int ResultBuffID, int HistoryBuffID )
{
  Result *wRe;        /* work */
  History *wHis;      /* work */


  for( wRe=ResultHead ; wRe!=NULL ; wRe=wRe->next )
      wRe->buffnum = ResultBuffID;

  for( wHis=HistoryHead ; wHis!=NULL ; wHis=wHis->next )
      wHis->buffnum = HistoryBuffID;
}
