/**************************************************
**  golden.c                                     **
**  linear search method                         **
**************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "LIB.h"
#include "NPEcmd.h"
#include "ex_global_vars.h"

#define  GOLD    1.618034
#define  GLIMIT  100.0
#define  TINY    1.0e-20
#define  R       0.61803399
#define  C       (1.0 - R)
#define  Max_It  100
#define  TOL     1.0e-12
#define  ZEPS    1.0e-10


/* global variable */
extern double  *Direction;

/* function prototype */
void  interpolation(void);
void  bracketing(double *ax, double *bx, double *cx, double *Xk);


/* function body */
void  interpolation(void)
{
  int  i, iteration=0;
  double  x0, x1, x2, x3, f1, f2,ftmp;
  double  ax, bx, cx;
  double  *Xk;

   if((Xk = (double *)malloc(sizeof(double)*NumVarParam)) == NULL){
      printf("Error : Out of Memory\n");
      exit(1);
   }

  ax = 0.0;
  bx = LsearchHead->value;
  for(i=0;i<NumVarParam;i++)  Xk[i] = VarParam[i]->value;

  bracketing(&ax,&bx,&cx,Xk);

  x0 = ax;
  x3 = cx;
  if(fabs(cx - bx) > fabs(bx - ax)){
    x1 = bx;
    x2 = bx + C*(cx - bx);  /* new point */
  } else {
    x2 = bx;
    x1 = bx - C*(bx - ax);
  }

  for (i=0;i<NumVarParam;i++)
    VarParam[i]->value = Xk[i] + x1*Direction[i];
  f1 = errorFunc( AllParam );

  for (i=0;i<NumVarParam;i++)
    VarParam[i]->value = Xk[i] + x2*Direction[i];
  f2 = errorFunc( AllParam );

  while( fabs(x3 - x0) > TOL*( fabs(x1)+fabs(x2) )/2.0 + ZEPS){
    printf("."); fflush( stdout );
    if( f2 < f1 ){
      x0 = x1; x1 = x2; x2 = R*x1 + C*x3;

      for (i=0;i<NumVarParam;i++)
        VarParam[i]->value = Xk[i] + x2*Direction[i];
        ftmp = errorFunc( AllParam );
      f1 = f2; f2 = ftmp;
    } else {
      x3 = x2; x2 = x1; x1 = R*x2+C*x0;

      for (i=0;i<NumVarParam;i++)
        VarParam[i]->value = Xk[i] + x1*Direction[i];
      ftmp = errorFunc( AllParam );
      f2 = f1; f1 = ftmp;
    }
    if(iteration++ >= Max_It){
      printf("\nWarrning: Over Max iteration[%d] ... why ?\n", Max_It);
      break;
    }
  }
printf("\n");

  if(f1 < f2) {
    for (i=0;i<NumVarParam;i++)
      VarParam[i]->value = Xk[i] + x1*Direction[i];
     ErrorValue = f1;
  } else {
    for (i=0;i<NumVarParam;i++)
      VarParam[i]->value = Xk[i] + x2*Direction[i];
     ErrorValue = f2;
  }
  free(Xk);
} 


void bracketing(double *ax, double *bx, double *cx, double *Xk)
{
  int  i;
  double  fa, fb, fc;
  double  ulim, u,r,q,fu,dum,tmp;


  if( Xk == NULL ){
    printf(" Error : tmporal vector is NULL in linear search\n");
    exit(1);
  }

  for(i=0;i<NumVarParam;i++)
    VarParam[i]->value = Xk[i] + (*ax)*Direction[i];
  fa = errorFunc( AllParam );  /* fa = func(ax) */

  for(i=0;i<NumVarParam;i++)
    VarParam[i]->value = Xk[i] + (*bx)*Direction[i];
  fb = errorFunc( AllParam ); /* fb = func(bx) */

  if( fb > fa ){
      dum = *ax;  *ax = *bx;
      *bx = dum;
      dum = fb; fb = fa;
      fa = dum;  
  }

  *cx = (*bx) + GOLD*(*bx - *ax); /* initial estimated value of cx */

  for(i=0;i<NumVarParam;i++)
    VarParam[i]->value = Xk[i] + (*cx)*Direction[i];
  fc = errorFunc( AllParam );     /* fc = func(cx) */

  while( fb > fc ){ /* iterate until enclose */

    /* cal. u from a,b,c */
    r = (*bx - *ax)*(fb - fc);
    q = (*bx - *cx)*(fb - fa);
    if(fabs(q-r) > TINY) tmp = fabs(q-r);
    else tmp = TINY;
    if( q-r > 0.0 ) dum = fabs(tmp);
    else dum = -fabs(tmp);
    dum *= 2.0;
    u = (*bx) - ((*bx - *cx)*q - (*bx - *ax)*r)/dum;
    /* avoid to divide by zero */
    ulim = (*bx) + GLIMIT*(*cx - *bx);
    if( (*bx - u)*(u - *cx) > 0.0 ){ /* b < u < c */
      for(i=0;i<NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = errorFunc( AllParam );    /* fu = func(u) */
      if( fu < fc ){          /* found a minimum value between b and c  */
        *ax = (*bx);
        *bx = u;
        fa = fb;
        fb = fu;
        return;
      } else if( fu > fb ) {  /* found a minimum value between a and u  */
        *cx = u;
        fc = fu;
        return;
      }
      u = (*cx) + GOLD*(*cx - *bx);
      for(i=0;i<NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = errorFunc( AllParam );
    } else if((*cx - u)*(u - ulim) > 0.0){
      for(i=0;i<NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = errorFunc( AllParam );

      if( fu < fc ){
        *bx = *cx; *cx = u;
        u = (*cx + GOLD*(*cx - *bx));

        for(i=0;i<NumVarParam;i++)
          VarParam[i]->value = Xk[i] + u*Direction[i];
        tmp = errorFunc( AllParam );
        fb = fc; fc = fu; fu = tmp;
      }
    } else if((u - ulim)*(ulim - *cx) >= 0.0) {
      u = ulim;
      for(i=0;i<NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = errorFunc( AllParam );
    } else {
      u = (*cx) + GOLD*(*cx - *bx);
      for(i=0;i<NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = errorFunc( AllParam );
    }
    *ax = *bx; *bx = *cx; *cx = u;
    fa = fb; fb = fc; fc = fu;
  }
}
