#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "NPE.h"
#include "NPEcmd.h"
#include "global_vars.h"

#define	MAXITERATION	10000 
#define	ALPHA	(1.0)	/* 0 <  ALPHA      */ 
#define	BETA	(0.5)	/* 0 <  BETA  < 1  */
#define	GAMMA	(2.0)	/* 1 <  GAMMA      */
#define	Value(pp)	(pp->value*pp->scale)

/* global vars in this file */
double	Pn, Qn;		/* Constants for initial Simplex            */
PARAM	**Simplex,	/* All      Parameters of apices of Simplex */
	***vSimplex;	/* Variable Parameters of apices of Simplex */

PARAM	*Xo, **vXo,	/* Center Point of Simplex                  */
	*Xr, **vXr,	/* Reflection Apex                          */
	*Xe, **vXe,	/* Expansion Apex                           */
	*Xc, **vXc;	/* Contraction Apex                         */
uint	Xh,       	/* Indices of highest &                     */
	Xs,   	   	/*            secondary &                   */
	Xl;    	   	/*            lowest      apex of Simplex   */


/* function prototypes */
#if defined(ANSI)
void  simplexInit(void);
void  sortSimplex(void);
int  oneIteration(void);
double  calcX(char type );
void  reduction(void);
void  paramCopy(PARAM **source, PARAM **dest);
void StoreBuffID(int ResultBuffID , int HistoryBuffID );

#else
void  simplexInit();
void  sortSimplex();
int  oneIteration();
double  calcX();
void  reduction();
void  paramCopy();
void StoreBuffID();

#endif

/*********************************************************************
**  Main routine of Simplex Method                                  **
*********************************************************************/
int main(int argc, char **argv )
{
  void  renpe_(char *filename), cleanUp(char **argv);
  int     ResultBuffID , HistoryBuffID;

  renpe_( argv[1] );
  ResultBuffID = atoi(argv[2]);
  HistoryBuffID = atoi(argv[3]);
  StoreBuffID(ResultBuffID , HistoryBuffID);

  /* start up message */
  printf("[ NPE ] SIMPLEX \n");
  fflush(stdout);

  simplexInit();
  for( Step=0; Step < MAXITERATION; Step++ ) {
    sortSimplex();
    display( Simplex[Xl], Err[Xl] );

    storeParamHist( Simplex[Xl], FALSE );

    if( term() ) break;

    oneIteration();
  }
  storeParamHist( Simplex[Xl], TRUE );
  cleanUp( argv );
  return( 0 );
}

/*********************************************************************
**  Subroutine of Initializition                                    **
**********************************************************************/
void  simplexInit(void)
{
  uint  i, j, k;

  /*
  **  reading information
  */
  readData();
  readParam();
  readTermCriterion();
  readSystemParam( &samp );
  readStoreFile();

  /*
  **  allocating variables
  */
  Simplex  = (PARAM  **)malloc2Dim( NumVarParam+1, NumAllParam, 'P');
  vSimplex = (PARAM ***)malloc2Dim( NumVarParam+1, NumVarParam, 'I');
  Xo  = (PARAM  *)malloc2Dim( 0, NumAllParam,   'P' );
  vXo = (PARAM **)malloc2Dim( 0, NumVarParam,   'I' );
  Xr  = (PARAM  *)malloc2Dim( 0, NumAllParam,   'P' );
  vXr = (PARAM **)malloc2Dim( 0, NumVarParam,   'I' );
  Xe  = (PARAM  *)malloc2Dim( 0, NumAllParam,   'P' );
  vXe = (PARAM **)malloc2Dim( 0, NumVarParam,   'I' );
  Xc  = (PARAM  *)malloc2Dim( 0, NumAllParam,   'P' );
  vXc = (PARAM **)malloc2Dim( 0, NumVarParam,   'I' );
  Err = (double *)malloc2Dim( 0, NumVarParam+1, 'D' );

  /*
  **  initializing variables
  */
  for( i=k=0; i<NumAllParam; i++ ) {
    Xo[i].scale = Xr[i].scale = 
    Xe[i].scale = Xc[i].scale = 1.0;

    Xo[i].flag  = Xr[i].flag  = 
    Xe[i].flag  = Xc[i].flag  = AllParam[i].flag;

    if( AllParam[i].flag == VAR ) {
      vXo[k  ] = &Xo[i];
      vXr[k  ] = &Xr[i];
      vXe[k  ] = &Xe[i];
      vXc[k++] = &Xc[i];
    } else {
      Xo[i].value = Xr[i].value = 
      Xe[i].value = Xc[i].value = AllParam[i].value;
    }
  }

  /*
  **  Constants for initial Simplex
  */
  Pn  = ( sqrt((double)(NumVarParam + 1)) + (double)(NumVarParam - 1) ) /
         ( NumVarParam * sqrt(2.0) );
  Qn  = ( sqrt((double)(NumVarParam + 1)) - 1 ) /
         ( NumVarParam * sqrt(2.0) );
  
  /*
  **  generating initial Simplex
  */
  for( i=0; i<NumVarParam+1; i++ ) {
    for( j=k=0; j<NumAllParam; j++ ) {
      Simplex[i][j].value = AllParam[j].value;
      Simplex[i][j].scale = AllParam[j].scale;
      Simplex[i][j].flag  = AllParam[j].flag;
      if( Simplex[i][j].flag == VAR ) {
        vSimplex[i][k++] = (PARAM *)& Simplex[i][j];
      } else {
        continue;
      }
      if( i != 0 ) {
        if( j == i-1 )
          Simplex[i][j].value += Pn;
        else
          Simplex[i][j].value += Qn;
      }
    }
    Err[i] = errorFunc( Simplex[i] );
  }
}

/*********************************************************************
**  Subroutine of Sorting Apecies of Simplex                        **
*********************************************************************/
void  sortSimplex(void)
{
  int  i, j;
  double  ErrXh, ErrXs, ErrXl;
  
  ErrXh = ErrXl = ErrXs = Err[0];
  Xh = Xl = Xs = 0;
  
  for( i=1; i<NumVarParam+1; i++ ) {
    if( Err[i] > ErrXh ) {           /*** finding Xh:Highest    ***/
      ErrXs = ErrXh;
      Xs    = Xh;
      ErrXh = Err[Xh=i];
    } else if( Err[i] > ErrXs ) {    /*** finding Xs:Secondary  ***/
      ErrXs = Err[Xs=i];
    }
    if( Err[i] < ErrXl ) {           /*** finding Xl:Lowest     ***/
      ErrXl = Err[Xl=i];
    }
  }

  paramCopy( vSimplex[Xl], VarParam );
  ErrorValue = Err[Xl];              /**  Terminate   **/

  for( i=0; i<NumVarParam; i++ )     /*** Initializing Xo    ***/
    vXo[i]->value = 0.0;

  for( i=0; i<= NumVarParam; i++ ){  /*** Calculating Xo     ***/
    if( i == Xh )
      continue;
    for( j=0; j<NumVarParam; j++ )
      vXo[j]->value += Value(vSimplex[i][j]);
  }
  for( i=0; i<NumVarParam; i++ )
    vXo[i]->value /= NumVarParam ;
}

/*********************************************************************
**  Subroutine of Main Algorithm of Simplex Method                  **
*********************************************************************/
int  oneIteration(void)
{
  double  ErrXr, ErrXe, ErrXc;

  ErrXr = calcX('r');

  if( ErrXr <= Err[Xs] ) {
    if( ErrXr < Err[Xl] ) {
      /*
      **  Reflection is better than Xl!
      */
      ErrXe = calcX('e');
      if( ErrXe < Err[Xl] ) {
        /*
        **  Expansion is best!!
        */
        paramCopy( vXe, vSimplex[Xh] );
        Err[Xh] = ErrXe;
        return(0);
      }
    }  
    paramCopy( vXr, vSimplex[Xh] );
    Err[Xh] = ErrXr;
    return(0);
  }

  if( ErrXr < Err[Xh] ) {
    /*
    **  Xr is better than Xh.
    */
    paramCopy( vXr, vSimplex[Xh] );
    Err[Xh] = ErrXr;
  }

  ErrXc = calcX('c');
  if( ErrXc < Err[Xh] ) {
    /*
    **  Contraction is better than Xh.
    */
    paramCopy( vXc, vSimplex[Xh] );
    Err[Xh] = ErrXc;
  } else {
    reduction();
  }

  return(0);
}

/*********************************************************************
**  Subroutine Calculating 'r'eflection,                            **
**                         'e'xpansion,                             **
**                         'c'ontraction                            **
*********************************************************************/
double  calcX(char type )
{
  int  i;

  switch( type ) {
    case 'r':
      for( i=0; i<NumVarParam; i++ ) {
        vXr[i]->value = (ALPHA + 1) * Value(vXo[i])
             - ALPHA * Value(vSimplex[Xh][i]);
      }
      return( errorFunc(Xr) );

    case 'e':
      for( i=0; i<NumVarParam; i++ ) {
        vXe[i]->value = (1 - GAMMA) * Value(vXo[i])
             + GAMMA * Value(vXr[i]);
      }
      return( errorFunc(Xe) );

    case 'c':
      for( i=0; i<NumVarParam; i++ ) {
         vXc[i]->value = (1 - BETA) * Value(vXo[i])
             + BETA * Value(vSimplex[Xh][i]);
      }
      return( errorFunc(Xc) );

    default:
      puts( "Unknown Apex" );
      exit( -1 );
  }
  return(0);
}


/*********************************************************************
**  Subrourine of reduction                                         **
**********************************************************************/
void  reduction(void)
{
  int  i, j;

  for( i=0; i<NumVarParam+1; i++ ) {
    for( j=0; j<NumVarParam; j++ ) {
      vSimplex[i][j]->value = 
        (vSimplex[i][j]->value + vSimplex[Xl][j]->value )/2.0;
    }
    Err[i] = errorFunc( Simplex[i] );
  }
}

/********************************************************
**  Subrourine copying parameters : 'source' -> 'dest' **
*********************************************************/
void     paramCopy(PARAM **source, PARAM **dest )
{
  int     i;

  for( i=0; i<NumVarParam; i++ ) {
    dest[i]->value = source[i]->value;
    dest[i]->scale = source[i]->scale;
  }
}

void StoreBuffID(int ResultBuffID , int HistoryBuffID )
{
  Result *wRe;        /* work */
  History *wHis;      /* work */


  for( wRe = ResultHead ; wRe != NULL ; wRe = wRe->next )
      wRe->buffnum = ResultBuffID;

  for( wHis = HistoryHead ; wHis != NULL ; wHis = wHis->next )
      wHis->buffnum = HistoryBuffID;
}
