#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "defs.h"
#include "prototype.h"
#include "y.tab.h"

#define NEXTPOINT(p)		((p + 1 == HSIZE) ? 0 : p + 1 )
#define PREVPOINT(p, siz)	((p - 1 <  0) ? siz - 1 : p - 1 )
#define LASTPOINT		((hc == 0) ? HSIZE - 1 : hc - 1)
#define CRNTPOINT		(hc)
#define CP			CRNTPOINT

static  int     HSIZE = 50, OLDSIZE = 0;
static  History **ring = NULL;	/* private history buffer */
static  int     hc = 0;		/* private history counter */

extern int      lineno; /* declared in infile.c */


#if defined(DEBUG) && defined(DEBUG_MORE)
static void
debug()
{
  int             i;

  puts("---------------------------------------");
  for (i = 0; i < HSIZE; i++) {
    if (ring[i] != NULL) {
      printf("[%d:%s]\n", i, ring[i]->buf);
    }
  }
  puts("---------------------------------------");
}
#endif


static int
h_size(hist, siz, cp)
     History  **hist;
     int        siz, cp;
{
  int  i = cp, n = 0;

  do {
    i = PREVPOINT(i, siz);
    if (hist[i] == NULL)
      break;
    ++n;
  } while (i != cp);
  return n;
}

static int
h_copy(to, from, cp)
     History  **to, **from;
     int        cp;
{
  int   i = cp;
  int   newcp, j = Min(OLDSIZE, HSIZE);
  int   hsize = h_size(from, OLDSIZE, cp);

  j = Min(j, hsize);
  newcp = (j >= HSIZE) ? 0 : j;

  do {
    i = PREVPOINT(i, OLDSIZE);
    if (from[i] == NULL)
      break;
    to[--j] = from[i];
  } while (i != cp && j != 0);

  return newcp;
}


static int
h_free(hist, cp)
     History  **hist;
     int        cp;
{
  int  i = cp;
  int  j = Min(OLDSIZE, HSIZE);
  int  hsize = h_size(hist, OLDSIZE, cp);

  j = Min(j, hsize);

  if (HSIZE >= OLDSIZE)
    return 0;

  do {
    i = PREVPOINT(i, OLDSIZE);
    if (--j < 0) {
      if (hist[i] != NULL) {
	efree(hist[i]->buf);
	efree((char*)(hist[i]));
      }
    }
  } while (i != cp);
  efree((char*)hist);
  return 0;
}


void
init_history()
{
  History       **tmp;
  int             newcp;

  tmp = (History **) emalloc(HSIZE * sizeof(History *));
  Bzero(tmp, HSIZE * sizeof(History *));
  if (ring != NULL) {
    newcp = h_copy(tmp, ring, CP);
    h_free(ring, CP);
    CP = newcp;
  }
  ring = tmp;

  OLDSIZE = HSIZE;
}

static char   *
skipspaces(s, p)
     register char  *s, *p;
{
  char    backchar = '\0', inquote = FALSE;
  while (*s != '\0') {
    if (backchar != '\\' && *s == '"')
      inquote = !inquote;
    if (!(*s == ' ' && backchar == ' ') || inquote) {
      *p++ = *s;
    }
    backchar = *s++;
  }
  *p = '\0';
  return p;
}

void
h_append(str)
     char  *str;
{
  char     *s = str;
  char     *tmp = emalloc(strlen(str) + 2);
  char     *p = tmp;
  int      hour, minute, second;

  if (ring[CP] != NULL) {
    efree(ring[CP]->buf);
    efree((char*)(ring[CP]));
  }
  skipspaces(s, p);

  ring[CP] = (History *) emalloc(sizeof(History));
  ring[CP]->buf = (char *) emalloc(strlen(tmp) + 1);
  strcpy(ring[CP]->buf, tmp);
  efree(tmp);

  get_time(&hour, &minute, &second);
  sprintf(ring[CP]->time, "%02d:%02d:%02d", hour, minute, second);

  ring[CP]->lineno = lineno;

  /* hc indicate next point */
  CP = NEXTPOINT(CP);

#if defined(DEBUG) && defined(DEBUG_MORE)
  debug();
#endif
}


static int
found(dbase, key)
     History  *dbase;
     char     *key;
{
  register char  *p;;

  if (dbase == NULL || (p = dbase->buf) == NULL)
    return EOF;

  while (isspace(*p))
    p++;

  return !strncmp(p, key, strlen(key));
}


char    *
h_search(key)
     char  *key;
{
  register int    i = CP;

  if (*key == '!')
    return ring[PREVPOINT(CP, HSIZE)]->buf;

  do {
    i = PREVPOINT(i, HSIZE);
    switch (found(ring[i], key)) {
    case EOF:			/* end of history	 */
      return NULL;
    case FALSE:			/* zero : not found	 */
      break;
    case TRUE:			/* found		 */
    default:			/* not use : not zero	 */
      return ring[i]->buf;
    }
  } while (i != CP);
  return (char*)NULL;
}


History  *
h_get(n)
     int  n;	/* CP - n : relative address */
{
  register int    i, addr = CP;

  if (n < 0)
    return NULL;

  for (i = 0; i < n; i++)
    addr = PREVPOINT(addr, HSIZE);

  if (addr == CP)
    return NULL;
  return ring[addr];
}

#define STARTPOINT ((ring[CP] == NULL) ? 0 : CP)

static void
h_print()
{
  int             sp = STARTPOINT, lp = LASTPOINT;
  register int    i = sp;

  do {
    if (ring[i] == NULL)
      continue;
    if (istty(1)) {		/* output to tty */
      printf("%c%3d:[%s]  %s", ((i == lp) ? '>' : ' '),
	     ring[i]->lineno, ring[i]->time, ring[i]->buf);
    } else {			/* output to pipe */
      printf("%s", ring[i]->buf);
    }
  } while ((i = NEXTPOINT(i)) != sp);
}


int
read_history(h_file)
     char           *h_file;
{
  register int    n = 0;
  FILE           *fp;
  char            oneline[ONELINE];

  if ((fp = fopen(h_file, "r")) == NULL)
    return n;

  while (fgets(oneline, ONELINE, fp) != NULL) {
    h_append(oneline);
    n++;
    lineno++;
  }

  fclose(fp);
  return n;
}


int
write_history(h_file)
     char           *h_file;
{
  int             sp = STARTPOINT, i = sp;
  register int    n = 0;
  FILE           *fp;

  fp = fopen(h_file, "w");
  if (fp == NULL)
    return n;
  do {
    if (ring[i] != NULL) {
      fprintf(fp, "%s", ring[i]->buf);
      n++;
    }
  } while ((i = NEXTPOINT(i)) != sp);
  fclose(fp);
  return n;
}


double
history_size()
{
  return (double) HSIZE;
}


static char   *
get_this_line(n)
     int             n;
{
  register int    i;
  register History *hist;

  for (i = 1; (hist = h_get(i)) != NULL; i++) {
    if (hist->lineno == n) {
      return hist->buf;
    }
  }
  return NULL;
}


static Object *
pickup_history(obj)
     Object         *obj;
{
  register int    i;
  int             lines, dim, idx[MAX_INDEX];
  int             id = (*(int *) obj->val);
  Buffer         *buf = ReadBuffer(id, &dim, idx);
  int             siz = IndexSize(dim, idx);
  char          **argv = (char **) emalloc(siz * sizeof(char *));
  Object         *acc;

  if (buf == NULL) {
    fprintf(stderr, "failed to evaluate series");
    return NULL;
  }
  for (lines = 0, i = 0; i < siz; i++) {
    argv[lines] = get_this_line((int) buf[i]);
    if (argv[lines] != NULL)
      lines++;
  }
  FreeBuffer(buf);
  acc = newacc(argv, 1, &lines, &String);
  efree((char*)argv);
  return acc;
}


Object *
history(nargs, objs)
     int             nargs;
     Object         *objs[];
{
  int             type, n = 0;
  double          d;

  if (nargs == 0) {
    h_print();
    return (Object *) NULL;
  }
  if (nargs != 1)
    execerror("history,", "mismatch: number of arguments");

  type = TypeofOBJ(objs[0]);
  /*
   * if (type != SCALAR_T && type != STRING_T) execerror("history(), ",
   * "illegal object type"); 
   */

  switch (type) {
  case STRING_T:{
    char           *s = ((char **) objs[0]->val)[0];
    if (*s == '\0')
      n = 0;
    else if (equal(s, "?"))
      n = -1;
    else
      n = atoi(s);
    break;
  }
  case SCALAR_T:
    n = (int) (*(double *) objs[0]->val);
    break;
  case SERIES_T or SNAPSHOT_T:
    return pickup_history(objs[0]);
    break;
  default:
    break;
  }

  if (n < 1) {
    d = (double) history_size();
    return newacc(&d, 0, NULL, &Scalar);
  } else {
    HSIZE = n;
    init_history();
    return (Object *) NULL;
  }
}
