/**********************************************************************
	code.c :  Part of Stack Machine Code

			Coded by Shigeru Hitomi  May. 4, 1992
***********************************************************************/
#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include "defs.h"
#include "prototype.h"
#include "y.tab.h"

#define EXTERN			/* entity */
#include "code.h"

#if 0
#define DEBUGF(a)	fprintf a
#else
#define DEBUGF(a)
#endif

int      code_offset;

extern int      freeCell;
extern BOOLEAN  in_module, indef;
extern BOOLEAN  doprint;
extern BOOLEAN  inpipe, open_pipe;
extern BOOLEAN  in_eval; /* in code.eval.c */


#if defined(OUT_STACK) || defined(DEBUG)
extern int PID; /* for stack pointer check */
#endif

/*
 * extern FILE    *fin;
 */

/**********************************************************************
	STACK MACHINE   CODE  : Stack Operation Code
***********************************************************************/


void
initcode()
{                           /* initialize for code generation */
#if 0
  if (prog == 0)            /* one time only */
    progalloc();            /* allocation for program area */
#endif
  if (open_pipe == TRUE)
    close_pipe();

  /* reset flags */
  doprint = TRUE;
  inpipe = FALSE;
  returning = breaking = continuing = 0;

  stackp  = stackbase;     /* reset stack pointer         */
  prog    = mainprog;      /* reset program start address */
  progp   = mainprogbase;  /* reset program counter       */
  PROGSIZ = MAINPROGSIZ;   /* reset code area size        */
  fp      = framebase;     /* reset frame for function    */
  code_offset = 0;         /* code offset */

  if ( freeCell > START_COLLECT &&
#if 0
      !( in_eval || indef )
#else
      !( indef )
#endif
       ) {
#if DEBUG
  fprintf(stderr, "\nauto garbage collection ... ");
#endif

    FreeSymbols();         /* start garbage collection    */

#if DEBUG
  fprintf(stderr, "done\n");
#endif
  }
}


void
push(d)				/* push d onto stack */
     Datum  d;
{
  if (stackp >= &stack[NSTACK])
    execerror("stack too deep", (char *) 0);
  *stackp++ = d;
#if defined(OUT_STACK) || defined(DEBUG)
  if ( PID != 0 )
    fprintf(stderr,"(push: %d [%lx])",
	    (int)(stackp - stack), (unsigned long)d.sym);
  else
    fprintf(stderr,"(c-push: %d [%lx])",
	    (int)(stackp - stack), (unsigned long)d.sym);
  fflush(stderr);  
#endif
#ifdef OUT_STACK
  out_stack(NULL);
#endif
}


Datum 
pop()
{				/* pop and return top elem from stack */
  if (stackp == stack)
    execerror("stack underflow", (char *) 0);
#if defined(OUT_STACK) || defined(DEBUG)
  {
    Datum           d = *--stackp;

    if ( PID != 0 )
      fprintf(stderr,"(pop: %d [%lx])",
	      (int)(stackp - stack), (unsigned long) d.sym);
    else
      fprintf(stderr,"(c-pop: %d [%lx])",
	      (int)(stackp - stack), (unsigned long) d.sym);
    fflush(stderr);
#ifdef OUT_STACK
    out_stack(NULL);
#endif
    return d;
  }
#else
  return *--stackp;
#endif
}


void
push_obj(d)			/* push d onto stack */
     Datum   d;
{
  if (stackp >= &stack[NSTACK])
    execerror("stack too deep", (char *) 0);
  if (d.obj != (Object *) NULL)
    d.obj->link++;
  *stackp++ = d;
#if defined(OUT_STACK) || defined(DEBUG)
  if ( PID != 0 )
    fprintf(stderr,"(push_obj: %d [%lx])",
	    (int)(stackp - stack), (unsigned long)d.obj);
  else
    fprintf(stderr,"(c-push_obj: %d [%lx])",
	    (int)(stackp - stack), (unsigned long)d.obj);
  fflush(stderr);
#endif
#ifdef OUT_STACK
  out_stack(NULL);
#endif
}

Datum
pop_obj()
{				/* pop and return top elem from stack */
  Datum           d;
  if (stackp == stack)
    execerror("stack underflow", (char *) 0);
  d = *--stackp;
  if (d.obj != (Object *) NULL)
    d.obj->link--;
#if defined(OUT_STACK) || defined(DEBUG)
  if ( PID != 0 )
    fprintf(stderr,"(pop_obj: %d [%lx])",
	    (int)(stackp - stack), (unsigned long)d.obj);
  else
    fprintf(stderr,"(c-pop_obj: %d [%lx])",
	    (int)(stackp - stack), (unsigned long)d.obj);
  fflush(stderr);
#endif
#ifdef OUT_STACK
  out_stack(NULL);
#endif
  return d;
}

void
POP()
{				/* used in parse.y ... not return Datum */
  Datum           d;
  if (stackp == stack)
    execerror("stack underflow", (char *) 0);
  d = *--stackp;
  if (d.obj != (Object *) NULL)
    d.obj->link--;

#if defined(OUT_STACK) || defined(DEBUG)
  if ( PID != 0 )
    fprintf(stderr,"(POP: %d [%lx])",
	    (int)(stackp - stack), (unsigned long)d.obj);
  else
    fprintf(stderr,"(c-POP: %d [%lx])",
	    (int)(stackp - stack), (unsigned long)d.obj);
  fprintf(stderr,"\n");
#endif
}

void
constpush()
{				/* push constant onto stack */
  Datum           d;
  Symbol         *sym = (Symbol *) * pc++;

  /* address copy , entity -> symbolic table */
  d.obj = getobj(sym);

#if defined(OUT_STACK) || defined(DEBUG)
  if ( PID != 0 )
    fprintf(stderr,"(constpush");
  else
    fprintf(stderr,"(c-constpush: %d", (int)(stackp - stack));
#endif

  push_obj(d);

#if defined(OUT_STACK) || defined(DEBUG)
  fprintf(stderr,")"), fflush(stderr);
#endif
}


void
varpush()
{
  Datum           d;

#if defined(OUT_STACK) || defined(DEBUG)
  if ( PID != 0 )
    fprintf(stderr,"(varpush");
  else
    fprintf(stderr,"(c-varpush");
#endif
  d.sym = (Symbol *) (*pc++);
  push(d);

#if defined(OUT_STACK) || defined(DEBUG)
  fprintf(stderr,")"), fflush(stderr);
#endif
}

static void
push_quote(sym)
     Symbol         *sym;
{
  Datum           d;
  char           *name = sym->name;

  setobj(sym, (*String.method->new) ());
  d.obj = getobj(sym);

#if defined(OUT_STACK) || defined(DEBUG)
  if ( PID != 0 )
    fprintf(stderr,"(push_quote");
  else
    fprintf(stderr,"(c-push_quote");
  if ( name != NULL )
    fprintf(stderr, " [name: %s]", name ), fflush(stderr);
#endif

  (*String.method->setvalue) (d.obj, &name, 0, NULL);
  push_obj(d);

#if defined(OUT_STACK) || defined(DEBUG)
  fprintf(stderr,")"), fflush(stderr);
#endif

}

void
eval()
{				/* evaluate variable on stack */
  Datum           d;

#ifdef DEBUG
  fprintf(stderr,"eval("), fflush(stderr);
#endif

  d = pop();		/* pop variable name (in symbolic table) */

  if (in_module && d.sym->type == UNDEF) {	/* for external command */
    push_quote(d.sym);
    return;
  }

  if (d.sym->type == UNDEF)
    execerror("undefined variable", d.sym->name);

  if (d.sym->onstack == -1 && d.sym->obj == NULL)
    execerror("NULL object", d.sym->name);

  d.obj = getobj(d.sym);
  push_obj(d);

#ifdef DEBUG
  fprintf(stderr,")\n"), fflush(stderr);
#endif
}


void
constant()
{				/* assign top value to next value */
  Datum           d1, d2;
  Object         *x, *obj;

  d1 = pop();		/* pop variable from stack */
  d2 = pop_obj();		/* pop object from stack */
  if (d1.sym->type != CONSTANT && d1.sym->type != UNDEF)
    execerror("assingment to non-variable", d1.sym->name);
  x = getobj(d1.sym);
  obj = (Object *) (*d2.obj->method->copy) (x, d2.obj);
  setobj(d1.sym, obj);
  d1.sym->type = CONSTANT;
}

void
assign()
{				/* assign top value to next value */
  Datum           d1, d2;
  Object         *x, *y, *obj;

#ifdef DEBUG
  fprintf(stderr,"assign("), fflush(stderr);
#endif   

  d1 = pop();		/* pop variable from stack */
  d2 = pop();		/* pop object from stack */
  if (d1.sym->type != VAR && d1.sym->type != UNDEF)
    execerror("assingment to non-variable", d1.sym->name);

  x = getobj(d1.sym);
  if ((y = d2.obj) == NULL)
    execerror("null object assignment", NULL);

  if (x != NULL && TypeofOBJ(x) != TypeofOBJ(y)) {
    char            mesg[100];
    sprintf(mesg, "illegal combination of %s and right, (op =)",
	    d1.sym->name);
    warning("WARNING ...", mesg);
  }
  obj = (Object *) (*y->method->copy) (x, y);
  setobj(d1.sym, obj);
  d1.sym->type = VAR;
  push(d2);

#ifdef DEBUG
  fprintf(stderr,")\n");
#endif   
}

void
selfasgn()
{				/* assign top value to next value */
  Datum           d1, d2, result;
  static Object  *argm[3] = {0, 0, 0};
  Object         *(*opcode) (), *obj;

#ifdef DEBUG
  fprintf(stderr,"selfasgn("), fflush(stderr);
#endif   

  d1 = pop();		/* pop variable from stack */
  d2 = pop();		/* pop object from stack */
  if (d1.sym->type != VAR)
    execerror("assingment to non-variable", d1.sym->name);
  if (d2.obj == NULL)
    execerror("null object assignment", NULL);
  
  obj = getobj(d1.sym);
  argm[0] = obj;
  argm[1] = d2.obj;
  opcode = obj->method->opcode;
  
  /**********************************************************
    sp		:  message
    sp->obj->val	: (function address)
    sp->name     : (function name)
    **********************************************************/
  result.obj = (*opcode) (pc++, "assign operation", argm);
  if (TypeofOBJ(result.obj) != TypeofOBJ(obj))
    execerror(d1.sym->name,
	      ", illegal combination of left and right, (op = )");

  (Object *)(*obj->method->copy) (obj, result.obj);

  /*   push(result); */
  push(d2); /* appended by take  1993.09.11 */

#ifdef DEBUG
  fprintf(stderr,")\n");
#endif   
}


static Object *
increment(sym)
     Symbol         *sym;
{
  static Object  *obj, *argm[3] = {0, 0, 0};
  static Symbol  *one = 0;
  if (one == 0)
    one = (Symbol *) lookup("+1");

  if (sym->type != VAR)
    execerror("increment/decrement to non-variable", sym->name);
  if (sym->onstack == -1 && sym->obj == NULL)
    execerror("illegal object (null)", sym->name);

  obj = getobj(sym);
  if (TypeofOBJ(obj) == STRING_T)
    execerror("illegal object type", sym->name);
  argm[0] = obj;
  argm[1] = one->obj;
  return (*obj->method->opcode) (pc++, "inc/dec", argm);
}

void
prefix()
{
  Object         *obj;
  Datum           d;
  d = pop();

  obj = getobj(d.sym);
  (*obj->method->copy) (obj, increment(d.sym));
  d.obj = obj;
  push_obj(d);
}

void
postfix()
{
  Object         *obj;
  Datum           dacc, d;
  d = pop();
  if (d.sym->type == UNDEF)
    execerror("undefined variable", d.sym->name);
  obj = getobj(d.sym);
  dacc.obj = cpyacc(obj);
  push_obj(dacc);
  (*obj->method->copy) (obj, increment(d.sym));
}


Inst    *
code(f)
     Inst f;
{
  if (progp >= prog + PROGSIZ) {
    int             old_code_offset = code_offset;
    RETSIGTYPE      (*istat) () = (RETSIGTYPE (*) ()) signal(SIGINT, SIG_IGN);

    code_offset = relocate();
    DEBUGF((stderr, "code_offset  = %d\n", code_offset));

    if (code_offset == 0)
      code_offset = old_code_offset + ResizeCodeArea();
    signal(SIGINT,istat);
  }
#ifdef OUT_STACK_MACHINE_CODE
  out_stack_code(progp, f);
#endif
  *progp = f;
  return code_offset + progp++;
}

void
undef_var()
{
  Symbol  *sp = (Symbol *) * pc++;
  freeCell -= undef(sp->name);
}

void
isdef_var()
{
  Datum       d;
  Symbol    *sp = (Symbol *) * pc++;
  double   flag = 0;

  switch ( sp->type ) {
  case VAR:
  case CONSTANT:
    flag = 1.0;
    break;
  case UNDEF:
    flag = 0;
    break;
  default:
    flag = -1.0;
  }
  d.obj = newacc(&flag, 0, NULL, &Scalar);
  push_obj(d);
}


void
execute(p)			/* run the machine */
Inst           *p;
{
#ifdef  SHOW_STACK_MACHINE_CODE
  show_code(p);
#endif
  pc = p;
  while (*pc != STOP && !returning && !breaking && !continuing) {
#ifdef DEBUG
    fprintf(stderr,"/%lx(%s)/", (unsigned long)pc, code_name(*pc));
    fflush(stderr);
#endif
    (*(*pc++)) ();
  }
}
/**********************************************************************
	END OF STACK MACHINE CODE
***********************************************************************/
