/**********************************************************************
  code.func.c : Part of Stack Machine Code
  (function & Procedure Parts)
  
  Coded by Shigeru Hitomi  Oct. 3, 1992
  Last Modified            Mar. 8, 1993
  ***********************************************************************/

/**********************************************************************
  INCLUDES
  ***********************************************************************/
#include <stdio.h>
#include <signal.h>
#include "defs.h"
#include "prototype.h"
#include "code.h"
#include "y.tab.h"

#ifdef DEBUG
#define DEBUGF(a) fprintf a
FILE	*debug_fp = 0;
#else
#define DEBUGF(a)
#endif


/**********************************************************************
 * INCLUDES
 ***********************************************************************/
BOOLEAN           indef;       /* in the definition */
static Symbol    *savesp;      /* save pointer of symbolic table */
static char      *funcname;    /* function/procedure name for Error-Message */

extern int        code_offset; /* offset of sub-code pointer */

static RETSIGTYPE (* istat)();
     
/**********************************************************************
 * PROTOTYPES
 ***********************************************************************/
static int   regist_args      _ANSI_ARGS_((Symbol *arg, Symbol *basesp));
static int   regist_autos     _ANSI_ARGS_((Symbol *entry, Symbol *arg,
					   int narg));
static void  autovar_alloc    _ANSI_ARGS_((SubProg *sub));
static void  destroy_sub      _ANSI_ARGS_((Symbol *proc));
     

/**********************************************************************
 * FUNCTIONS
 ***********************************************************************/
void
beginSub(proc)
     Symbol *proc;
{
  istat   = (RETSIGTYPE (*)())signal(SIGINT, SIG_IGN);
  indef   = TRUE;			/* start of definition */
  savesp  = GetSymbolTable();
  prog    = subprog;
  progp   = subprogbase;
  PROGSIZ = SUBPROGSIZ;

  /* for recursive function call, by take */
  switch (proc->type) {
  case UNDEF:
    proc->type = FUNCTION;
    break;
  case FUNCTION:
  case PROCEDURE:
    /* do nothing */
    break;
  default:
    endSub();
    execerror("specified name is already defined as object:", proc->name );
  }
}

#ifdef DEBUG
void
EndSub(sub)			/* Debugging Tool : Display of Sub-program
				 * Code */
     SubProg        *sub;
{
#if defined(OUT_STACK_MACHINE_CODE) || defined(SHOW_STACK_MACHINE_CODE)
  register Relative p;		/* relative address */
  register Absolute ap;		/* absolute address */
  char           *name;
#endif

  if (debug_fp == 0)
    debug_fp = fopen("debug.code", "w");
  DEBUGF((debug_fp, "--------------------------------------\n"));
  DEBUGF((debug_fp, "funcname = [%s] : ", funcname));
  DEBUGF((debug_fp, "subprog = %lx, progp = %lx\n",
	  (unsigned long)subprog, (unsigned long)progp));
  DEBUGF((debug_fp, "--------------------------------------\n"));
#if defined(OUT_STACK_MACHINE_CODE) || defined(SHOW_STACK_MACHINE_CODE)
  for (p = sub->begin_code; p != sub->end_code; p++) {
    ap = AbsoluteAddr(p);
    name = (char *) code_name(*ap);
    if (!equal(name, "?"))
      DEBUGF((debug_fp, "%lx:\t%s\n", (unsigned long)ap, name));
    else
      DEBUGF((debug_fp, "%lx:\t%c[7m%lx%c[0m\n",
	      (unsigned long)ap, ESC, (unsigned long)*ap, ESC));
  }
#endif
  DEBUGF((debug_fp, "--------------------------------------\n"));
  fflush(debug_fp);
  /*
   * getchar();
   */
}
#endif

void
endSub()
{
  prog = mainprog;
  progp = mainprogbase;
  PROGSIZ = MAINPROGSIZ;
  PutSymbolTable(savesp);
  indef = FALSE;		/* end of definition */
  code_offset = 0;		/* offset of sub-code program counter */
  signal(SIGINT, istat);
}

void
define(proc, type, arg)		/* put func/proc in symbol table */
     Symbol         *proc, *arg;
     unsigned type;
{
  SubProg        *sub;		/* new info. for subprogram */
  Symbol         *entry = GetSymbolTable(); /* auto variable entry */
  int             narg;	        /* number of arguments      */
  int             nauto;	/* number of auto variables */
  
  
  if ( proc->type == FUNCTION || proc->type == PROCEDURE
      || proc->type == UNDEF ) {
    proc->type = type;
  } else {
    endSub();
    execerror("specified name is already defined as object:", proc->name );
    return;
  }
  
  sub = (SubProg *) emalloc(sizeof(SubProg));
  
  /* register the stack-number for arguments */
  narg = regist_args(arg, savesp);
  /* register the stack-number for auto variables */
  nauto = regist_autos(entry, arg, narg);

  funcname = proc->name;
  sub->begin_symbol = entry;	/* entry of symbolic table  */
  sub->end_symbol = savesp;	/* end of symbolic table */
  sub->narg = narg;		/* number of arguments      */
  sub->nauto = nauto;		/* number of auto variables */
  
  /* beginning of subcode */
  sub->begin_code = RelativeAddr(subprogbase);
  
  /* end of subcode  */
  sub->end_code = RelativeAddr(progp);
#ifdef DEBUG
  DEBUGF((stderr, "\nbegin_code = %lx\n", (unsigned long)subprogbase));
  DEBUGF((stderr, "end_code = %lx\n", (unsigned long)progp));
  DEBUGF((stderr, "code_length = %d\n", progp - subprogbase));
#endif
  
  if (proc->obj != NULL)
    destroy_sub(proc);		/* destroy subprogram       */
  proc->obj = (*Builtin.method->new) (); /* new object for subprog.  */
  proc->obj->val = (Inst *) sub; /* subcode information      */
  subprogbase = progp;		/* next code starts here    */
#ifdef DEBUG
  EndSub(sub);			/* end of subprogram [Debugging Tool] */
#endif
  endSub();			/* end of subprogram        */
}


void
call()
{				/* call a function */
  SubProg        *sub;		/* for subcode information */
  Symbol         *sp = (Symbol *) pc[0]; /* symbol table entry for
					  * function */
  if (sp->obj == NULL)
    execerror("incomplete definition: function/procedure", funcname);

  if (fp++ >= &frame[NFRAME - 1]) /* increment the Frame pointer */
    execerror(sp->name, "call nested too deeply");
  funcname = sp->name;		/* set a function name for error message */
  fp->sp = sp;			/* the function entry in symbolic table */
  fp->nargs = *(int *)&pc[1];	/* number of arguments */
  fp->retpc = pc + 2;		/* next program address */
  fp->argn = stackp - 1;	/* last argument */
  sub = (SubProg *) sp->obj->val; /* get subcode information */
  autovar_alloc(sub);		/* stack area allocation for auto variable */

  execute(AbsoluteAddr(sub->begin_code)); /* start of code */

  returning = 0;		/* reset returning flag */
}


void
ret()
{				/* common return from func ro proc */
#ifdef DEBUG
  fprintf(stderr,"ret("), fflush(stderr);
#endif
  stackp = free_autovars();	/* pop autovars */
  pc = (Inst *) fp->retpc;	/* restore the program counter */
  --fp;				/* decrement Frame pointer     */
  returning = 1;		/* exist return object         */
#ifdef DEBUG
  fprintf(stderr,")\n");
#endif
}


void
funcret()
{				/* return from a function */
  Datum   d;

#ifdef DEBUG
  fprintf(stderr,"funcret("), fflush(stderr);
#endif

  if (fp->sp->type == PROCEDURE) /* Procedure not returns value */
    execerror(funcname, "(proc) returns value");
  d = pop_obj();		/* preserve function return value */
  if (d.obj != NULL)
    d.obj = cpyacc(d.obj);	/* return to copy object          */
  ret();			/* set program counter at next program */
  push_obj(d);			/* push a return object */

#ifdef DEBUG
  fprintf(stderr,")\n");
#endif
}


void
procret()
{				/* return from a procedure */
#ifdef DEBUG
  fprintf(stderr,"procret("), fflush(stderr);
#endif
  if (fp->sp->type == FUNCTION)	/* Function always returns value */
    execerror(funcname, "(func) returns no value");
  ret();
#ifdef DEBUG
  fprintf(stderr,")\n");
#endif
}

/**********************************************************************
  Object Management
  ***********************************************************************/

static int
regist_args(arg, basesp)
     Symbol  *arg, *basesp;
{
  register Symbol *sp;
  int              n, narg = 0;

  /* count up number of arguments */
  for (sp = arg; sp != basesp; sp = sp->next)
    narg++;

  n = narg;
  /* register the stack number for arguments */
  for (sp = arg; sp != basesp; sp = sp->next) {
    sp->onstack = --n;		/* stack number */
    sp->type = VAR;
  }
  return narg;
}

static int
regist_autos(entry, arg, narg)
     Symbol  *entry, *arg;
     int      narg;
{
  register Symbol *sp;
  int             nauto = 0;	/* number of auto variables */
  /* count up number of auto variables  &  register the stack number */
  for (sp = entry; sp != arg; sp = sp->next) {
    if (!equal(sp->name, " ")) {
      sp->onstack = nauto + narg;
      nauto++;
    }
  }
  return nauto;
}


static void
autovar_alloc(sub)
     SubProg   *sub;
{
  register int    i;
  Datum           d;

#ifdef DEBUG
  fprintf(stderr,"autovar_alloc("), fflush(stderr);
#endif

  /* check number of arguments */
  if (fp->nargs != sub->narg)
    execerror("arguments mismatch", 0);

  /* allocation of stack area for auto variables */
  d.obj = NULL;
  for (i = 0; i < sub->nauto; i++)
    push(d);

#ifdef DEBUG
  fprintf(stderr,")\n");
#endif
}


Datum   *
free_autovars()
{
  /* sp : beginning of auto-vars in stack */
  register Datum *sp;		/* stack pointer */
  Datum          *frame_stack_begin = fp->argn - fp->nargs + 1;
  SubProg        *sub = (SubProg *) fp->sp->obj->val;
  int             nauto = sub->nauto;
  
#ifdef DEBUG
  fprintf(stderr,"free_autovars("), fflush(stderr);
#endif
  /* free stack area for arguments */
  for (sp = frame_stack_begin; sp != stackp; sp++) {
    register Datum  d;
    d = *sp;
    if (d.obj == NULL)
      continue;
    if (d.obj->link != 0)
      d.obj->link--;		/* decrement link counter */
    /* auto-vars only (arguments not destroy) */
    if (fp->argn < sp && sp <= fp->argn + nauto) {
#ifdef DEBUG
      fprintf(stderr,"[%lx]", (unsigned long)d.obj), fflush(stderr);
#endif
      (*d.obj->method->destroy) (d.obj);
    }
  }
#ifdef DEBUG
  fprintf(stderr,")"), fflush(stderr);
#endif
  return frame_stack_begin;
}


void
external()
{
  Symbol *d, *s;
  Object *obj;

  d   = (Symbol *)*pc++;
  obj = getobj(d);
  s = lookup(d->name);

  if ( s != 0 && d->obj == NULL && s->obj != NULL ) {
    obj = s->obj;
    obj->link += 2;
    setobj(d,obj);
    d->type = VAR;
  }
#ifdef DEBUG
  fprintf(stderr,"external(%s,%lx,%d", d->name, (unsigned long) d, d->onstack);
  if ( obj != NULL )
    fprintf(stderr,",%lx,%d",(unsigned long) obj, obj->link);
  fprintf(stderr,")\n");
#endif
}


static Datum  *
stack_address(n)	/* return pointer to argument or auto variable */
     int     n;   	/* rank of stack */
{
  return &(fp->argn[n - fp->nargs + 1]); /* return address of stack */
}

void
setobj(sp, obj)
     Symbol         *sp;
     Object         *obj;
{
  int             n = sp->onstack;
  if (n != -1)			/* onto the stack */
    stack_address(n)->obj = obj;
  else				/* link to symbolic table */
    sp->obj = obj;
}


Object  *
getobj(sp)			/* return pointer to object */
     Symbol         *sp;
{
  int             n = sp->onstack;
  return (n != -1) ? stack_address(n)->obj : sp->obj;
}

/**********************************************************************
  Destory Subprogram
  ***********************************************************************/

static void
destroy_sub(proc)
     Symbol         *proc;
{
  extern   int    freeCell;
  register int    nfree = 0;
  register Symbol *sp, *this, *endsp;
  SubProg        *sub = (SubProg *) proc->obj->val;
  /*
   * int             nfree = sub->narg + sub->nauto;
   */

  /* free private symbols */
  sp = sub->begin_symbol;	/* begin of local symbolic table */
  endsp = sub->end_symbol;	/* end of local symbolic table */
  while (sp != endsp) {
    this = sp;
    sp = sp->next;		/* point to next symbol */

    if (this->obj != NULL) {
      (*this->obj->method->destroy) (this->obj);
    }
    efree(this->name);
    efree((char*)this);
    nfree++;
  }
  freeCell -= nfree;
#ifdef DEBUG
  DEBUGF((stderr, "free %d symbols.\n", nfree));
#endif
  /* destroy a symbolic table of proc/func */
  (*proc->obj->method->destroy) (proc->obj);
}

/**********************************************************************
	End of Stack Machine Code
***********************************************************************/
