/**********************************************************************
  code.memroy.c : Part of Stack Machine Code
  (function & Procedure Parts)
  
  Coded by Shigeru Hitomi  Oct. 3, 1992
  ***********************************************************************/

#include <stdio.h>
#include <string.h>
#include "defs.h"
#include "prototype.h"
#include "code.h"
#include "y.tab.h"

#ifdef DEBUG 	
#define DEBUGF(a) fprintf a
#else
#define DEBUGF(a)
#endif

#ifndef RELOCATION_SIZE
#define RELOCATION_SIZE 1024
#endif

typedef struct proglist {	/* regist sub-programs */
  SubProg        *sub;		/* beginning of sub-program pointer */
  struct proglist *next;	/* next pointer */
}               proglist;

extern BOOLEAN  indef;		/* in the definition, in code.func.c */

/**********************************************************************
  Memory	Management	( Memory Allcation )
  ***********************************************************************/

void
init_stackmachine()
{
  /* memory allocation for main program area */
  mainprog = (Inst *) emalloc(MAINPROGSIZ * sizeof(Inst));
  Bzero(mainprog, MAINPROGSIZ * sizeof(Inst));
  mainprogbase = mainprog;

  /* memory allocation for sub program area */
  subprog = (Inst *) emalloc(SUBPROGSIZ * sizeof(Inst));
  Bzero(subprog, SUBPROGSIZ * sizeof(Inst));
  subprogbase = subprog;

  /* set a base pointer of stack */
  stackbase = stack;

  /* set a base pointer of frame */
  framebase = frame;
}


int
incode(sym)			/* check of symbol in the code */
     register Symbol *sym;
{
  register Inst  *p;
  for (p = prog; p != progp; p++) {
    if (*p == (Inst) sym)
      return TRUE;
  }
  return FALSE;
}

int
incode2(sym)    /* check of symbol in the code, does appear twice? */
     register Symbol *sym;
{
  register Inst  *p;
  register int incode_count = 0;
  for (p = prog; p != progp; p++) {
    if (*p == (Inst) sym)
      if ( (++incode_count) > 1 )
        return TRUE;
  }
  return FALSE;
}


static Inst   *
ReallocCodeArea(prg, siz)
     Inst           *prg;
     int            *siz;
{
  Inst           *tmp;

#ifdef DEBUG
  if (indef) {
    fprintf(stderr, "\nMemory Reallocation for Sub-program: [%lx:%d] -->",
	    (unsigned long)prg, *siz);
  } else {
    fprintf(stderr, "\nMemory Reallocation for Main-program: [%lx:%d] -->",
	    (unsigned long)prg, *siz);
  }
#endif

  *siz += RELOCATION_SIZE;
  tmp = (Inst *) realloc(prg, (*siz) * sizeof(Inst));
  DEBUGF((stderr, " [%lx:%d]\n", (unsigned long)tmp, *siz));
  if (tmp == NULL) {
    *siz -= RELOCATION_SIZE;
    execerror("program is too large", (char *) 0);
  }
  return tmp;
}

int
ResizeCodeArea()
{
  Inst           *old_prog = prog;
  int             offset;

  if (indef) {
    prog = subprog = ReallocCodeArea(subprog, &SUBPROGSIZ);
    offset = prog - old_prog;
    subprogbase += offset;
    progp += offset;
    PROGSIZ = SUBPROGSIZ;
  } else {
    prog = mainprog = ReallocCodeArea(mainprog, &MAINPROGSIZ);
    offset = prog - old_prog;
    mainprogbase += offset;
    progp += offset;
    PROGSIZ = MAINPROGSIZ;
  }

  /* match ResizeCodeArea's offset to relocat's offset  */
  return (-offset);
}

/**********************************************************************
  Memory	Management	( Garbage Collection at Subprog  )
  ***********************************************************************/

static proglist *
pickup_func()
{
  register Symbol *sp;
  proglist       *plist = 0, *p;

  for (sp = GetSymbolTable(); sp != (Symbol *) 0; sp = sp->next) {
    if (sp->obj == NULL)	/* incomplete difinition */
      continue;
    if (sp->type == FUNCTION || sp->type == PROCEDURE) {
      /* where free ? */
      p = (proglist *) emalloc(sizeof(proglist));
      p->sub = (SubProg *) sp->obj->val;
      p->next = plist;
      plist = p;
    }
  }
  return plist;
}

static SubProg *
follow_sub(sp)			/* return sub-program code in order of
				 * low-address */
     proglist      **sp;	/* head of sub-prog list */
{
  SubProg        *sub;
  register proglist *cp;	/* current sub-prog */
  register proglist *bp = 0;	/* backup  sub-prog */
  register proglist *lowest;	/* sub-prog of lowest address */
  register proglist *pre = 0;	/* previous sub-prog */

  if (*sp == (proglist *) 0)	/* end of sub-program list ? */
    return (SubProg *) 0;	/* return end-mark */

  /*
   * Search for the sub-prog which has the lowest address in the
   * sub-program list.
   */
  lowest = *sp;			/* top of sub-prog list */
  for (cp = *sp; cp != (proglist *) 0; bp = cp, cp = cp->next) {
    if (lowest->sub->begin_code > cp->sub->begin_code) {
      pre = bp;
      lowest = cp;
    }
  }

  if (pre == 0)
    *sp = lowest->next;
  else
    pre->next = lowest->next;
  sub = lowest->sub;
  efree((char*)lowest);
  return sub;
}

int
relocate()
{
  int             len;
  SubProg        *sub;
  Absolute        addr;
  Inst           *savebase = subprogbase;
  proglist       *plist = pickup_func(); /* pickup sub-program */

  DEBUGF((stderr, "RELOCATE"));
  subprogbase = subprog;

  /* relocate subprogram code */
  while ((sub = follow_sub(&plist)) != (SubProg *) 0) {
    if (AbsoluteAddr(sub->begin_code) != subprogbase) {
      len = sub->end_code - sub->begin_code;
      addr = AbsoluteAddr(sub->begin_code);
      Bcopy((char*)addr, (char*)subprogbase, len * sizeof(Inst));
      sub->begin_code = RelativeAddr(subprogbase);
      sub->end_code = RelativeAddr(subprogbase + len);
      subprogbase += len;
    } else {
      subprogbase = AbsoluteAddr(sub->end_code);
    }
  }

  if (!indef) {
    return 0;
  }
  /* relocate current coding subprogram */
  if (savebase != subprogbase) {
    len = progp - savebase;
    Bcopy((char*)savebase, (char*)subprogbase, len * sizeof(Inst));
    progp = subprogbase + len;
    return savebase - subprogbase; /* return move length */
  } else {
    return 0;			/* move length = 0 */
  }
}



/**********************************************************************
	End of Stack Machine Code
***********************************************************************/
