/**********************************************************************
	code.stream.c :  Part of Stack Machine Code

			Coded by Shigeru Hitomi  Sep. 24, 1992
***********************************************************************/
#include "defs.h"

#include <stdio.h>
#include <string.h>
#include <signal.h>

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#include "prototype.h"
#include "y.tab.h"
#include "code.h"

#define DEBUGF(a)	fprintf a

/**********************************************************************
	STACK MACHINE   CODE  : Stack Operation Code
***********************************************************************/

BOOLEAN         open_pipe = FALSE;
int             PID = -1;
extern int      tty;

void
varread()
{
  Symbol         *class = (Symbol *) (*pc++);
  Datum           d;

  if (class->obj->method->read == NULL)
    execerror("not supported method", class->name);
  d.obj = (*class->obj->method->read) ();
  push_obj(d);
}

void
auto_print()
{
  Datum           d;

#ifdef DEBUG
  fprintf(stderr,"auto_print("); fflush(stderr);
#endif

  d = pop();
  if (d.obj != (Object *)NULL) {
    (*d.obj->method->print) (NULL, d.obj);
    putchar('\n');
    fflush(stdout);
  }
  push(d);

#ifdef DEBUG
  fprintf(stderr,")\n");
#endif
}

void
print()
{
  Datum           d;

  d = pop_obj();
  if (d.obj != (Object *)NULL) {
    (*d.obj->method->print) (NULL, d.obj);
    fflush(stdout);
  }
}

void
sh()
{				/* buld-in command */
  Symbol         *sym = (Symbol *) * pc++;
  intersh(*(char **) sym->obj->val, 0);
}

void
unix_system()
{
  int             usepipe;
  char           *str, *cmd;
  Datum           d;
  d = pop_obj();
  if (d.obj == (Object *)NULL)
    execerror("NULL object operation", 0);
  if (TypeofOBJ(d.obj) != STRING_T)
    execerror("illegal string", 0);

  cmd = *(char **) d.obj->val;
  if (strlen(cmd) >= ONELINE) {
    char            errcom[ONELINE + 30];
    strcpy(errcom, "(");
    strncat(errcom, cmd, ONELINE);
    strcat(errcom, " ... )");
    execerror(errcom, ": command too long");
  }

  str = (char *) intersh(cmd, usepipe = TRUE);
  if (str != NULL) {
#if 0
    int             n = 1;
    char          **argv = &str;
    d.obj = newacc(argv, 1, &n, &String);
    efree(str);
#endif
    int             n = argcount(str, " \t\n");
    char          **argv;

#ifdef DEBUG
    fprintf(stderr,"unix_system(): str=%s\n", str);
#endif

    if ( n > 0 ) { /* take */
      argv = (char **) emalloc( n* sizeof(char *));
      Bzero(argv, n * sizeof(char *));
      separate(str, argv, n, " \t\n");
      d.obj = newacc(argv, 1, &n, &String);
      efree((char*)argv);
    } else /* take */
      d.obj = (Object *)NULL; /* take */
    efree(str);
  } else
    d.obj = (Object *)NULL;

  push_obj(d);
}

static void
joint_pipe(in)
int             in;
{
  close(1), dup(in), close(in);
}

void
pipe2str()
{
  Inst	     *savepc = pc++;	      /* jump counter in savepc[0] */
  Inst	     *origin = savepc - 1; /* address at pipe2str */
  Datum       d;
  int         ctop[2];
  RETSIGTYPE  (*istat) (), (*qstat) (), (*tstat) (); /**/
  
#ifdef DEBUG
    fprintf(stderr,"in pipe2str()\n");
#endif

  istat = (RETSIGTYPE (*) ()) signal(SIGINT,  SIG_IGN);
  qstat = (RETSIGTYPE (*) ()) signal(SIGQUIT, SIG_IGN); /* take */
  tstat = (RETSIGTYPE (*) ()) signal(SIGTERM, SIG_IGN); /* take */

  pipe(ctop);

  PID = fork();
  if (PID == -1) {
    execerror("fork failed", 0);
  } else if (PID == 0) {		/* Clone Process */
    
#ifdef CORE
    fprintf(stderr,"forked in pipe2str\n");
#endif

    close(ctop[0]);
    close(1), dup(ctop[1]), close(ctop[1]);
    open_pipe = TRUE;

    signal(SIGINT,  istat);
    signal(SIGQUIT, qstat);
    signal(SIGTERM, tstat);

    return;	/* continue to execute stack machine code */
  }
  close(ctop[1]);
  d = pop_obj();		/* pop variable name */
  read_pipe(ctop[0]);	/* waitting for clone output */
  push_obj(d);		/* push variable name for assign() */
  assign();

  pc = origin + *(int *)&savepc[0]; /* jump for next program */
  
  signal(SIGINT,  istat);
  signal(SIGQUIT, qstat); /* take */
  signal(SIGTERM, tstat); /* take */

#ifdef DEBUG
    fprintf(stderr,"return from pipe2str()\n");
#endif
}

void
read_pipe(fd)
     int fd;
{
  char           *str;
  Datum           d;

#ifdef DEBUG
    fprintf(stderr,"in read_pipe()\n");
#endif

  str = (char *) readpipe(fd);
  close(fd);
  if (str != NULL) {
    int        n = argcount(str, " \t\n");
    char  **argv = (char **) emalloc(n * sizeof(char *));
    Bzero(argv, n * sizeof(char *));
    separate(str, argv, n, " \t\n");
    d.obj = newacc(argv, 1, &n, &String);
    efree((char*)argv);
    efree(str);
  } else
    d.obj = (Object *)NULL;
  push_obj(d);

#ifdef DEBUG
    fprintf(stderr,"return from read_pipe()\n");
#endif
}

void
make_pipe()
{
  int             in;
  char           *bin;
  Datum           d;

#ifdef DEBUG
  fprintf(stderr,"in make_pipe()\n");
#endif

  d = pop_obj();
  if (d.obj == (Object *)NULL)
    execerror("null object in the pipe", 0);
  if (TypeofOBJ(d.obj) != STRING_T)
    execerror("illegal object type", 0);
  bin = *(char **) d.obj->val;
  if (strlen(bin) >= ONELINE) {
    char            errcom[ONELINE + 30];
    strcpy(errcom, "(");
    strncat(errcom, bin, ONELINE);
    strcat(errcom, " ... )");
    execerror(errcom, ": command too long");
  }
  in = mkpipe(bin);
  if (in == -1)
    execerror(": failed to making pipe", 0);
  joint_pipe(in);
  open_pipe = TRUE;
  push(d);

#ifdef DEBUG
  fprintf(stderr,"return from make_pipe()\n");
#endif

}

void
close_pipe()
{
  int             pid;
  RETSIGTYPE  (*istat) (), (*qstat) (); /**/
  
#ifdef DEBUG
    fprintf(stderr,"in close_pipe(stack = %d)\n", (int)(stackp-stack));
#endif

  istat = (RETSIGTYPE (*) ()) signal(SIGINT,  SIG_IGN);
  qstat = (RETSIGTYPE (*) ()) signal(SIGQUIT, SIG_IGN); /* take */

  fflush(stdout);
  close(1), dup(tty);
  open_pipe = FALSE;
  while ((pid = wait(0)) != -1)
    /* wait for children */ ;

  pop(); /* take */

  if (PID == 0) {		/* clone process is die */
#ifdef CORE
    puts("I am a Child.");
    fprintf(stderr,"exit from close_pipe()\n");
#endif
    _exit(0);
  }

  signal(SIGINT,  istat);
  signal(SIGQUIT, qstat); /* take */

#ifdef DEBUG
    fprintf(stderr,"return from close_pipe()\n");
#endif
}


/**********************************************************************
	END OF STACK MACHINE CODE
***********************************************************************/
