/**********************************************************************
  code.c :  Part of Stack Machine Code
  
  Coded by Shigeru Hitomi  May. 4, 1992
  ***********************************************************************/


#include <stdio.h>
#include <math.h>
#include "defs.h"
#include "prototype.h"
#include "operator.h"

#include "y.tab.h"
#include "code.h"


#if defined(DEBUG) || defined(OUT_STACK_MACHINE_CODE) || defined(SHOW_STACK_MACHINE_CODE)
/**********************************************************************
  STACK MACHINE   CODE  : Stack Operation Code
  ***********************************************************************/
typedef  int (*opcodefunc_t)();

static struct {
  char  *name;
  int  (*addr) ();
}               opcode_table[] = {
  /* code.c */
  {"POP",            (opcodefunc_t) POP},
  {"push",           (opcodefunc_t) push},
  {"push_obj",       (opcodefunc_t) push_obj},
  {"constpush",      (opcodefunc_t) constpush},
  {"varpush",        (opcodefunc_t) varpush},
  {"eval",           (opcodefunc_t) eval},
  {"assgin",         (opcodefunc_t) assign},
  {"selfasgn",       (opcodefunc_t) selfasgn},
  {"prefix",         (opcodefunc_t) prefix},
  {"postfix",        (opcodefunc_t) postfix},
  {"new",            (opcodefunc_t) new},
  {"sh",             (opcodefunc_t) sh},
  {"snapshot",       (opcodefunc_t) snapshot},
  {"array",          (opcodefunc_t) array},
  {"snap_asgn",      (opcodefunc_t) snapshot_asgn},
  {"array_asgn",     (opcodefunc_t) array_asgn},
  {"make_pipe",      (opcodefunc_t) make_pipe},
  {"close_pipe",     (opcodefunc_t) close_pipe},
  {"pipe2str",       (opcodefunc_t) pipe2str},
  {"read_pipe",      (opcodefunc_t) read_pipe},
  {"pop",            (opcodefunc_t) pop},
  {"pop_obj",        (opcodefunc_t) pop_obj},
  {"auto_print",     (opcodefunc_t) auto_print},
  {"print",          (opcodefunc_t) print},
  {"unix_system",    (opcodefunc_t) unix_system},
  {"undef_var",      (opcodefunc_t) undef_var},
  
  /* func.c */
  {"define",         (opcodefunc_t) define},
  {"call",           (opcodefunc_t) call},
  {"funcret",        (opcodefunc_t) funcret},
  {"procret",        (opcodefunc_t) procret},
  {"bltin",          (opcodefunc_t) bltin},
  {"basic1",         (opcodefunc_t) basic1},
  {"basic2",         (opcodefunc_t) basic2},
  {"external",       (opcodefunc_t) external},
  
  /* flow.c */
  {"forcode",        (opcodefunc_t) forcode},
  {"whilecode",      (opcodefunc_t) whilecode},
  {"dowhilecode",    (opcodefunc_t) dowhilecode},
  {"ifcode",         (opcodefunc_t) ifcode},
  {"breaker",        (opcodefunc_t) breaker},
  {"continuer",      (opcodefunc_t) continuer},
  
  /* file.c */
  {"load",           (opcodefunc_t) load},
  {"store",          (opcodefunc_t) store},
  
  /* combin.c */
  {"ramp",           (opcodefunc_t) ramp},
  {"combin",         (opcodefunc_t) combin},
  
  /* mdl.code.c */
  {"define_module",  (opcodefunc_t) define_module},
  {"exec",           (opcodefunc_t) exec},
  {"InModule",       (opcodefunc_t) InModule},
  {"install_module", (opcodefunc_t) install_module},
  {"print_module",   (opcodefunc_t) print_module},
  
  /* symbol.c */
  {"install",        (opcodefunc_t) install},
  {"lookup",         (opcodefunc_t) lookup},

  /* Relative Operator       */
  {"and",            (opcodefunc_t) logic_and},
  {"or",             (opcodefunc_t) logic_or},
  {"not",            (opcodefunc_t) not},
  {"gt",             (opcodefunc_t) gt},
  {"ge",             (opcodefunc_t) ge},
  {"lt",             (opcodefunc_t) lt},
  {"le",             (opcodefunc_t) le},
  {"eq",             (opcodefunc_t) eq},
  {"ne",             (opcodefunc_t) ne},
  
  /* Basic Operators         */
  {"pow",            (opcodefunc_t) tpow},
  {"negate",         (opcodefunc_t) negate},
  {"mul",            (opcodefunc_t) mul},
  {"div",            (opcodefunc_t) Div},
  {"add",            (opcodefunc_t) add},
  {"sub",            (opcodefunc_t) sub},
  {"mod",            (opcodefunc_t) mod},

  /* End of Keywors	 */

  {NULL, NULL}
};

static struct {		/* Keywords */
  char           *name;
  int             kval;
}               keycode[] = {
  
  {"(0) or STOP", 0},
  
  /* Define Module	 */
  {"Command_Path", COM_DIR},
  {"Command_File", COM_FILE},
  {"Message_File", MSG_FILE},
  {"Error_File",   ERR_FILE},
  {"Setup_File",   SETUP},
  {"Clean_File",   CLEAN},
  {"module",       MODULE},
  {"define",       DEFINE},
  {"set",          SET},
  
  /* Object Type	 */
  {"series",       SERIES_T},
  {"snapshot",     SNAPSHOT_T},
  {"snap",         SNAPSHOT_T},
  {"scalar",       SCALAR_T},
  {"string",       STRING_T},
  
  /* Reserved Words	 */
  {"proc",         PROC},
  {"func",         FUNC},
  {"return",       RETURN},
  {"if",           IF},
  {"else",         ELSE},
  {"while",        WHILE},
  {"for",          FOR},
  {"print",        PRINT},
  {"read",         READ},
  {"undef",        UNDEF_VAR},
  {"unix",         UNIX},
  {"continue",     CONTINUE},
  {"break",        BREAK},
  
  /* End of Keywors	 */
  {NULL, 0}
};


char *
code_name(p)
     Inst  p;
{
  register int  i;
  char         *name;

  for (i = 0; opcode_table[i].name != 0; i++)
    if (p == (Inst) opcode_table[i].addr)
      return opcode_table[i].name;

  for (i = 0; keycode[i].name != 0; i++)
    if ((int) p == keycode[i].kval)
      return keycode[i].name;

  name = symbol_name(p);
  return name;
}


#ifdef SHOW_STACK_MACHINE_CODE
void
show_code(p)			/* show the stack machine code */
     Inst           *p;
{
  char           *name;

  if (istty(1) == FALSE)
    return;
  fprintf(stderr, "[%c[7m%lx%c[0m]::", ESC, (unsigned long)p, ESC);
  for (pc = p; *pc != STOP; pc++) {
    name = (char *) code_name(*pc);
    if (!equal(name, "?"))
      fprintf(stderr, "[%s]", name);
    else
      fprintf(stderr, "[%c[7m%lx%c[0m]", ESC, (unsigned long)*pc, ESC);
  }
  fprintf(stderr, "\n");
}
#endif				/* SHOW_STACK_MACHINE_CODE */

#ifdef OUT_STACK_MACHINE_CODE
void
out_stack_code(p, f)
     Inst           *p, f;
{
  static FILE    *fp = NULL;
  static char    *fname = "sl.opcode";
  char           *name = code_name(f);
  if (fp == (FILE *) - 1)
    return;
  if (fp == NULL) {
    unlink(fname);
    fp = fopen(fname, "w");
    if (fp == NULL) {
      fp = (FILE *) - 1;
      execerror(fname, "open failed");
    }
  }
  if (mainprogbase == p || subprogbase == p) {
    fprintf(fp, "\n[address]:\t[opcode]\n");
    fprintf(fp, "------------------------\n");
  }
  fprintf(fp, "%8lx :\t %s\n", (unsigned long)p, name);
  fflush(fp);
}
#endif				/* OUT_STACK_MACHINE_CODE */

#endif


#ifdef OUT_STACK
static char *
object_value(obj)
     Object         *obj;
{
  static char     tmp[256];
  if (-100 < (int) obj && (int) obj < 100) {
    sprintf(tmp, "(%d)", (int) obj);
    return tmp;
  }
  switch (TypeofOBJ(obj)) {
  case SERIES_T or SNAPSHOT_T:
    sprintf(tmp, "BUF%d", *(int *) obj->val);
    return tmp;
  case SCALAR_T:
    sprintf(tmp, "%g", *(double *) obj->val);
    return tmp;
  case STRING_T:
    sprintf(tmp, "\"%s\"", *(char **) obj->val);
    return tmp;
  default:
    return 0;
  }
}

void
out_stack(fout)
     FILE *fout;
{
  register Datum *p;
  static FILE    *fp = 0;
  static char    *destfile = "sl.stack";
  char           *name;

  if (fp == (FILE *) (-1))
    return;
  if (fp == 0) {
    unlink(destfile);
    if ((fp = fopen(destfile, "w")) == 0)
      fp = (FILE *) (-1);
  }
  if(fout == NULL)
    fout = fp;
  for (p = stack; p != stackp; p++) {
    name = symbol_name((Inst)p->sym);
    if (*name == '?')
      name = object_value(p->obj);
    fprintf(fout, "[%s]", name);
  }
  fprintf(fout, "\n");
  fflush(fout);
}
#endif				/* OUT_STACK */



/**********************************************************************
	END OF STACK MACHINE CODE
***********************************************************************/
