/**********************************************************************
	obj.series.c : Part of Stack Machine Code
			(operater parts)
	
			Coded by Shigeru Hitomi  May. 4, 1992
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <signal.h>

#include "defs.h"
#include "prototype.h"
#include "operator.h"
#include "y.tab.h"
#include "obj.series.h"

static Object *
series_new()
{
  Object         *sr;

  sr = (Object *) emalloc(sizeof(Object));
  sr->link = 0;
  sr->dim = 0;
  sr->index = NULL;
  sr->val = (int *) emalloc(sizeof(int));
  *(int *) sr->val = NextBuffer();
  sr->method = Series.method;

  return sr;
}

static  Object *
series_copy(to, from)
     Object       *to, *from;
{
  Buffer         *buf;
  static int             dim, index[MAX_INDEX], m, n;

  if (to == NULL || TypeofOBJ(to) != TypeofOBJ(from)) {
    if (to != NULL) 
      (*to->method->destroy) (to);
    to = (*from->method->new) ();
  }

  m = *(int *) to->val;
  n = *(int *) from->val;
  if ((buf = ReadBuffer(n, &dim, index)) == NULL)
    execerror("failed to evaluate series", 0);
  if (WriteBuffer(m, dim, index, buf) == -1)
    execerror("failed to set series", 0);
  FreeBuffer(buf);

  if (to->dim < dim) {
    efree((char*)(to->index));
    to->index = (int *) emalloc(dim * sizeof(int));
  }
  to->dim = dim;
  CopyIndex(to->index, index, dim);
  return to;
}


static void
series_setvalue(sr, buf, dim, index)
     Object         *sr;
     Buffer         *buf;
     int             dim, *index;
{
  int             buf_no = *(int *) sr->val;

  if (buf == NULL)
    execerror("null pointer assignment", 0);

  if (sr->dim < dim) {
    efree((char*)(sr->index));
    sr->index = (int *) emalloc(dim * sizeof(int));
  }
  sr->dim = dim;
  CopyIndex(sr->index, index, dim);
  if (WriteBuffer(buf_no, dim, index, buf) == -1)
    execerror("failed to set value", 0);
}

static void
series_destroy(sr)
     Object         *sr;
{
  if (sr == NULL)
    return;
  if (sr->link == 0) {
    DestroyBuffer(*(int *) sr->val);
    efree((char*)(sr->val));
    efree((char*)(sr->index));
    efree((char*)sr);
  } else
    sr->link--;
}

static Object *
series_snap(obj, time)
     Object         *obj;
     int             time;
{
  Object         *acc;
  int             dim, index[MAX_INDEX];
  int             no = *(int *) obj->val;
  Buffer         *snap = ReadSubBuffer(no, time, &dim, index);
  if (snap == NULL)
    execerror("failed to evaluate snapshot", 0);
  if (dim == 0)
    acc = newacc(snap, 0, 0, &Scalar);
  else
    acc = newacc(snap, dim, index, &Snapshot);
  FreeBuffer(snap);
  return acc;
}


static void
series_snap_asgn(sr, time, vobj)
     Object         *sr;
     int             time;
     Object         *vobj;
{
  Buffer         *buf = NULL;
  int             wdpt, nv, n = *(int *) sr->val;
  int             sub_dim, sub_index[MAX_INDEX];

  switch (TypeofOBJ(vobj)) {
  case SERIES_T or SNAPSHOT_T:
    nv = *(int *) vobj->val;
    if ((buf = ReadBuffer(nv, &sub_dim, sub_index)) == NULL)
      execerror("failed evaluate value in stack", 0);
    break;
  case SCALAR_T:
    buf = AllocBuffer(1);
    buf[0] = *(double *) vobj->val;
    sub_dim = 1;
    sub_index[0] = 1;
    break;
  default:
    execerror("object type mismatch", "(snapshot assgnment)");
    break;
  }

  wdpt = WriteSubBuffer(n, time, sub_dim, sub_index, buf);
  FreeBuffer(buf);
  if (wdpt == -1)
    execerror("failed to set a snapshot", 0);
  sr->index[0] = wdpt;
}


static Object *
series_array(obj, sub_dim, sub_index)
     Object         *obj;	/* series object        */
     int             sub_dim;/* number of index      */
     int            *sub_index;	/* a specified location */
{
  Buffer         *bufp;
  Object         *acc;
  int             no = *(int *) obj->val, length;

  bufp = ReadTimeSeries(no, sub_dim, sub_index, &length);
  if (bufp == NULL)
    execerror("failed to evaluate time-series", 0);
  acc = newacc(bufp, 1, &length, &Series);
  FreeBuffer(bufp);
  return acc;
}


static void
series_array_asgn(sr, sub_dim, sub_index, vobj)
     Object         *sr;
     int             sub_dim;
     int            *sub_index;
     Object         *vobj;
{
  Buffer         *buf = NULL;
  int             wdpt, length = 1, dim = 1, index[MAX_INDEX];
  int             nv, n = *(int *) sr->val;

  switch (TypeofOBJ(vobj)) {
  case SERIES_T or SNAPSHOT_T:
    nv = *(int *) vobj->val;
    buf = ReadBuffer(nv, &dim, index);
    if (buf == NULL)
      execerror("failed to evaluate value in stack", 0);
    if (dim != 1) {
      char            tmp[10];
      sprintf(tmp, "[%d]", dim);
      execerror("illegal demension of series,", tmp);
    }
    length = IndexSize(dim, index);
    break;
  case SCALAR_T:
    buf = AllocBuffer(length = 1);
    buf[0] = *(double *) vobj->val;
    break;
  default:
    execerror("object type mismatch", "(array assgnment)");
    break;
  }
  wdpt = WriteTimeSeries(n, sub_dim, sub_index, buf, length);
  FreeBuffer(buf);
  if (wdpt == -1)
    execerror("failed to set a time series", 0);
  if (wdpt < sr->index[0]) {
    char            s[20];
    sprintf(s, "%d (%d)", length, sr->index[0]);
    warning("Warning ... too short time-series", s);
    /*
     * warning("Warning ... cut off a tail short", s);
     */
  }
  if (sr->index[0] < length)
    sr->index[0] = length;
}
/*--------------------------------------------------------------------*/

static Object *
series_opcode(dptr, s, obj)
     DFUNC          *dptr;
     char           *s;
     Object         *obj[];
{
  register int    i, j, t;
  static char    *where = "at series operation";
  BOOLEAN         is_scalar, is_series;
  int            *index, indexx[MAX_INDEX], indexy[MAX_INDEX];
  int             dim, dimx, dimy;
  int             aBlockSize, time_scope;
  DFUNC           operator = *dptr;
  Buffer         *bufp, *bufx, *bufy;
  Object         *acc;

  bufx = (Buffer *) getbuffer(obj[0], &dimx, indexx);
  dim = dimy = CopyIndex(indexy, indexx, dimx);	/* for one argument */
  bufy = (Buffer *) getbuffer(obj[1], &dimy, indexy);
  if (dimx != dimy) {
    FreeBuffer(bufx);
    FreeBuffer(bufy);
    execerror("dimension mismatch", where);
  }
  index = (indexx[0] >= indexy[0]) ? indexx : indexy;
  bufp = (indexx[0] >= indexy[0]) ? bufx : bufy;
  time_scope = Min(indexx[0], indexy[0]);
  aBlockSize = IndexSize(dim - 1, SubIndex(index));
  if (aBlockSize == 0)
    aBlockSize = 1;

  is_scalar = (obj[1] != NULL && TypeofOBJ(obj[1]) == SCALAR_T);
  is_series = (bufy != NULL);
  if (is_series || is_scalar || bufy == NULL) {
    double          d = 0;
    if (is_scalar)
      d = *(double *) obj[1]->val;
    for (t = 0; t < time_scope; t++) {
      for (i = 0; i < aBlockSize; i++) {
	j = t * aBlockSize + i;
	if (bufy != NULL)
	  d = bufy[j];
	bufp[j] = (*operator) (bufx[j], d);
      }
    }
    check_result(s, TRUE);
  } else {
    FreeBuffer(bufx);
    FreeBuffer(bufy);
    execerror("illegal object type", where);
  }

  if (TypeofOBJ(obj[0]) == SNAPSHOT_T)
    acc = newacc(bufp, dim, index, &Snapshot);
  else
    acc = newacc(bufp, dim, index, &Series);
  FreeBuffer(bufx);
  FreeBuffer(bufy);
  return acc;
}


static void
series_print(fmt, obj)
     char           *fmt;
     Object         *obj;
{
  register int    i;
  Buffer         *buf;
  int		id = *(int *) obj->val;
  int             dim2, index2[MAX_INDEX], length;
  BOOLEAN		isTTY = TRUE; /*isTTY = istty(1);*/

  if (obj->dim == 0) {
    (void) printf("(zero point series)");
    return;
  }
  if ((buf = ReadBuffer(id, &dim2, index2)) == NULL) {
    execerror("failed to print series", 0);
  }

  length = IndexSize(dim2, index2);
  for (i = 0; i < length; i++) {
    if(isTTY) {
      index_header(i, dim2, index2, fmt, &Series);
      if (fmt != NULL)
	putchar(' ');
    } else {
      putchar('\n');
    }
    print_number(fmt, "%12.4g", buf[i]);
  }
  FreeBuffer(buf);
}

static Object *
series_read()
{				/* read into variable */
  register int    i;
  char            str[ONELINE], *p = str;
  Buffer          buf[1024];
  int             dim = 1, length = 0;

  Bzero(buf, 1024 * sizeof(Buffer));
  if (fgets(str, ONELINE, stdin) == NULL) {
    clearerr(stdin);		/* reset EOF */
    return newacc(buf, dim, &length, &Series);
  }
  for (i = 0; *p != '\0'; i++) {
    sscanf(p, "%lf", &buf[i]);
    while (*p != ',' && *p != '\0')
      p++;
    if (*p == ',')
      p++;
  }
  length = i;
  return newacc(buf, dim, &length, &Series);
}

/**********************************************************************
        End of Stack Machine Code (operater parts)
***********************************************************************/
