/**********************************************************************
	obj.string.c : (STRING Object)
	
			Coded by Shigeru Hitomi  May. 13, 1992
			Last Edited at Dec. 6, 1992.
***********************************************************************/
#include <stdio.h>
#include <string.h>

#include "defs.h"
#include "prototype.h"
#include "operator.h"
#include "y.tab.h"
#include "obj.string.h"

static void    free_strings  _ANSI_ARGS_((Object *obj));
static Object *getstrings    _ANSI_ARGS_((Object *obj));
static Object *comp          _ANSI_ARGS_((int (*op)(), char **s0, char **s1,
					  int u0, int u1, int dim, int *idx));
static Object *sadd          _ANSI_ARGS_((int (*op)(), char **s0, char **s1,
					  int u0, int u1, int dim, int *idx));
static Object *ssub          _ANSI_ARGS_((int (*op)(), char **s0, char **s1,
					  int u0, int u1, int dim, int *idx));
static Object *sdiv          _ANSI_ARGS_((int (*op)(), char **s0, char **s1,
					  int u0, int u1, int dim, int *idx));
static Object *smul          _ANSI_ARGS_((int (*op)(), char **s0, char **s1,
					  int u0, int u1, int dim, int *idx));

static struct {
  char      *(*func)  ();
  Object    *(*method)();
}               bind[] = {
  {(SFUNC) eq,  comp},
  {(SFUNC) ne,  comp},
  {(SFUNC) add, sadd},
  {(SFUNC) sub, ssub},
  {(SFUNC) Div, sdiv},
  {(SFUNC) mul, smul},
  {0, 0}
};


static Object *
string_new()
{
  Object         *str;

  str = (Object *) emalloc(sizeof(Object));
  str->dim = 1;
  str->index = (int *) emalloc(sizeof(int));
  str->index[0] = 1;
  str->link = 0;
  str->val = NULL;
  str->method = String.method;

  return str;
}

static Object *
string_copy(to, from)
     Object         *to, *from;
{
  if (to == NULL || TypeofOBJ(to) != TypeofOBJ(from)) {
    if (to != NULL)
      (*to->method->destroy) (to);
    to = (*from->method->new) ();
  }
  (*to->method->setvalue) (to, from->val, from->dim, from->index);
  return to;
}


static void
string_setvalue(obj, argv, dim, idx)
     Object         *obj;
     char           *argv[];
     int             dim, *idx;
{
  register int    i, siz;
  

  free_strings(obj);
  if (dim == 0) {
    dim = 1;
    idx = (int *) emalloc(sizeof(int));
    idx[0] = 1;
  }

  siz = IndexSize(dim, idx);
  obj->val = (char **) emalloc(siz * sizeof(char *));


  for (i = 0; i < siz; i++) {
    char  *s = argv[i];

    ((char **)(obj->val))[i] = (char *)
      emalloc( ( s != NULL ) ? (sizeof(char)*strlen(s)+1) : 1 );

    if ( s != NULL ) {
      strcpy(((char **)(obj->val))[i], s);
    } else {
      strcpy(((char **)(obj->val))[i], "\0" );
    }
  }
  if (obj->dim < dim)
    obj->index = (int *) erealloc((char*)(obj->index), dim * sizeof(int));
  obj->dim = dim;
  CopyIndex(obj->index, idx, dim);
}

static void
free_strings(obj)
     Object         *obj;
{
  int   siz = IndexSize(obj->dim, obj->index);
  char  **p = (char **) obj->val;
  if (p != NULL)
    free2d(p, siz);
}

static void
string_destroy(str)
     Object         *str;
{
  if (str == NULL)
    return;
  if (str->link == 0) {
    free_strings(str);
    efree((char*)(str->index));
    efree((char*)(str));
  } else
    str->link--;
}

static Object *
string_array(obj, sub_dim, sub_idx)
     Object         *obj;	/* series object        */
     int             sub_dim;   /* number of index      */
     int            *sub_idx;	/* a specified location */
{
  char           *str;
  int             n;
  if (sub_dim != obj->dim)
    execerror("dimension mismatch", 0);
  if (!RegularIndex(sub_idx, obj->index, obj->dim))
    execerror("illegal index", 0);

  n = Index(sub_idx, obj->dim, obj->index);
  str = ((char **) obj->val)[n];
  return newacc(&str, 0, NULL, &String);
}

static void
string_array_asgn(obj, sub_dim, sub_idx, vobj)
     Object         *obj;
     int             sub_dim;
     int            *sub_idx;
     Object         *vobj;
{
  register int    i;
  char          **p, *str;
  int             siz = IndexSize(obj->dim, obj->index);
  int             n = Index(sub_idx, obj->dim, obj->index);

  if (sub_dim != obj->dim)
    execerror("dimension mismatch", 0);
  if (obj->dim != 1 && !RegularIndex(sub_idx, obj->index, obj->dim))
    execerror("illegal index", 0);

  if (TypeofOBJ(vobj) != STRING_T)
    execerror("illegal object type", 0);
  str = ((char **) vobj->val)[0];

  if (n < siz) {		/* in regular index */
    p = (char **) obj->val;
    if (strlen(p[n]) < strlen(str)) {
      efree(p[n]);
      p[n] = emalloc(strlen(str) + 1);
    }
    strcpy(p[n], str);
  } else {			/* out of regular index (1 dimension only) */
    obj->val = (char **) erealloc(obj->val, (n + 1) * sizeof(char *));
    p = (char **) obj->val;
    for (i = siz; i < n; i++) {
      p[i] = emalloc(sizeof(char));
      strcpy(p[i], "");
    }
    p[n] = emalloc(strlen(str) + 1);
    strcpy(p[n], str);
    obj->index[0] = n + 1;
  }
}


/*--------------------------------------------------------------------*/

static Object *
string_opcode(sptr, s, obj)
     SFUNC          *sptr;
     char           *s;
     Object         *obj[];
{
  register int    i;
  SFUNC           op = *sptr;
  char          **s0, **s1;
  int             siz0, siz1, dim0, dim1, dim, *idx, u0, u1;
  BOOLEAN         a_word;

  /* search for method */
  for (i = 0; bind[i].func != 0; i++) {
    if (bind[i].func == op)
      break;
  }
  if (bind[i].func == 0)
    execerror("string", "not supported method");

  obj[0] = getstrings(obj[0]);
  obj[0]->link++;		/* LOCK   ACC */
  obj[1] = getstrings(obj[1]);
  obj[0]->link--;		/* UNLOCK ACC */

  dim0 = obj[0]->dim;
  dim1 = obj[1]->dim;
  siz0 = IndexSize(dim0, obj[0]->index);
  siz1 = IndexSize(dim1, obj[1]->index);
  a_word = (Min(siz0, siz1) == 1); /* one word ? */

  if (!a_word && dim0 != dim1)
    execerror("string", "illegal index");
  if (!a_word && !EqualIndex(obj[0]->index, obj[1]->index, dim1))
    execerror("string", "illegal index");

  s0 = (char **) obj[0]->val;
  s1 = (char **) obj[1]->val;
  dim = (siz0 == 1) ? dim1 : dim0;
  idx = (siz0 == 1) ? obj[1]->index : obj[0]->index;
  u0 = (siz0 == 1) ? 0 : 1;
  u1 = (siz1 == 1) ? 0 : 1;
  return (*bind[i].method) (op, s0, s1, u0, u1, dim, idx);
}


static Object *
comp(op, s0, s1, u0, u1, dim, idx)
     int             (*op) ();
     char          **s0, **s1;
     int             u0, u1, dim, *idx;
{
  register int    i;
  BOOLEAN         true;
  int             siz = IndexSize(dim, idx);
  Buffer         *buf = (Buffer *) AllocBuffer(siz);
  Object         *class = (siz == 1) ? &Scalar : &Series;

  for (i = 0; i < siz; i++) {
    true = equal(s0[i * u0], s1[i * u1]);
    buf[i] = (Buffer) ((op == (int (*) ()) eq) ? true : !true);
  }
  return newacc(buf, dim, idx, class);
}

static Object *
sadd(op, s0, s1, u0, u1, dim, idx)
     int             (*op) ();
     char          **s0, **s1;
     int             u0, u1, dim, *idx;
{
  register int    i;
  Object         *acc;
  char          **work, *p0, *p1;
  int             length, siz = IndexSize(dim, idx);

  work = (char **) emalloc(siz * sizeof(char *));
  for (i = 0; i < siz; i++) {
    p0 = s0[i * u0];
    p1 = s1[i * u1];
    length = strlen(p0) + strlen(p1) + 1;
    work[i] = (char *) emalloc(length);
    strcpy(work[i], p0), strcat(work[i], p1);
  }
  acc = newacc(work, dim, idx, &String);
  free2d(work, siz);
  return acc;
}

static Object *
ssub(op, s0, s1, u0, u1, dim, idx)
     int             (*op) ();
     char          **s0, **s1;
     int             u0, u1, dim, *idx;
{
  register int    i, j;
  Object         *acc;
  char           *p0, *p1, **work;
  int             siz = IndexSize(dim, idx);

  work = (char **) emalloc(siz * sizeof(char *));
  for (i = 0; i < siz; i++) {
    p0 = s0[i * u0];
    p1 = s1[i * u1];
    work[i] = (char *) emalloc(strlen(p0) + 1);
    strcpy(work[i], p0);
    for (j = strlen(p0); j >= 0; j--) {
      if (equal(&work[i][j], p1)) {
	work[i][j] = '\0';
	break;
      }
    }
  }
  acc = newacc(work, dim, idx, &String);
  free2d(work, siz);
  return acc;
}

static Object *
sdiv(op, s0, s1, u0, u1, dim, idx)
     int             (*op) ();
     char          **s0, **s1;
     int             u0, u1, dim, *idx;
{
  register int    i, j, k;
  Object         *acc;
  int             n, total, siz = IndexSize(dim, idx);
  char          **work, ***argvp, **argv, *p0, *p1;

  if (dim != 1) {
    char            tmp[20];
    sprintf(tmp, "(%d) -> (1)", dim);
    warning("WARNING ... change dimension", tmp);
  }
  work  = (char **) emalloc(siz * sizeof(char *));
  argvp = (char ***) emalloc(siz * sizeof(char **));

  for (total = 0, i = 0; i < siz; i++) {
    p0 = s0[i * u0];
    p1 = s1[i * u1];
    argvp[i] = (char **) emalloc( ONELINE * sizeof(char *));
    work[i] = (char *) emalloc(strlen(p0) + 1);
    strcpy(work[i], p0);
    if ((n = separate(work[i], argvp[i], ONELINE, p1)) == ONELINE)
      warning("WARNING ... ", "perhaps, cut off a tail short");
    total += n;
  }

  if (total > 0) {
    argv = (char **) emalloc(total * sizeof(char *));
    for (k = 0, i = 0; k < total && i < siz; i++) {
      for (j = 0; k < total && argvp[i][j] != 0; j++)
	argv[k++] = argvp[i][j];
    }
    acc = newacc(argv, 1, &total, &String);
  } else {
    total = 0;
    argv = (char **) emalloc(sizeof(char *));
    argv[0] = "\0";
    acc = newacc(argv, 0, NULL, &String);
  }

  /* Free Working Area */
  free2d(work, siz);
  free2d((char **) argvp, siz);
  efree((char*)argv);
  return acc;
}

static Object *
smul(op, s0, s1, u0, u1, dim, idx)
     int             (*op) ();
     char          **s0, **s1;
     int             u0, u1, dim, *idx;
{
  register int    i, j;
  Object         *acc;
  char          **work, *p0;
  int             n, siz = IndexSize(dim, idx);

  work = (char **) emalloc(siz * sizeof(char *));
  for (i = 0; i < siz; i++) {
    p0 = s0[i * u0];
    n = atoi(s1[i * u1]);
    work[i] = (char *) emalloc(strlen(p0) * n + 1);
    strcpy(work[i], "");
    for (j = 0; j < n; j++)
      strcat(work[i], p0);
  }
  acc = newacc(work, dim, idx, &String);
  free2d(work, siz);
  return acc;
}


static Object *
getstrings(obj)
     Object         *obj;
{
  register int    i;
  int             siz, n, dim, idx[MAX_INDEX];
  char          **strp, sbuf[100], *sp = sbuf;
  Buffer         *buf;
  Object         *acc;
  if (obj == NULL)
    execerror("string", "illegal object type");

  switch (TypeofOBJ(obj)) {
  case STRING_T:
    return obj;
  case SCALAR_T:
    sprintf(sp, "%g", *(double *) obj->val);
    return newacc(&sp, 0, NULL, &String);
  case SERIES_T:
  case SNAPSHOT_T:
    n = *(int *) obj->val;
    if ((buf = ReadBuffer(n, &dim, idx)) == NULL)
      execerror("failed to evaluate a series/snapshot", 0);
    siz = IndexSize(dim, idx);
    strp = (char **) emalloc(siz * sizeof(char *));
    for (i = 0; i < siz; i++) {
      sprintf(sbuf, "%g", buf[i]);
      strp[i] = (char *) emalloc(strlen(sbuf) + 1);
      strcpy(strp[i], sbuf);
    }
    acc = newacc(strp, dim, idx, &String);
    FreeBuffer(buf);
    free2d(strp, siz);
    return acc;
  default:
    execerror("string", "illegal object type");
    break;
  }
  return NULL;			/* not execute */
}


static void
string_print(fmt, obj)
     char           *fmt;
     Object         *obj;
{				/* pop top value from stack, print it */
  register int    i;
  int             n = IndexSize(obj->dim, obj->index), length, columns;
  char          **argv = (char **) obj->val, deffmt[10];
  extern int	COLUMNS;
  BOOLEAN		isTTY = TRUE; /*isTTY = istty(1);*/

  if (argv == NULL) {
    printf("(nil)");
    return;
  }
  if (n == 1) {
    char  *format = (fmt == NULL) ? "%s" : fmt;
    printf(format, (*argv[0] != '\0') ? argv[0] : "(nil)");
    return;
  }
  /* max length of strings */
  for (length = 5, i = 0; i < n; i++) {
    int  len = strlen(argv[i]);
    if (argv[i][len - 1] == '\n')
      length = COLUMNS;
    if (length < len)
      length = len;
  }
  sprintf(deffmt, "%%%ds", length);
  columns = COLUMNS - (4 * obj->dim + 3); /* COLUMNS - index_width */
  if(length > columns)
    length = columns;

  for (i = 0; i < n; i++) {
    if(isTTY)
      index_header(i, obj->dim, obj->index, deffmt, &String);
    else
      putchar('\n');
    printf("%-*s ", length, (*argv[i] != '\0') ? argv[i] : "(nil)");
  }
  return;
}


static Object *
string_read()
{				/* read into variable */
  char  str[ONELINE], *argv[1];
  int   n;

  Bzero(str, ONELINE);
  if (fgets(str, ONELINE, stdin) == NULL) {
    clearerr(stdin);		/* reset EOF */
  }

  /* remove newline located end of input line */
  if ((n = strlen(str)) > 0)
    if (str[n-1] == '\n') str[n-1] = '\0';

  argv[0] = str;
  return newacc(argv, 0, NULL, &String);
}

/**********************************************************************
        End of Stack Machine Code (operater parts)
***********************************************************************/
