/*******************************************************************
  infile.c : Stream Procedure
  
  Coded by Shigeru Hitomi  May, 20, 1992
  *******************************************************************/

/*******************************************************************
  INCLUDE FILES
  *******************************************************************/
#include "defs.h"
#include "stream.h"		/* include stdio.h */

#include <string.h>
#include <ctype.h>

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#include <fcntl.h>

#include "y.tab.h"
#include "prototype.h"

/*******************************************************************
  DEFINITION
  *******************************************************************/
#define MAX_NESTING	10

/*******************************************************************
  GLOBAL VARIABLES
  *******************************************************************/

typedef struct fstack {
  FILE           *fin;
  char           *infile;
  char           *stream;
  int             lineno;
}               FILE_P;

static BOOLEAN NeedSysSetup = TRUE;	/* system setup flag  */
static BOOLEAN NeedUsrSetup = TRUE;	/* user setup flag    */
static BOOLEAN NeedUsrClean = TRUE;	/* user clean-up flag */

static int     Gargc;
static char  **Gargv;
static int     SaveLineNo = 0;

static char    homedir[PATHLEN];	/* home directory     */
static FILE_P  fin_stack[MAX_NESTING];	/* file stack              */
static FILE_P *fin_stackp;	/* file stack pointer      */
static int     fd_null, fd_tty;/* file descriptor         */
static FILE   *fnull;		/* "/dev/null" file pointer */
static char    rc_file[PATHLEN];
static char    setup_file[PATHLEN];
static char    clean_file[PATHLEN];


char     errlog[PATHLEN];/* export to waring()      */

MEMORY	*fin_mp = NULL;
FILE    *fin;		/* input file pointer: getstream() */
FILE    *pout;		/* for prompt output : prompt()    */

char    *infile = NULL;	/* input file name   : warning()   */

int      lineno = 1;	/* line number       : warning()   */
char     logdir[PATHLEN];/* directory of log-file           */

char    *progname = NULL; /* program name      : warning()   */


/*******************************************************************
  PRIVATE PROTOTYPES
  *******************************************************************/
static int   pop_fin  _ANSI_ARGS_((void));
static void  push_fin _ANSI_ARGS_((FILE *fp));
static void  print_usage _ANSI_ARGS_((char *comname));

/*******************************************************************
  FUNCTIONS
  *******************************************************************/

void
init_files()
{
  static char           *silent = "/dev/null";


  fin_stackp = fin_stack;	/* set stack pointer  */

  if (logdir[0] != '\0')	/* for warnings */
    sprintf(errlog, "%s/%s", logdir, ERROR_LOG);

  /* (fnull : GLOVAL VARIABLE) open /dev/null */
  if ((fnull = fopen(silent, "w")) == NULL) {
    fprintf(stderr, "%s: can't open %s\n", progname, silent);
    exit(0);
  }
  fd_tty = opentty();
  fd_null = open(silent, 2);
  if (fd_null == -1) {
    fprintf(stderr, "%s: can't open %s\n", progname, silent);
    exit(0);
  }
}


static void
push_fin(fp)
     FILE           *fp;
{
  FILE_P         *tmp;
  if (fin_stackp >= &fin_stack[MAX_NESTING])
    execerror("inline:  nested too deep", 0);

  tmp = fin_stackp++;
  tmp->fin = fin;
  tmp->lineno = lineno;

  if ( tmp->infile != NULL )
    efree(tmp->infile);

  if ( infile != NULL && strlen(infile) > 0 ) {
    tmp->infile = emalloc(strlen(infile)+1);
    strcpy( tmp->infile, infile );
  } else
    tmp->infile = NULL;

  tmp->stream = emalloc(strlen(streamptr) + 1);
  strcpy(tmp->stream, streamptr);

  lineno = 1;			/* top of file */
  fin = fp;
  pout = (fin == stdin) ? stderr : fnull;
}


static int
pop_fin()
{
  FILE_P         *tmp;
  if (fin_stackp == fin_stack) {
    stream[0] = EOF;

    if ( fin != NULL && fin != stdin ) {
      fclose(fin);
      fin = NULL;
    } /* take */
    return 0;
  }
  tmp = --fin_stackp;

  fclose(fin);
  fin = tmp->fin;
  lineno = tmp->lineno;

  if ( infile != NULL )
    efree(infile);

  if ( tmp->infile != NULL ) {
    infile = emalloc(strlen(tmp->infile)+1);
    strcpy( infile, tmp->infile );
  } else
    infile = NULL;
	
  strcpy(stream, tmp->stream);
  streamptr = stream;
  efree(tmp->stream);

  pout = (fin == stdin) ? stderr : fnull;
  return 1;
}

void
begin_inline(sym)
     Symbol         *sym;
{
  FILE           *fp;
  register char  *fname = *(char **) sym->obj->val;


  while (isspace(*fname))	/* space skip */
    fname++;
  if ((fp = fopen(fname, "r")) == NULL) {
    warning("WARNING ... can't open ", fname);
    return;
  }
  /* space skip */
  while (isspace(*streamptr) || *streamptr == ';') {
    if(*streamptr == '\n')
      break;
    streamptr++;
  }

  push_fin(fp);

  if ( infile != NULL )
    efree(infile);

  if ( fname != NULL ) {
    infile = emalloc(strlen(fname)+1);
    strcpy(infile, fname); 
  } else
    infile = NULL;

  strcpy(stream, ";\n");	/* forced truncation */
  streamptr = stream;		/* reset stream pointer */
}

int
end_inline()
{
  return pop_fin();
}


int				/* recursive function */
moreinput()
{
  register char *infilename;
  
  if (*Gargv == NULL)
    return 0;
  
  lineno = (SaveLineNo != 0) ? SaveLineNo : 1;
  SaveLineNo = 0;
  infilename = *Gargv++;	/* input file  */
  while (isspace(*infilename))	/* space skip */
    infilename++;
  pout = fnull;			/* output file for prompt */

  if ( infile != NULL )
    efree(infile);
  if ( infilename != NULL ) {
    infile = emalloc(strlen(infilename)+1);
    strcpy(infile, infilename);
  } else
    infile = NULL;

  if (equal(infile, "-")) {	/* standard input/output */
    strcpy(infile,"\0");
    fin = stdin;
    pout = stderr;
  } else if ((fin = fopen(infile, "r")) == NULL) {
    if (equal(infile, rc_file) ||
	equal(infile, setup_file) ||
	equal(infile, clean_file))
      return moreinput();
    
    /* warning */
    fprintf(stderr, "%s: can't open %s\n", progname, infile);
    /* try again -> next files */
    return moreinput();
  }
  return 1;
}

void
redo()
{
  SaveLineNo = lineno;
  --Gargv;
}


int
regist_setupfiles()
{
  int     i = 0;
  int     argc = Gargc + NeedSysSetup + NeedUsrSetup + NeedUsrClean + 1;
  char  **argv = (char **) emalloc(argc * sizeof(char *));
  char  **gp;

  gethomedir(homedir, PATHLEN);	/* set home directory */

  if (NeedSysSetup) {		/* loading initialize file */
    sprintf(rc_file, "%s/%s", SL_RC_DIR, RC_FILE);
    argv[i++] = rc_file;	/* rc.sl */
  }
  if (NeedUsrSetup) {		/* loading initialize file */
    sprintf(setup_file, "%s/%s", homedir, SETUP_FILE);
    if(access(setup_file,R_OK)==-1)
      sprintf(setup_file, "%s/setup.sl", SL_RC_DIR);
    argv[i++] = setup_file;	/* .setup.sl */
  }
  for (gp = Gargv; *gp != NULL; gp++)
    argv[i++] = *gp;

  if (NeedUsrClean) {		/* loading clean-up file */
    sprintf(clean_file, "%s/%s", homedir, CLEAN_FILE);
    if(access(clean_file,R_OK)==-1)
      sprintf(clean_file, "%s/clean.sl", SL_RC_DIR);
    argv[i++] = clean_file;	/* .clean.sl */
  }
  argv[i] = NULL;
  efree((char*)Gargv);

  Gargc = i;
  Gargv = argv;
  return i;
}


int
read_arg(argc, argv)
     int             argc;
     char          **argv;
{
  extern char     WorkingArea[];
  char            *work_env = NULL;
  register int    i;
  int             in_stdin = FALSE;

  Gargc = 0;
  Gargv = (char **) emalloc((argc + 1) * sizeof(char *));

  progname = argv[0];
  if ((work_env = getenv("SL_WORK_DIR")) == NULL) {
    strcpy(WorkingArea, TMP_DIR);	/* set default working area */
  } else {
    strcpy(WorkingArea, work_env);
  }

  for (i = 1; i < argc; i++) {
    if (equal(argv[i], "-rc")) {
      NeedSysSetup = FALSE;
    } else if (equal(argv[i], "-setup")) {
      NeedUsrSetup = FALSE;
    } else if (equal(argv[i], "-clean")) {
      NeedUsrClean = FALSE;
    } else if (equal(argv[i], "-work") || equal(argv[i], "-temp")) {
      if (++i < argc)
	strcpy(WorkingArea, argv[i]);
    } else if (equal(argv[i], "-log")) {
      if (++i < argc)
	strcpy(logdir, argv[i]);
    } else if (equal(argv[i], "-help") || equal(argv[i], "-h")) {
      print_usage(argv[0]);
      exit(0);
    } else {
      if (equal(argv[i], "-"))
	in_stdin = TRUE;
      Gargv[Gargc++] = argv[i];
    }
  }
  Gargv[Gargc] = NULL;

  /* set the program name for execerror() */
  if (Gargc == 0) {		/* take an argument list */
    static char    *stdinonly = "-";
    Gargv[Gargc++] = stdinonly;
    Gargv[Gargc++] = NULL;
    in_stdin = TRUE;
  }
  return in_stdin;
}


char *
history_file()
{
  static BOOLEAN  called = FALSE;
  static char     fname[PATHLEN];
  if (called)
    return fname;

  sprintf(fname, "%s/%s", homedir, HIST_FILE);
  called = TRUE;
  return fname;
}


static void
print_usage(comname)
  char *comname;
{
  printf("\nSATELLITE Version %s\n", SL_VERSION);
  printf("\n%s [options] [files ...]\n", comname);
  printf("\nOPTIONS:\n");
  printf("  -clean         : don't read ~/.clean.sl\n");
  printf("  -log fname     : write log into specified file\n");
  printf("  -rc            : doesn't read system resouce file\n");
  printf("  -setup         : doesn't read ~/.setup.sl\n");
  printf("  -work dir      : use other working directory\n");
  printf("  -help or -h    : print this message\n");
  printf("  -              : read instructions from stdin\n");
  printf("\nENVIRONMENT VARIABLES:\n");
  printf("  SL_FIXED_TITLE : doesn't change window title of terminal\n");
  printf("  SL_WORK_DIR    : working directory path\n");
  printf("                   (ex. setenv SL_WORK_DIR /var/tmp)\n");
  printf("\n");
}




/******************************************************************
	End of Stream Procedure
*******************************************************************/
